package cn.ipokerface.common.utils;

import org.apache.commons.codec.binary.Base64;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * Created by       PokerFace
 * Create Date      2019-11-23.
 * Email:
 * Version          1.0.0
 * <p>
 * Description:     API to append RSA digest numbers
 *
 */
public class RSADigestUtils {


    public static final String ENCRYPT_TYPE = "RSA";



    /**
     * key pair(key size=1024)
     *
     * @return keypair
     * @throws NoSuchAlgorithmException exception
     */
    public RSADigestUtils.KeyPairInfo getKeyPair()throws NoSuchAlgorithmException {
        return getKeyPair(1024);
    }

    /**
     *
     *
     * @param keySize keySize
     * @return keypair
     * @throws NoSuchAlgorithmException exception
     */
    public RSADigestUtils.KeyPairInfo getKeyPair(int keySize) throws NoSuchAlgorithmException {
        KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance(ENCRYPT_TYPE);
        // 初始化密钥对生成器，密钥大小一般要大于1024位，
        keyPairGen.initialize(keySize);
        // 生成一个密钥对，保存在keyPair中
        KeyPair keyPair = keyPairGen.generateKeyPair();
        // 得到私钥
        RSAPrivateKey oraprivateKey = (RSAPrivateKey) keyPair.getPrivate();
        // 得到公钥
        RSAPublicKey orapublicKey = (RSAPublicKey) keyPair.getPublic();

        RSADigestUtils.KeyPairInfo pairInfo = new RSADigestUtils.KeyPairInfo(keySize);
        //公钥
        byte[] publicKeybyte = orapublicKey.getEncoded();
        String publicKeyString = Base64.encodeBase64String(publicKeybyte);
        pairInfo.setPublicKey(publicKeyString);
        //私钥
        byte[] privateKeybyte = oraprivateKey.getEncoded();
        String privateKeyString = Base64.encodeBase64String(privateKeybyte);
        pairInfo.setPrivateKey(privateKeyString);

        return pairInfo;
    }


    /**
     *
     *
     * @param publicKeyBase64 public key
     * @return publicKey
     * @throws InvalidKeySpecException exception
     * @throws NoSuchAlgorithmException exception
     */
    public static PublicKey getPublicKey(String publicKeyBase64)
            throws InvalidKeySpecException, NoSuchAlgorithmException {

        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        X509EncodedKeySpec publicpkcs8KeySpec =
                new X509EncodedKeySpec(Base64.decodeBase64(publicKeyBase64));
        PublicKey publicKey = keyFactory.generatePublic(publicpkcs8KeySpec);
        return publicKey;
    }

    /**
     *
     *
     * @param privateKeyBase64 privateKey
     * @return PrivateKey
     * @throws NoSuchAlgorithmException  exception
     * @throws InvalidKeySpecException exception
     */
    public static PrivateKey getPrivateKey(String privateKeyBase64)
            throws NoSuchAlgorithmException, InvalidKeySpecException {
        KeyFactory keyFactory = KeyFactory.getInstance(ENCRYPT_TYPE);
        PKCS8EncodedKeySpec privateKCS8KeySpec =
                new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKeyBase64));
        PrivateKey privateKey = keyFactory.generatePrivate(privateKCS8KeySpec);
        return privateKey;
    }


    /**
     *
     * @param content  content
     * @param publicKeyBase64 publicKeyBase64
     * @return encrypted
     */
    public static String encipher(String content, String publicKeyBase64) {
        return encipher(content, publicKeyBase64, 117);
    }

    /**
     *
     *
     * @param content content
     * @param publicKeyBase64 publicKeyBase64
     * @param segmentSize segmentSize
     * @return encrypted
     */
    public static String encipher(String content, String publicKeyBase64, int segmentSize) {
        try {
            PublicKey publicKey = getPublicKey(publicKeyBase64);
            return encipher(content, publicKey, segmentSize);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     *
     * @param ciphertext ciphertext
     * @param key key
     * @param segmentSize segmentSize
     * @return encrypted
     */
    public static String encipher(String ciphertext, Key key, int segmentSize) {
        try {
            // 用公钥加密
            byte[] srcBytes = ciphertext.getBytes();

            // Cipher负责完成加密或解密工作，基于RSA
            Cipher cipher = Cipher.getInstance("RSA");
            // 根据公钥，对Cipher对象进行初始化
            cipher.init(Cipher.ENCRYPT_MODE, key);
            byte[] resultBytes = null;

            if (segmentSize > 0)
                resultBytes = cipherDoFinal(cipher, srcBytes, segmentSize); //分段加密
            else
                resultBytes = cipher.doFinal(srcBytes);

            String base64Str = Base64.encodeBase64String(resultBytes);
            return base64Str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 分段大小
     *
     * @param cipher
     * @param srcBytes
     * @param segmentSize
     * @return
     * @throws IllegalBlockSizeException
     * @throws BadPaddingException
     * @throws IOException
     */
    private static byte[] cipherDoFinal(Cipher cipher, byte[] srcBytes, int segmentSize)
            throws IllegalBlockSizeException, BadPaddingException, IOException {
        if (segmentSize <= 0)
            throw new RuntimeException("分段大小必须大于0");
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int inputLen = srcBytes.length;
        int offSet = 0;
        byte[] cache;
        int i = 0;
        // 对数据分段解密
        while (inputLen - offSet > 0) {
            if (inputLen - offSet > segmentSize) {
                cache = cipher.doFinal(srcBytes, offSet, segmentSize);
            } else {
                cache = cipher.doFinal(srcBytes, offSet, inputLen - offSet);
            }
            out.write(cache, 0, cache.length);
            i++;
            offSet = i * segmentSize;
        }
        byte[] data = out.toByteArray();
        out.close();
        return data;
    }

    /**
     *
     * @param contentBase64 content
     * @param privateKeyBase64 private key
     * @return decrypted data
     * @throws NoSuchPaddingException exception
     * @throws NoSuchAlgorithmException exception
     * @throws InvalidKeySpecException exception
     * @throws InvalidKeyException exception
     * @throws IllegalBlockSizeException exception
     * @throws BadPaddingException exception
     * @throws IOException exception
     */
    public static String decipher(String contentBase64, String privateKeyBase64) throws
            NoSuchPaddingException,NoSuchAlgorithmException, InvalidKeySpecException
            ,InvalidKeyException,IllegalBlockSizeException, BadPaddingException, IOException  {
        return decipher(contentBase64, privateKeyBase64, 128);
    }

    /**
     *
     *
     * @param contentBase64 base
     * @param privateKeyBase64 privateKey
     * @param segmentSize size
     * @return decrypted data
     * @throws NoSuchPaddingException exception
     * @throws NoSuchAlgorithmException exception
     * @throws InvalidKeyException exception
     * @throws InvalidKeySpecException exception
     * @throws IllegalBlockSizeException exception
     * @throws BadPaddingException exception
     * @throws IOException exception
     */
    public static String decipher(String contentBase64, String privateKeyBase64, int segmentSize)
            throws NoSuchPaddingException,NoSuchAlgorithmException
            ,InvalidKeyException,InvalidKeySpecException,IllegalBlockSizeException, BadPaddingException, IOException {
        PrivateKey privateKey = getPrivateKey(privateKeyBase64);
        return decipher(contentBase64, privateKey, segmentSize);
    }

    /**
     *
     *
     * @param contentBase64 content
     * @param key key
     * @param segmentSize size
     * @return data
     * @throws NoSuchAlgorithmException exception
     * @throws NoSuchPaddingException exception
     * @throws InvalidKeyException  exception
     * @throws IllegalBlockSizeException exception
     * @throws BadPaddingException exception
     * @throws IOException exception
     */
    public static String decipher(String contentBase64, java.security.Key key, int segmentSize) throws NoSuchAlgorithmException, NoSuchPaddingException
            ,InvalidKeyException,IllegalBlockSizeException, BadPaddingException, IOException {
        // 用私钥解密
        byte[] srcBytes = Base64.decodeBase64(contentBase64);
        // Cipher负责完成加密或解密工作，基于RSA
        Cipher deCipher = Cipher.getInstance(ENCRYPT_TYPE);
        // 根据公钥，对Cipher对象进行初始化
        deCipher.init(Cipher.DECRYPT_MODE, key);
        byte[] decBytes = null;//deCipher.doFinal(srcBytes);
        if (segmentSize > 0)
            decBytes = cipherDoFinal(deCipher, srcBytes, segmentSize); //分段加密
        else
            decBytes = deCipher.doFinal(srcBytes);

        String decrytStr = new String(decBytes);
        return decrytStr;
    }


    /**
     *
     */
    public class KeyPairInfo {
        public KeyPairInfo(int keySize) {
            setKeySize(keySize);
        }

        public KeyPairInfo(String publicKey, String privateKey) {
            setPrivateKey(privateKey);
            setPublicKey(publicKey);
        }

        String privateKey;
        String publicKey;
        int keySize = 0;

        public String getPrivateKey() {
            return privateKey;
        }

        public void setPrivateKey(String privateKey) {
            this.privateKey = privateKey;
        }

        public String getPublicKey() {
            return publicKey;
        }

        public void setPublicKey(String publicKey) {
            this.publicKey = publicKey;
        }

        public int getKeySize() {
            return keySize;
        }

        public void setKeySize(int keySize) {
            this.keySize = keySize;
        }
    }

}
