package cn.ipokerface.common.utils;

import cn.ipokerface.common.exception.ServiceException;
import org.springframework.lang.Nullable;

/**
 * Created by       PokerFace
 * Create Date      2019-11-23.
 * Email:
 * Version          1.0.0
 * <p>
 * Description:     This class has some apis to validate parameter ,  if you want more about validate object or array
 *                  please use {@link org.springframework.util.Assert} instead.
 */
public abstract class AssertUtils {


    /**
     * Assert a boolean expression, throwing an {@code IllegalStateException}
     * if the expression evaluates to {@code false}.
     *
     * <pre class="code">AssetUtils.state(id == null, "name");</pre>
     * @param expression a boolean expression
     * @param name the exception message to use if the assertion fails like 'name is not true'
     * @throws ServiceException if {@code expression} is {@code false}
     */
    public static void isTrue(boolean expression, String name) {
        if (!expression) {
            throw new ServiceException(name + " is not true!");
        }
    }




    /**
     * Assert that an object is {@code null}.
     * <pre class="code">AssetUtils.isNull(value, "name");</pre>
     *
     * @param object the object to check
     * @param name the exception message to use if the assertion fails
     * @throws ServiceException if the object is not {@code null} 'name must be null!'
     */
    public static void isNull(@Nullable Object object, String name) {
        if (object != null) {
            throw new ServiceException(name + " must be null!");
        }
    }


    /**
     * Assert that an object is not {@code null}.
     * <pre class="code">AssetUtils.notNull(clazz, "name");</pre>
     *
     * @param object the object to check
     * @param name the exception message to use if the assertion fails
     *
     * @throws ServiceException if the object is {@code null}
     */
    public static void notNull(@Nullable Object object, String name) {
        if (object == null || "".equals(object)) {
            throw new ServiceException(name + " is required!");
        }
    }


    /**
     * Assert that an object is not {@code null}.
     * <pre class="code">AssetUtils.notNull(clazz, "name");</pre>
     *
     * @param string the string to check
     * @param name the exception message to use if the assertion fails
     *
     * @throws ServiceException if the object is {@code null}
     */
    public static void notNull(@Nullable String string, String name) {
        if (string == null || "".equals(string)) {
            throw new ServiceException(name + " is required!");
        }
    }



    /**
     *
     *
     * @param number numbervalue
     * @param name name of number 'name must not be zero'
     *
     * @throws ServiceException if the number is {@code null} or {@code 0}
     */
    public static void notZero(Number number, String name) {
        notNull(number, name);
        if (number.intValue() == 0) throw new ServiceException(name + "must not be zero");
    }


    /**
     * Assert that value is in size [min~max]
     * <code>AssetUtils.length(obj,name,0,12)</code>
     * <ul>
     *     <li>name out of size[0~12]</li>
     *     <li>name out of size[0~]</li>
     *     <li>name out of size[~12]</li>
     * </ul>
     *
     * @param value value
     * @param min minimum size of value . {@code -1} of skip validate
     * @param max maximum size of value . {@code -1} of skip validate
     * @param name name of value
     *
     * @throws ServiceException if value size out of min ~ max.
     */
    public static void length(String value,String name,  int min, int max){
        if (min >=0 && max >= 0){
            if (value != null && value.length() >= min && value.length() <= max )
                return;
            throw new ServiceException(name + " out of size["+min+"~"+max+"]");
        }
        else if (min >=0 ){
            if (value != null && value.length() >= min) return;

            throw new ServiceException(name + " out of size[~"+max+"]");
        }
        else if (max >= 0){
            if (value != null && value.length()> max)
                throw new ServiceException(name + " out of size["+min+"~]");
        }

    }


    /**
     *  Assert number or range [min~max]
     *
     * @param value value
     * @param name name
     * @param min min
     * @param max max
     *
     */
    public static void range(Number value, String name, int min, int max){
        if (value == null) return;
        int intValue = value.intValue();

        if (min != -1 && intValue < min)
            throw new ServiceException(name + " out of range["+min+"~"+max+"]");
        if (max != -1 && intValue > max){
            throw new ServiceException(name + " out of range["+min+"~"+max+"]");
        }
    }

}
