package cn.ipokerface.common.collection;

import java.io.Serializable;

/**
 * author      PokerFace
 * Date        2017/8/10
 * Email
 *
 * version     1.0
 *
 * Description 双向链表
 */
public class LinkedList<K> implements Serializable
{

    private static final long serialVersionUID = -8276923460661712738L;

    /**
     *  链表的第一个元素
     */
    private LinkedNode<K> _head;
    /**
     *  链表的最后一个元素
     */
    private LinkedNode<K> _tail;
    /**
     *  链表的当前元素
     */
    private LinkedNode<K> _current;

    /**
     *  链表的元素个数
     */
    private int length;

    public LinkedList()
    {
        length = 0;
    }

    /**
     * 返回链表当前的元素个数
     *
     * @return size of this LinkedList object
     */
    public synchronized int size()
    {
        return this.length;
    }

    /**
     *  当前链表是否有数据
     *
     * @return 是否有节点
     */
    public synchronized boolean hasNode()
    {
        return this.length > 0;
    }

    public synchronized boolean hasMore()
    {
        return _current != null;
    }


    /**
     *  将数据节点存入链表中
     *
     * @param node 由链表节点存储的数据  {@link LinkedNode}
     */
    public synchronized void put(LinkedNode<K> node)
    {

        if ( length == 0 )
        {
            _head = node;
            _current = node;
        }
        else
        {
            if (length == 1)
            {
                _head.next = node;
                node.previous = _head;
            }
            else {
                _tail.next = node;
                node.previous = _tail;
            }

        }
        _tail = node;
        node.setContainer(this);
        // 长度+1
        length ++;
    }


    /**
     *  获取链表的下一个元素节点,此操作会使当前节点推移
     *
     * @return 返回下一个节点的节点元素，没有下一个节点时返回空null
     */
    public synchronized LinkedNode<K> take()
    {
        if (length == 0 || _current == null)
            return null;
        LinkedNode<K> node = _current;
        _current = _current.next;
        return node;
    }

    /**
     * 获取链表元素的当前节点元素,此操作不会改变链表的当前节点指针
     *
     * @return 链表为空时返回空
     */
    public synchronized LinkedNode<K> get()
    {
        return _current;
    }

    public synchronized void seekToHead()
    {
        _current = _head;
    }


    /**
     * 移除一个节点
     *
     * @param node 节点
     * @return 是否成功移除
     */
    public synchronized boolean remove(LinkedNode<K> node)
    {
        if ( node == null || length == 0)
            return false;

        if (node.previous == null && node.next == null)
        {
            // 当前只有一个节点时
            _current = null;
            _head = null;
            _tail = null;
        }
        else if (node.previous == null)
        {
            // 为第一个节点时
            node.next.previous = null;
            _head = node.next;
            // 更新当前节点
            if (_current.equals(node))
                _current = node.next;

        }
        else if (node.next == null)
        {
            // 为最后一个节点时
            node.previous.next = null;
            _tail = node.previous;
            // 更新当前节点
            if (_current.equals(node))
                _current = node.previous;
        }
        else
        {
            node.previous.next = node.next;
            node.next.previous = node.previous;
            // 更新当前节点
            if (_current.equals(node))
                _current = node.next;
        }


        // 移除节点
        node.previous = null;
        node.next = null;
        length --;
        return true;

    }

    /**
     *  移除第一个节点
     *
     * @return 被移除的节点
     */
    public synchronized LinkedNode<K> removeFirst()
    {
        LinkedNode<K> result = _head;
        if (remove(_head))
            return result;
        return null;
    }

    /**
     *  移除最后一个节点
     *
     * @return 被移除的节点
     */
    public synchronized LinkedNode<K> removeLast()
    {
        LinkedNode<K> result = _tail;
        if (remove(_tail))
            return result;
        return null;
    }

    /**
     *  清空当前集合
     */
    public synchronized void clear()
    {
        _head = null;
        _tail = null;
        _current = null;
        length = 0;
    }

    /**
     *  集合中是否包含一个对象节点
     * @param node 需要检查的节点
     * @return true : 包含  false: 不包含
     */
    public synchronized boolean contains(LinkedNode<K> node)
    {
        if ( node == null)
            return false;
        if ( length == 0 || _head == null )
            return false;
        if ( node.equals(_head) )
            return true;
        LinkedNode<K> compareNode = _head;
        for (int i=1; i<length; i++)
        {
            compareNode = _head.next;
            if (compareNode == null)
                break;
            if ( node.equals(compareNode) )
                return true;
        }

        return false;
    }

    /**
     *  集合中是否包含一个对象数据
     *
     * @param ele  需要检查的对象
     * @return  true : 包含  false: 不包含
     */
    public synchronized boolean contains(K ele)
    {
        if ( length == 0 || _head == null )
            return false;
        if ( (ele == null && _head.object() == null) || ele.equals(_head.object()) )
            return true;
        LinkedNode<K> compareNode = _head;
        for (int i=1; i<length; i++)
        {
            compareNode = _head.next;
            if (compareNode == null)
                break;
            if ( (ele == null && compareNode.object() == null) ||ele.equals(compareNode.object()) )
                return true;
        }

        return false;
    }




}
