package cn.ipokerface.common.model;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by       PokerFace
 * Create Date      2020-11-03.
 * Email:
 * Version          1.0.0
 * <p>
 * Description:
 */
public class ResultBody {


    private int code;

    private String message;

    private Long timestamp;

    private Map<String, Object> body;


    /**
     * Default status code if success
     */
    public static final int code_success = 10000;

    public static final int code_error = 10001;

    public static final String success = "success";

    public static final String error = "error";


    /**
     *  constructor use {@link ResultBody(int,String,long,Map}
     */
    private ResultBody(){}

    /**
     *  Please use {@link Builder#build()} bu build Response
     *
     * @param code response code
     * @param message  response message
     * @param timestamp response timestamp
     * @param body response body
     */
    private ResultBody(int code,String message, long timestamp, Map<String, Object> body) {
        this.code = code;
        this.body = body;
        this.message = message;
        this.timestamp = timestamp;
    }

    public int getCode() {
        return code;
    }

    public String getMessage() {
        return message;
    }

    public Map<String,Object> getBody() {
        return body;
    }

    public long getTimestamp() {
        return timestamp;
    }



    public static Builder builder(){
        return new Builder();
    }


    public static class Builder{

        private int code = code_success;
        private String message = success;
        private Map<String, Object> data;

        public Builder(){
            this.data = new HashMap<String, Object>();
        }

        /**
         *  push data to body
         *
         * @param key data key
         * @param data data value
         * @return this builder
         */
        public Builder push(String key, Object data){
            this.data.put(key,data);
            return this;
        }

        /**
         *  remove data to body
         *
         * @param key data key
         * @return this builder
         */
        public Builder pop(String key){
            this.data.remove(key);
            return this;
        }

        /**
         *  Push an object into body. This will push all property with name &gt; value into body.
         *
         * @param obj entity object
         * @return this builder
         */
        public Builder object(Object obj){
            if (obj != null) {
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
                for(String key: jsonObject.keySet()) {
                    Object value = jsonObject.get(key);
                    if (value != null && !"".equals(value)) {
                        this.data.put(key, value);
                    }
                }
            }
            return this;
        }

        /**
         *  Push an map into body. This will push all key with key &gt; value into body.
         *
         * @param obj map object
         * @return this builder
         */
        public Builder object(Map<String, Object> obj){
            if (obj != null && obj.size() > 0 ){
                for(Map.Entry<String, Object> entry: obj.entrySet()){
                    if (entry.getValue() != null && !"".equals(entry.getValue())){
                        this.data.put(entry.getKey(), entry.getValue());
                    }
                }
            }

            return this;
        }

        /**
         *  Build an error Response witch <code>message = 'error'</code>
         *
         * @return this builder
         */
        public Builder error(){
            return this.error(error);
        }

        /**
         *  Build an error Response witch <code>message = 'error'</code>
         *
         * @param code suspend key
         * @return this builder
         */
        public Builder error(int code){
            return this.error(code, error);
        }


        /**
         *  Build an error Response witch <code>key = {@link ResultBody#code_error}</code>
         *
         * @param message suspend message
         * @return this builder
         */
        public Builder error(String message){
            return this.error(code_error, message);
        }

        /**
         *  Build an error Response with error . this will clear all data in body data.
         * <code>
         *   {
         *       code : {@link ResultBody#code}
         *       message : ''
         *       timestamp: 123121231232
         *   }
         * </code>
         *
         *
         *
         * @param code
         * @param message
         * @return
         */
        public Builder error(int code, String message){
            this.code = code;
            this.message = message;
            this.data.clear();
            return this;
        }


        /**
         *  Build Response by using builder properties
         *  use {@link #push(String, Object)}  to push data into body
         *  use {@link #pop(String)} to remove data from body
         *  use {@link #object(Object)} to push all data into body
         *
         *
         * @return Response of this builder {@link ResultBody}
         */
        public ResultBody build(){

            return new ResultBody(code, message, System.currentTimeMillis(), data);
        }
    }
}
