package cn.ipokerface.admin.service;


import cn.ipokerface.admin.AdminCacheService;
import cn.ipokerface.admin.entity.*;
import cn.ipokerface.admin.exception.RoleNotExistException;
import cn.ipokerface.admin.mapper.AdminAccountRoleMapper;
import cn.ipokerface.admin.mapper.AdminRoleAuthorityMapper;
import cn.ipokerface.admin.mapper.AdminRoleMapper;
import cn.ipokerface.common.enums.State;
import cn.ipokerface.common.model.query.PaginationSortableSearchQueryModel;
import cn.ipokerface.snowflake.SnowflakeIdGenerator;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by       PokerFace
 * Create Date      2020-05-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */

@Service
public class RoleServiceImpl implements RoleService {

    @Autowired
    private AdminRoleMapper systemRoleMapper;
    @Autowired
    private AdminAccountRoleMapper systemAccountRoleMapper;
    @Autowired
    private AdminRoleAuthorityMapper systemRoleAuthorityMapper;

    @Autowired
    private AdminCacheService adminCacheService;

    @Autowired
    private SnowflakeIdGenerator snowflakeIdGenerator;


    @Override
    public List<AdminRole> pageRole(PaginationSortableSearchQueryModel query) {
        return systemRoleMapper.page(query);
    }

    @Override
    public long totalRole(PaginationSortableSearchQueryModel query) {
        return systemRoleMapper.total(query);
    }

    @Override
    public List<AdminRole> all() {
        return systemRoleMapper.selectAll();
    }

    @Override
    public Long addRole(AdminRole role) {
        role.initial();
        role.setId(snowflakeIdGenerator.nextId());
        systemRoleMapper.insertSelective(role);
        return role.getId();
    }


    @Override
    public void updateRole(AdminRoleUpdate role) {
        AdminRole adminRole = systemRoleMapper.selectByPrimaryKey(role.getId());
        if (adminRole == null) throw new RoleNotExistException();


        systemRoleMapper.updateRole(role.getId(), role.getName(), role.getRemark(),
                adminRole.getVersion() + 1, LocalDateTime.now());

    }

    @Override
    public void remove(List<Long> roleIds) {
        if (CollectionUtils.isEmpty(roleIds)) return;
        systemRoleMapper.delete(roleIds);
        // 删除账号角色
        systemAccountRoleMapper.deleteByAccountIds(roleIds);

    }

    @Override
    public void auth(List<Long> authIds, Long roleId) {
        List<Long> roleIds = new ArrayList<>();
        roleIds.add(roleId);
        systemRoleAuthorityMapper.deleteByRoleIds(roleIds);
        if (CollectionUtils.isNotEmpty(authIds))
        {
            List<AdminRoleAuthority> roleAuthorities = new ArrayList<>(authIds.size());

            for(Long authId: authIds)
            {
                AdminRoleAuthority roleAuthority = new AdminRoleAuthority();

                roleAuthority.setId(snowflakeIdGenerator.nextId());
                roleAuthority.initial();
                roleAuthority.setRoleId(roleId);
                roleAuthority.setAuthorityId(authId);

                roleAuthorities.add(roleAuthority);
            }

            // 🈯️批量插入  一次100条
            int volume = 100;
            int count = roleAuthorities.size() / volume + (roleAuthorities.size() % volume > 0 ? 1:0);
            for(int i=0; i< count; i++)
            {
                int start = i * volume;
                int end = (i+1) * volume;
                if (end > roleAuthorities.size()) end = roleAuthorities.size();

                systemRoleAuthorityMapper.insertBatch(roleAuthorities.subList(start, end));
            }
        }


        adminCacheService.setRoleAuthorities(roleId, authIds);
    }

    @Override
    public List<AdminRole> accountRoles(Long accountId) {
        return systemAccountRoleMapper.selectByAccountId(accountId);
    }


    @Override
    public void accountRoleConfig(Long accountId, List<Long> roleIds) {
        systemAccountRoleMapper.deleteByAccountIds(cn.ipokerface.common.utils.CollectionUtils.asList(accountId));
        if (CollectionUtils.isNotEmpty(roleIds)) {
            List<AdminAccountRole> accountRoles = new ArrayList<>(roleIds.size());
            for(Long roleId: roleIds) {
                AdminAccountRole systemAccountRole = new AdminAccountRole();
                systemAccountRole.setId(snowflakeIdGenerator.nextId());
                systemAccountRole.setState(State.State_C);
                systemAccountRole.setVersion(0);
                systemAccountRole.setCreatedTime(LocalDateTime.now());
                systemAccountRole.setUpdatedTime(LocalDateTime.now());
                systemAccountRole.setAccountId(accountId);
                systemAccountRole.setRoleId(roleId);
                accountRoles.add(systemAccountRole);
            }

            // 🈯️批量插入  一次50条
            int volume = 50;
            int count = accountRoles.size() / volume + (accountRoles.size() % volume > 0 ? 1:0);

            for(int i=0; i< count; i++)
            {

                int start = i * volume;
                int end = (i+1) * volume;
                if (end > accountRoles.size()) end = accountRoles.size();

                systemAccountRoleMapper.insertBatch(accountRoles.subList(start, end));
            }
        }
        adminCacheService.setAccountRoles(accountId, roleIds);
    }


}
