package cn.iotwasu.hutool.core.builder;

import cn.iotwasu.hutool.core.lang.Assert;
import cn.iotwasu.hutool.core.util.ArrayUtil;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * Assists in implementing {@link Object#hashCode()} methods.
 *
 * <p>This class enables a good <code>hashCode</code> method to be built for any class. It follows
 * the rules laid out in the book <a
 * href="http://www.oracle.com/technetwork/java/effectivejava-136174.html">Effective Java</a> by
 * Joshua Bloch. Writing a good <code>hashCode</code> method is actually quite difficult. This class
 * aims to simplify the process.
 *
 * <p>The following is the approach taken. When appending a data field, the current total is
 * multiplied by the multiplier then a relevant value for that data type is added. For example, if
 * the current hashCode is 17, and the multiplier is 37, then appending the integer 45 will create a
 * hashcode of 674, namely 17 * 37 + 45.
 *
 * <p>All relevant fields from the object should be included in the <code>hashCode</code> method.
 * Derived fields may be excluded. In general, any field used in the <code>equals</code> method must
 * be used in the <code>hashCode</code> method.
 *
 * <p>To use this class write code as follows:
 *
 * <pre>
 * public class Person {
 *   String name;
 *   int age;
 *   boolean smoker;
 *   ...
 *
 *   public int hashCode() {
 *     // you pick a hard-coded, randomly chosen, non-zero, odd number
 *     // ideally different for each class
 *     return new HashCodeBuilder(17, 37).
 *       append(name).
 *       append(age).
 *       append(smoker).
 *       toHashCode();
 *   }
 * }
 * </pre>
 *
 * <p>If required, the superclass <code>hashCode()</code> can be added using {@link #appendSuper}.
 *
 * <p>Alternatively, there is a method that uses reflection to determine the fields to test. Because
 * these fields are usually private, the method, <code>reflectionHashCode</code>, uses <code>
 * AccessibleObject.setAccessible</code> to change the visibility of the fields. This will fail
 * under a security manager, unless the appropriate permissions are set up correctly. It is also
 * slower than testing explicitly.
 *
 * <p>A typical invocation for this method would look like:
 *
 * <pre>
 * public int hashCode() {
 *   return HashCodeBuilder.reflectionHashCode(this);
 * }
 * </pre>
 *
 * TODO 待整理 来自于Apache-Commons-Lang3
 *
 * @author looly，Apache-Commons
 * @since 4.2.2
 */
public class HashCodeBuilder implements Builder<Integer> {
  private static final long serialVersionUID = 1L;

  /** The default initial value to use in reflection hash code building. */
  private static final int DEFAULT_INITIAL_VALUE = 17;

  /** The default multipler value to use in reflection hash code building. */
  private static final int DEFAULT_MULTIPLIER_VALUE = 37;

  /**
   * A registry of objects used by reflection methods to detect cyclical object references and avoid
   * infinite loops.
   *
   * @since 2.3
   */
  private static final ThreadLocal<Set<IDKey>> REGISTRY = new ThreadLocal<Set<IDKey>>();

  /*
   * NOTE: we cannot store the actual objects in a HashSet, as that would use the very hashCode()
   * we are in the process of calculating.
   *
   * So we generate a one-to-one mapping from the original object to a new object.
   *
   * Now HashSet uses equals() to determine if two elements with the same hashcode really
   * are equal, so we also need to ensure that the replacement objects are only equal
   * if the original objects are identical.
   *
   * The original implementation (2.4 and before) used the System.indentityHashCode()
   * method - however this is not guaranteed to generate unique ids (e.g. LANG-459)
   *
   * We now use the IDKey helper class (adapted from org.apache.axis.utils.IDKey)
   * to disambiguate the duplicate ids.
   */
  /** Constant to use in building the hashCode. */
  private final int iConstant;
  /** Running total of the hashCode. */
  private int iTotal = 0;

  /** Uses two hard coded choices for the constants needed to build a <code>hashCode</code>. */
  public HashCodeBuilder() {
    iConstant = 37;
    iTotal = 17;
  }

  /**
   * Two randomly chosen, odd numbers must be passed in. Ideally these should be different for each
   * class, however this is not vital.
   *
   * <p>Prime numbers are preferred, especially for the multiplier.
   *
   * @param initialOddNumber an odd number used as the initial value
   * @param multiplierOddNumber an odd number used as the multiplier
   * @throws IllegalArgumentException if the number is even
   */
  public HashCodeBuilder(final int initialOddNumber, final int multiplierOddNumber) {
    Assert.isTrue(initialOddNumber % 2 != 0, "HashCodeBuilder requires an odd initial value");
    Assert.isTrue(multiplierOddNumber % 2 != 0, "HashCodeBuilder requires an odd multiplier");
    iConstant = multiplierOddNumber;
    iTotal = initialOddNumber;
  }

  /**
   * Returns the registry of objects being traversed by the reflection methods in the current
   * thread.
   *
   * @return Set the registry of objects being traversed
   * @since 2.3
   */
  private static Set<IDKey> getRegistry() {
    return REGISTRY.get();
  }

  /**
   * Returns <code>true</code> if the registry contains the given object. Used by the reflection
   * methods to avoid infinite loops.
   *
   * @param value The object to lookup in the registry.
   * @return boolean <code>true</code> if the registry contains the given object.
   * @since 2.3
   */
  private static boolean isRegistered(final Object value) {
    final Set<IDKey> registry = getRegistry();
    return registry != null && registry.contains(new IDKey(value));
  }

  /**
   * Appends the fields and values defined by the given object of the given <code>Class</code>.
   *
   * @param object the object to append details of
   * @param clazz the class to append details of
   * @param builder the builder to append to
   * @param useTransients whether to use transient fields
   * @param excludeFields Collection of String field names to exclude from use in calculation of
   *     hash code
   */
  private static void reflectionAppend(
      final Object object,
      final Class<?> clazz,
      final HashCodeBuilder builder,
      final boolean useTransients,
      final String[] excludeFields) {
    if (isRegistered(object)) {
      return;
    }
    try {
      register(object);
      final Field[] fields = clazz.getDeclaredFields();
      AccessibleObject.setAccessible(fields, true);
      for (final Field field : fields) {
        if (false == ArrayUtil.contains(excludeFields, field.getName())
            && (field.getName().indexOf('$') == -1)
            && (useTransients || !Modifier.isTransient(field.getModifiers()))
            && (!Modifier.isStatic(field.getModifiers()))) {
          try {
            final Object fieldValue = field.get(object);
            builder.append(fieldValue);
          } catch (final IllegalAccessException e) {
            // this can't happen. Would get a Security exception instead
            // throw a runtime exception in case the impossible happens.
            throw new InternalError("Unexpected IllegalAccessException");
          }
        }
      }
    } finally {
      unregister(object);
    }
  }

  /**
   * Uses reflection to build a valid hash code from the fields of {@code object}.
   *
   * <p>It uses <code>AccessibleObject.setAccessible</code> to gain access to private fields. This
   * means that it will throw a security exception if run under a security manager, if the
   * permissions are not set up correctly. It is also not as efficient as testing explicitly.
   *
   * <p>Transient members will be not be used, as they are likely derived fields, and not part of
   * the value of the <code>Object</code>.
   *
   * <p>Static fields will not be tested. Superclass fields will be included.
   *
   * <p>Two randomly chosen, non-zero, odd numbers must be passed in. Ideally these should be
   * different for each class, however this is not vital. Prime numbers are preferred, especially
   * for the multiplier.
   *
   * @param initialNonZeroOddNumber a non-zero, odd number used as the initial value. This will be
   *     the returned value if no fields are found to include in the hash code
   * @param multiplierNonZeroOddNumber a non-zero, odd number used as the multiplier
   * @param object the Object to create a <code>hashCode</code> for
   * @return int hash code
   * @throws IllegalArgumentException if the Object is <code>null</code>
   * @throws IllegalArgumentException if the number is zero or even
   */
  public static int reflectionHashCode(
      final int initialNonZeroOddNumber,
      final int multiplierNonZeroOddNumber,
      final Object object) {
    return reflectionHashCode(
        initialNonZeroOddNumber, multiplierNonZeroOddNumber, object, false, null);
  }

  // -------------------------------------------------------------------------

  /**
   * Uses reflection to build a valid hash code from the fields of {@code object}.
   *
   * <p>It uses <code>AccessibleObject.setAccessible</code> to gain access to private fields. This
   * means that it will throw a security exception if run under a security manager, if the
   * permissions are not set up correctly. It is also not as efficient as testing explicitly.
   *
   * <p>If the TestTransients parameter is set to <code>true</code>, transient members will be
   * tested, otherwise they are ignored, as they are likely derived fields, and not part of the
   * value of the <code>Object</code>.
   *
   * <p>Static fields will not be tested. Superclass fields will be included.
   *
   * <p>Two randomly chosen, non-zero, odd numbers must be passed in. Ideally these should be
   * different for each class, however this is not vital. Prime numbers are preferred, especially
   * for the multiplier.
   *
   * @param initialNonZeroOddNumber a non-zero, odd number used as the initial value. This will be
   *     the returned value if no fields are found to include in the hash code
   * @param multiplierNonZeroOddNumber a non-zero, odd number used as the multiplier
   * @param object the Object to create a <code>hashCode</code> for
   * @param testTransients whether to include transient fields
   * @return int hash code
   * @throws IllegalArgumentException if the Object is <code>null</code>
   * @throws IllegalArgumentException if the number is zero or even
   */
  public static int reflectionHashCode(
      final int initialNonZeroOddNumber,
      final int multiplierNonZeroOddNumber,
      final Object object,
      final boolean testTransients) {
    return reflectionHashCode(
        initialNonZeroOddNumber, multiplierNonZeroOddNumber, object, testTransients, null);
  }

  /**
   * Uses reflection to build a valid hash code from the fields of {@code object}.
   *
   * <p>It uses <code>AccessibleObject.setAccessible</code> to gain access to private fields. This
   * means that it will throw a security exception if run under a security manager, if the
   * permissions are not set up correctly. It is also not as efficient as testing explicitly.
   *
   * <p>If the TestTransients parameter is set to <code>true</code>, transient members will be
   * tested, otherwise they are ignored, as they are likely derived fields, and not part of the
   * value of the <code>Object</code>.
   *
   * <p>Static fields will not be included. Superclass fields will be included up to and including
   * the specified superclass. A null superclass is treated as java.lang.Object.
   *
   * <p>Two randomly chosen, non-zero, odd numbers must be passed in. Ideally these should be
   * different for each class, however this is not vital. Prime numbers are preferred, especially
   * for the multiplier.
   *
   * @param <T> the type of the object involved
   * @param initialNonZeroOddNumber a non-zero, odd number used as the initial value. This will be
   *     the returned value if no fields are found to include in the hash code
   * @param multiplierNonZeroOddNumber a non-zero, odd number used as the multiplier
   * @param object the Object to create a <code>hashCode</code> for
   * @param testTransients whether to include transient fields
   * @param reflectUpToClass the superclass to reflect up to (inclusive), may be <code>null</code>
   * @param excludeFields array of field names to exclude from use in calculation of hash code
   * @return int hash code
   * @throws IllegalArgumentException if the Object is <code>null</code>
   * @throws IllegalArgumentException if the number is zero or even
   * @since 2.0
   */
  public static <T> int reflectionHashCode(
      final int initialNonZeroOddNumber,
      final int multiplierNonZeroOddNumber,
      final T object,
      final boolean testTransients,
      final Class<? super T> reflectUpToClass,
      final String... excludeFields) {

    if (object == null) {
      throw new IllegalArgumentException("The object to build a hash code for must not be null");
    }
    final HashCodeBuilder builder =
        new HashCodeBuilder(initialNonZeroOddNumber, multiplierNonZeroOddNumber);
    Class<?> clazz = object.getClass();
    reflectionAppend(object, clazz, builder, testTransients, excludeFields);
    while (clazz.getSuperclass() != null && clazz != reflectUpToClass) {
      clazz = clazz.getSuperclass();
      reflectionAppend(object, clazz, builder, testTransients, excludeFields);
    }
    return builder.toHashCode();
  }

  /**
   * Uses reflection to build a valid hash code from the fields of {@code object}.
   *
   * <p>This constructor uses two hard coded choices for the constants needed to build a hash code.
   *
   * <p>It uses <code>AccessibleObject.setAccessible</code> to gain access to private fields. This
   * means that it will throw a security exception if run under a security manager, if the
   * permissions are not set up correctly. It is also not as efficient as testing explicitly.
   *
   * <p>If the TestTransients parameter is set to <code>true</code>, transient members will be
   * tested, otherwise they are ignored, as they are likely derived fields, and not part of the
   * value of the <code>Object</code>.
   *
   * <p>Static fields will not be tested. Superclass fields will be included. If no fields are found
   * to include in the hash code, the result of this method will be constant.
   *
   * @param object the Object to create a <code>hashCode</code> for
   * @param testTransients whether to include transient fields
   * @return int hash code
   * @throws IllegalArgumentException if the object is <code>null</code>
   */
  public static int reflectionHashCode(final Object object, final boolean testTransients) {
    return reflectionHashCode(
        DEFAULT_INITIAL_VALUE, DEFAULT_MULTIPLIER_VALUE, object, testTransients, null);
  }

  /**
   * Uses reflection to build a valid hash code from the fields of {@code object}.
   *
   * <p>This constructor uses two hard coded choices for the constants needed to build a hash code.
   *
   * <p>It uses <code>AccessibleObject.setAccessible</code> to gain access to private fields. This
   * means that it will throw a security exception if run under a security manager, if the
   * permissions are not set up correctly. It is also not as efficient as testing explicitly.
   *
   * <p>Transient members will be not be used, as they are likely derived fields, and not part of
   * the value of the <code>Object</code>.
   *
   * <p>Static fields will not be tested. Superclass fields will be included. If no fields are found
   * to include in the hash code, the result of this method will be constant.
   *
   * @param object the Object to create a <code>hashCode</code> for
   * @param excludeFields Collection of String field names to exclude from use in calculation of
   *     hash code
   * @return int hash code
   * @throws IllegalArgumentException if the object is <code>null</code>
   */
  public static int reflectionHashCode(
      final Object object, final Collection<String> excludeFields) {
    return reflectionHashCode(object, ArrayUtil.toArray(excludeFields, String.class));
  }

  /**
   * Uses reflection to build a valid hash code from the fields of {@code object}.
   *
   * <p>This constructor uses two hard coded choices for the constants needed to build a hash code.
   *
   * <p>It uses <code>AccessibleObject.setAccessible</code> to gain access to private fields. This
   * means that it will throw a security exception if run under a security manager, if the
   * permissions are not set up correctly. It is also not as efficient as testing explicitly.
   *
   * <p>Transient members will be not be used, as they are likely derived fields, and not part of
   * the value of the <code>Object</code>.
   *
   * <p>Static fields will not be tested. Superclass fields will be included. If no fields are found
   * to include in the hash code, the result of this method will be constant.
   *
   * @param object the Object to create a <code>hashCode</code> for
   * @param excludeFields array of field names to exclude from use in calculation of hash code
   * @return int hash code
   * @throws IllegalArgumentException if the object is <code>null</code>
   */
  public static int reflectionHashCode(final Object object, final String... excludeFields) {
    return reflectionHashCode(
        DEFAULT_INITIAL_VALUE, DEFAULT_MULTIPLIER_VALUE, object, false, null, excludeFields);
  }

  /**
   * Registers the given object. Used by the reflection methods to avoid infinite loops.
   *
   * @param value The object to register.
   */
  static void register(final Object value) {
    synchronized (HashCodeBuilder.class) {
      if (getRegistry() == null) {
        REGISTRY.set(new HashSet<IDKey>());
      }
    }
    getRegistry().add(new IDKey(value));
  }

  /**
   * Unregisters the given object.
   *
   * <p>Used by the reflection methods to avoid infinite loops.
   *
   * @param value The object to unregister.
   * @since 2.3
   */
  static void unregister(final Object value) {
    Set<IDKey> registry = getRegistry();
    if (registry != null) {
      registry.remove(new IDKey(value));
      synchronized (HashCodeBuilder.class) {
        // read again
        registry = getRegistry();
        if (registry != null && registry.isEmpty()) {
          REGISTRY.remove();
        }
      }
    }
  }

  /**
   * Append a <code>hashCode</code> for a <code>boolean</code>.
   *
   * <p>This adds <code>1</code> when true, and <code>0</code> when false to the <code>hashCode
   * </code>.
   *
   * <p>This is in contrast to the standard <code>java.lang.Boolean.hashCode</code> handling, which
   * computes a <code>hashCode</code> value of <code>1231</code> for <code>java.lang.Boolean</code>
   * instances that represent <code>true</code> or <code>1237</code> for <code>java.lang.Boolean
   * </code> instances that represent <code>false</code>.
   *
   * <p>This is in accordance with the <i>Effective Java</i> design.
   *
   * @param value the boolean to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final boolean value) {
    iTotal = iTotal * iConstant + (value ? 0 : 1);
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>boolean</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final boolean[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final boolean element : array) {
        append(element);
      }
    }
    return this;
  }

  // -------------------------------------------------------------------------

  /**
   * Append a <code>hashCode</code> for a <code>byte</code>.
   *
   * @param value the byte to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final byte value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  // -------------------------------------------------------------------------

  /**
   * Append a <code>hashCode</code> for a <code>byte</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final byte[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final byte element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>char</code>.
   *
   * @param value the char to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final char value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>char</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final char[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final char element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>double</code>.
   *
   * @param value the double to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final double value) {
    return append(Double.doubleToLongBits(value));
  }

  /**
   * Append a <code>hashCode</code> for a <code>double</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final double[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final double element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>float</code>.
   *
   * @param value the float to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final float value) {
    iTotal = iTotal * iConstant + Float.floatToIntBits(value);
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>float</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final float[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final float element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for an <code>int</code>.
   *
   * @param value the int to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final int value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  /**
   * Append a <code>hashCode</code> for an <code>int</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final int[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final int element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>long</code>.
   *
   * @param value the long to add to the <code>hashCode</code>
   * @return this
   */
  // NOTE: This method uses >> and not >>> as Effective Java and
  //       Long.hashCode do. Ideally we should switch to >>> at
  //       some stage. There are backwards compat issues, so
  //       that will have to wait for the time being. cf LANG-342.
  public HashCodeBuilder append(final long value) {
    iTotal = iTotal * iConstant + ((int) (value ^ (value >> 32)));
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>long</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final long[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final long element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for an <code>Object</code>.
   *
   * @param object the Object to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final Object object) {
    if (object == null) {
      iTotal = iTotal * iConstant;

    } else {
      if (object.getClass().isArray()) {
        // 'Switch' on type of array, to dispatch to the correct handler
        // This handles multi dimensional arrays
        if (object instanceof long[]) {
          append((long[]) object);
        } else if (object instanceof int[]) {
          append((int[]) object);
        } else if (object instanceof short[]) {
          append((short[]) object);
        } else if (object instanceof char[]) {
          append((char[]) object);
        } else if (object instanceof byte[]) {
          append((byte[]) object);
        } else if (object instanceof double[]) {
          append((double[]) object);
        } else if (object instanceof float[]) {
          append((float[]) object);
        } else if (object instanceof boolean[]) {
          append((boolean[]) object);
        } else {
          // Not an array of primitives
          append((Object[]) object);
        }
      } else {
        iTotal = iTotal * iConstant + object.hashCode();
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for an <code>Object</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final Object[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final Object element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>short</code>.
   *
   * @param value the short to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final short value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  /**
   * Append a <code>hashCode</code> for a <code>short</code> array.
   *
   * @param array the array to add to the <code>hashCode</code>
   * @return this
   */
  public HashCodeBuilder append(final short[] array) {
    if (array == null) {
      iTotal = iTotal * iConstant;
    } else {
      for (final short element : array) {
        append(element);
      }
    }
    return this;
  }

  /**
   * Adds the result of super.hashCode() to this builder.
   *
   * @param superHashCode the result of calling <code>super.hashCode()</code>
   * @return this HashCodeBuilder, used to chain calls.
   * @since 2.0
   */
  public HashCodeBuilder appendSuper(final int superHashCode) {
    iTotal = iTotal * iConstant + superHashCode;
    return this;
  }

  /**
   * Return the computed <code>hashCode</code>.
   *
   * @return <code>hashCode</code> based on the fields appended
   */
  public int toHashCode() {
    return iTotal;
  }

  /**
   * Returns the computed <code>hashCode</code>.
   *
   * @return <code>hashCode</code> based on the fields appended
   * @since 3.0
   */
  @Override
  public Integer build() {
    return Integer.valueOf(toHashCode());
  }

  /**
   * The computed <code>hashCode</code> from toHashCode() is returned due to the likelihood of bugs
   * in mis-calling toHashCode() and the unlikeliness of it mattering what the hashCode for
   * HashCodeBuilder itself is.
   *
   * @return <code>hashCode</code> based on the fields appended
   * @since 2.5
   */
  @Override
  public int hashCode() {
    return toHashCode();
  }
}
