package cn.imkarl.sqldsl.table

import cn.imkarl.core.common.json.JsonUtils
import cn.imkarl.sqldsl.DuplicateColumnException
import cn.imkarl.sqldsl.column.*
import com.google.gson.reflect.TypeToken
import java.math.BigDecimal
import java.time.LocalDate
import java.time.LocalDateTime
import java.util.*
import kotlin.reflect.KClass

/**
 * Base class for any simple table.
 *
 * If you want to reference your table use [IdTable] instead.
 *
 * @param name Table name, by default name will be resolved from a class name with "Table" suffix removed (if present)
 */
abstract class Table(name: String = "") {
    open val tableName: String = if (name.isNotEmpty()) name else javaClass.simpleName

    /** Returns all the columns defined on the table. */
    private val _columns = mutableListOf<Column<*>>()
    open val columns: List<Column<*>> get() = _columns

    /**
     * Returns the primary key of the table if present, `null` otherwise.
     *
     * Currently, it is initialized with existing keys defined by [Column.primaryKey] function for a backward compatibility,
     * but you have to define it explicitly by overriding that val instead.
     */
    @Suppress("UNCHECKED_CAST")
    val primaryKey: PrimaryKey<Any>?
        get() = columns.firstOrNull { it is PrimaryKey<*> } as PrimaryKey<Any>?

    /** Returns the first auto-increment column on the table. */
    val autoIncColumn: Column<*>? get() = primaryKey?.takeIf { it.isAutoIncrement }


    fun <T : Any> Column<T>.primaryKey(
        pkName: String = "",
        isAutoIncrement: Boolean = false
    ): PrimaryKey<T> = replaceColumn(
        oldColumn = this,
        newColumn = PrimaryKey(this, if (pkName.isNotEmpty()) pkName else "pk_${table.tableName}", isAutoIncrement)
    )


    fun newRow(): Row {
        return Row()
    }

    fun newRow(builder: (Row) -> Unit): Row {
        return Row().also(builder)
    }

    inner class Row internal constructor() {
        internal val table: Table get() = this@Table
        internal val values: MutableMap<Column<*>, Any?> = LinkedHashMap()

        internal fun set(column: Column<*>, value: Any?) {
            val columnValue = value ?: column.defaultValueFun?.invoke()
            when {
                !column.columnType.nullable && columnValue == null -> error("Trying to set null to not nullable column $column")
                else -> values[column] = columnValue
            }
        }

        operator fun <S> set(column: Column<S>, value: S) {
            val columnValue = value ?: column.defaultValueFun?.invoke()
            when {
                !column.columnType.nullable && columnValue == null -> error("Trying to set null to not nullable column $column")
                else -> values[column] = columnValue
            }
        }


        infix operator fun <T : Any?> get(column: Column<T>): T {
            val value = (values[column] as T?) ?: column.defaultValueFun?.invoke()
            return value as T
        }

        fun clear() {
            values.clear()
        }

        override fun toString(): String {
            return "Table[${table.tableName}].Row: ${values.entries.joinToString(", ") { "${it.key.columnName}=${it.value}" }}"
        }
    }


    // Column registration

    /** Adds a column of the specified [type] and with the specified [name] to the table. */
    fun <T : Any?> registerColumn(name: String, type: ColumnType<T>): Column<T> =
        Column<T>(this, name, type).also { _columns.addColumn(it) }

    /**
     * Replaces the specified [oldColumn] with the specified [newColumn] in the table.
     * Mostly used internally by the library.
     */
    fun <TColumn : Column<*>> replaceColumn(oldColumn: Column<*>, newColumn: TColumn): TColumn {
        _columns.remove(oldColumn)
        _columns.addColumn(newColumn)
        return newColumn
    }

    private fun MutableList<Column<*>>.addColumn(column: Column<*>) {
        if (this.any { it.columnName == column.columnName }) {
            throw DuplicateColumnException(column.columnName, tableName)
        }
        this.add(column)
    }


    // Numeric columns

    /** Creates a numeric column, with the specified [name], for storing 1-byte integers. */
    fun byte(name: String): Column<Byte> = registerColumn(name, ByteColumnType())

    /** Creates a numeric column, with the specified [name], for storing 2-byte integers. */
    fun short(name: String): Column<Short> = registerColumn(name, ShortColumnType())

    /** Creates a numeric column, with the specified [name], for storing 4-byte integers. */
    fun integer(name: String): Column<Int> = registerColumn(name, IntegerColumnType())

    /** Creates a numeric column, with the specified [name], for storing 8-byte integers. */
    fun long(name: String): Column<Long> = registerColumn(name, LongColumnType())

    /** Creates a numeric column, with the specified [name], for storing 4-byte (single precision) floating-point numbers. */
    fun float(name: String): Column<Float> = registerColumn(name, FloatColumnType())

    /** Creates a numeric column, with the specified [name], for storing 8-byte (double precision) floating-point numbers. */
    fun double(name: String): Column<Double> = registerColumn(name, DoubleColumnType())

    /**
     * Creates a numeric column, with the specified [name], for storing numbers with the specified [precision] and [scale].
     *
     * To store the decimal `123.45`, [precision] would have to be set to 5 (as there are five digits in total) and
     * [scale] to 2 (as there are two digits behind the decimal point).
     *
     * @param name Name of the column.
     * @param precision Total count of significant digits in the whole number, that is, the number of digits to both sides of the decimal point.
     * @param scale Count of decimal digits in the fractional part.
     */
    fun decimal(name: String, precision: Int, scale: Int): Column<BigDecimal> =
        registerColumn(name, DecimalColumnType(precision, scale))

    // Character columns

    /** Creates a character column, with the specified [name], for storing single characters. */
    fun character(name: String): Column<Char> = registerColumn(name, CharacterColumnType())

    /**
     * Creates a character column, with the specified [name], for storing strings with the specified [length] using the specified text [collate] type.
     * If no collate type is specified then the database default is used.
     */
    fun char(name: String, length: Int): Column<String> = registerColumn(name, CharColumnType(length))

    /**
     * Creates a character column, with the specified [name], for storing strings with the specified maximum [length] using the specified text [collate] type.
     * If no collate type is specified then the database default is used.
     */
    fun varchar(name: String, length: Int): Column<String> = registerColumn(name, VarCharColumnType(length))

    /**
     * Creates a character column, with the specified [name], for storing strings of arbitrary length using the specified [collate] type.
     * If no collate type is specified then the database default is used.
     *
     * Some database drivers do not load text content immediately (by performance and memory reasons)
     * what means that you can obtain column value only within the open transaction.
     * If you desire to make content available outside the transaction use [eagerLoading] param.
     */
    fun text(name: String): Column<String> = registerColumn(name, TextColumnType())

    fun <T: Any> json(name: String, length: Int): Column<T> = registerColumn(name, CustomColumnType(
        columnType = VarCharColumnType(length),
        valueFromDB = { json ->
            JsonUtils.fromJson(json, object : TypeToken<T>() {})!!
        },
        valueToDB = {
            JsonUtils.toJson(it)
        },
        _defaultValue = JsonUtils.fromJson("", object : TypeToken<T>() {})!!
    ))

    fun <T: Any?, DBTYPE: Any> custom(
        name: String,
        columnType: ColumnType<DBTYPE>,
        valueFromDB: (value: DBTYPE) -> T,
        valueToDB: (value: T) -> DBTYPE,
        defaultValue: T
    ): Column<T> = registerColumn(name, CustomColumnType(
        columnType = columnType.also { it.nullable = true },
        valueFromDB = valueFromDB,
        valueToDB = valueToDB,
        _defaultValue = defaultValue
    ))


    // Binary columns

    /**
     * Creates a binary column, with the specified [name], for storing BLOBs.
     */
    fun blob(name: String): Column<ByteArray> = registerColumn(name, BlobColumnType())

    // Boolean columns

    /** Creates a column, with the specified [name], for storing boolean values. */
    fun bool(name: String): Column<Boolean> = registerColumn(name, BooleanColumnType())

    // Enumeration columns

    /** Creates an enumeration column, with the specified [name], for storing enums of type [klass] by their ordinal. */
    fun <T : Enum<T>> enumeration(name: String, klass: KClass<T>, defaultValue: T): Column<T> =
        registerColumn(name, EnumColumnType(klass, defaultValue))

    /**
     * A date column to store a date.
     *
     * @param name The column name
     */
    fun Table.date(name: String): Column<LocalDate> = registerColumn(name, DateColumnType())

    /**
     * A datetime column to store both a date and a time.
     *
     * @param name The column name
     */
    fun Table.datetime(name: String): Column<LocalDateTime> = registerColumn(name, DateTimeColumnType())


    // Auto-generated values

    /**
     * Make @receiver column an auto-increment to generate its values in a database.
     * Only integer and long columns supported.
     * Some databases like a PostgreSQL supports auto-increment via sequences.
     * In that case you should provide a name with [idSeqName] param and Exposed will create a sequence for you.
     * If you already have a sequence in a database just use its name in [idSeqName].
     *
     * @param idSeqName an optional parameter to provide a sequence name
     */
    fun <N : Any> PrimaryKey<N>.autoIncrement(): Column<N> = apply {
        this.isAutoIncrement = true
    }

    // Miscellaneous

    /** Marks this column as nullable. */
    fun <T : Any> Column<T>.nullable(): Column<T?> {
        val newColumn = Column<T?>(table, columnName, columnType)
        newColumn.columnType.nullable = true
        return replaceColumn(this, newColumn)
    }

    fun <T : Any> Column<T?>.notNull(defaultValueFun: () -> T): Column<T> {
        val newColumn = Column<T>(table, columnName, columnType)
        newColumn.columnType.nullable = false
        newColumn.default(defaultValueFun)
        return replaceColumn(this, newColumn)
    }

    fun <T : Any> Column<T?>.notNull(defaultValue: T): Column<T> {
        return notNull(defaultValueFun = { defaultValue })
    }


    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is Table) return false

        if (tableName != other.tableName) return false
        if (_columns != other._columns) return false

        return true
    }

    override fun hashCode(): Int {
        return tableName.hashCode()
    }

}
