package cn.imkarl.sqldsl.sql

import cn.imkarl.sqldsl.column.*
import cn.imkarl.sqldsl.database.SqlColumnValue

/**
 * SQLExpression
 */
abstract class SQLExpression {

    companion object {
        fun empty(): SQLExpression {
            return EmptySQLExpression()
        }
    }

    abstract fun toSQLBuilder(): String

    abstract fun bindArgs(): List<SqlColumnValue<*>>

}

fun SQLExpression.and(expression: SQLExpression): SQLExpression {
    if (this is EmptySQLExpression) {
        return expression
    }
    if (expression is EmptySQLExpression) {
        return this
    }
    return AndOp(listOf(this, expression))
}

fun SQLExpression.or(expression: SQLExpression): SQLExpression {
    if (this is EmptySQLExpression) {
        return expression
    }
    if (expression is EmptySQLExpression) {
        return this
    }
    return OrOp(listOf(this, expression))
}

fun <T: Any?> Column<T>.eq(value: T): SQLExpression {
    return EqOp(this, value)
}

fun <T: Any?> Column<T>.inList(list: Iterable<T>): SQLExpression {
    return InListOrNotInListOp(this, list, true)
}
fun <T: Any?> Column<T>.notInList(list: Iterable<T>): SQLExpression {
    return InListOrNotInListOp(this, list, false)
}

fun <T: Any?> Column<T>.like(arg: String): SQLExpression {
    return LikeOp(this, arg, true)
}
fun <T: Any?> Column<T>.notLike(arg: String): SQLExpression {
    return LikeOp(this, arg, false)
}

/**
 * 小于
 */
fun Column<Int>.less(value: Int): SQLExpression {
    return ComparisonOp(this, value, "<")
}
/**
 * 小于等于
 */
fun Column<Int>.lessEq(value: Int): SQLExpression {
    return ComparisonOp(this, value, "<=")
}
/**
 * 大于
 */
fun Column<Int>.greater(value: Int): SQLExpression {
    return ComparisonOp(this, value, ">")
}
/**
 * 大于等于
 */
fun Column<Int>.greaterEq(value: Int): SQLExpression {
    return ComparisonOp(this, value, ">=")
}

/**
 * 小于
 */
fun Column<Long>.less(value: Long): SQLExpression {
    return ComparisonOp(this, value, "<")
}
/**
 * 小于等于
 */
fun Column<Long>.lessEq(value: Long): SQLExpression {
    return ComparisonOp(this, value, "<=")
}
/**
 * 大于
 */
fun Column<Long>.greater(value: Long): SQLExpression {
    return ComparisonOp(this, value, ">")
}
/**
 * 大于等于
 */
fun Column<Long>.greaterEq(value: Long): SQLExpression {
    return ComparisonOp(this, value, ">=")
}




private class EmptySQLExpression : SQLExpression() {
    override fun toSQLBuilder() = ""
    override fun bindArgs() = emptyList<SqlColumnValue<*>>()
}


class ComparisonOp<T: Any?>(private val expr1: Column<T>, private val expr2: T, private val op: String) : SQLExpression() {
    override fun toSQLBuilder(): String = buildString {
        append(expr1.columnName)
        append(" $op ?")
    }

    override fun bindArgs(): List<SqlColumnValue<*>> {
        return listOf(
            SqlColumnValue(expr1.columnType, expr2)
        )
    }
}

/**
 * Represents an SQL operator that checks if [expr1] is equals to [expr2].
 */
class EqOp<T: Any?>(private val expr1: Column<T>, private val expr2: T) : SQLExpression() {
    override fun toSQLBuilder(): String = buildString {
        append(expr1.columnName)
        append(" = ?")
    }

    override fun bindArgs(): List<SqlColumnValue<*>> {
        return listOf(
            SqlColumnValue(expr1.columnType, expr2)
        )
    }
}

/**
 * Represents a logical operator that performs an `and` operation between all the specified [expressions].
 */
class AndOp(private val expressions: List<SQLExpression>) : SQLExpression() {
    override fun toSQLBuilder(): String = expressions.joinToString(" AND ") {
        "(${it.toSQLBuilder()})"
    }

    override fun bindArgs(): List<SqlColumnValue<*>> {
        return expressions.flatMap { it.bindArgs() }
    }
}

class OrOp(private val expressions: List<SQLExpression>) : SQLExpression() {
    override fun toSQLBuilder(): String = expressions.joinToString(" OR ") {
        "(${it.toSQLBuilder()})"
    }

    override fun bindArgs(): List<SqlColumnValue<*>> {
        return expressions.flatMap { it.bindArgs() }
    }
}


class InListOrNotInListOp<T>(
    /** Returns the expression compared to each element of the list. */
    val expr: Column<T>,
    /** Returns the query to check against. */
    val list: Iterable<T>,
    /** Returns `true` if the check is inverted, `false` otherwise. */
    val isInList: Boolean
) : SQLExpression() {
    override fun toSQLBuilder(): String = buildString {
        append(expr.columnName)
        when (isInList) {
            true -> append(" IN (")
            else -> append(" NOT IN (")
        }
        append(list.joinToString(",") { "?" })
        append(")")
    }

    override fun bindArgs(): List<SqlColumnValue<*>> {
        return list.map {
            SqlColumnValue(expr.columnType, it)
        }
    }
}

class LikeOp<T>(
    val expr: Column<T>,
    val pattern: String,
    val isLike: Boolean
) : SQLExpression() {
    override fun toSQLBuilder(): String = buildString {
        append(expr.columnName)
        when (isLike) {
            true -> append(" LIKE ?")
            else -> append(" NOT LIKE ?")
        }
    }

    override fun bindArgs(): List<SqlColumnValue<*>> {
        return listOf(
            SqlColumnValue(
                columnType = if (expr.columnType is CustomColumnType<*, *>) expr.columnType.columnType else expr.columnType,
                value = pattern
            )
        )
    }
}
