package cn.imkarl.sqldsl.database.h2

import cn.imkarl.sqldsl.column.*
import cn.imkarl.sqldsl.database.SqlColumnValue
import java.math.BigDecimal
import java.sql.*
import java.time.LocalDate
import java.time.LocalDateTime

/**
 * 转为List
 */
internal fun ResultSet.toList(): List<Map<String, Any?>> {
    val rows = mutableListOf<Map<String, Any?>>()
    val metaData = this.metaData
    val columnCount = metaData.columnCount
    while (this.next()) {
        val row = mutableMapOf<String, Any?>()
        for (i in 1..columnCount) {
            row[metaData.getColumnName(i)] = this.getObject(i)
        }
        rows.add(row)
    }
    return rows
}

/**
 * 绑定参数
 */
internal fun PreparedStatement.bindArg(index: Int, value: SqlColumnValue<*>) {
    val parameterIndex = index + 1
    val columnValue = value.value

    if (columnValue == null) {
        this.setNull(parameterIndex, value.columnType.sqlType, null)
        this.setObject(parameterIndex, null, value.columnType.sqlType)
        return
    }

    val columnType = value.columnType
    when (columnType) {
        is BooleanColumnType -> this.setBoolean(parameterIndex, columnType.valueToDB(columnValue as Boolean))
        is ByteColumnType -> this.setByte(parameterIndex, columnType.valueToDB(columnValue as Byte))
        is CharColumnType -> this.setString(parameterIndex, columnType.valueToDB(columnValue as String))
        is VarCharColumnType -> this.setString(parameterIndex, columnType.valueToDB(columnValue as String))
        is TextColumnType -> this.setString(parameterIndex, columnType.valueToDB(columnValue as String))
        is CharacterColumnType -> this.setString(parameterIndex, columnType.valueToDB(columnValue as Char))
        is DateColumnType -> this.setDate(parameterIndex, columnType.valueToDB(columnValue as LocalDate))
        is DateTimeColumnType -> this.setTimestamp(parameterIndex, columnType.valueToDB(columnValue as LocalDateTime))
        is ShortColumnType -> this.setShort(parameterIndex, columnType.valueToDB(columnValue as Short))
        is EnumColumnType<*> -> this.setInt(parameterIndex, (columnType as ColumnType<Any>).valueToDB(columnValue) as Int)
        is IntegerColumnType -> this.setInt(parameterIndex, columnType.valueToDB(columnValue as Int))
        is LongColumnType -> this.setLong(parameterIndex, columnType.valueToDB(columnValue as Long))
        is FloatColumnType -> this.setFloat(parameterIndex, columnType.valueToDB(columnValue as Float))
        is DoubleColumnType -> this.setDouble(parameterIndex, columnType.valueToDB(columnValue as Double))
        is DecimalColumnType -> this.setBigDecimal(parameterIndex, columnType.valueToDB(columnValue as BigDecimal))
        is BlobColumnType -> this.setBlob(parameterIndex, columnType.valueToDB(columnValue as ByteArray))
        is CustomColumnType<*, *> -> {
            bindArg(parameterIndex - 1, SqlColumnValue(columnType.columnType, (columnType.valueToDB as ((Any) -> Any)).invoke(columnValue)))
        }
        else -> throw IllegalArgumentException("bindArg value is unknow type: `${value::class}`")
    }
}

internal val ColumnType<*>.sqlType: Int
    get() {
        val columnType = if (this is CustomColumnType<*, *>) this.columnType else this
        return when (columnType) {
            is ByteColumnType -> Types.TINYINT
            is ShortColumnType -> Types.SMALLINT
            is IntegerColumnType -> Types.INTEGER
            is LongColumnType -> Types.BIGINT
            is FloatColumnType -> Types.FLOAT
            is DoubleColumnType -> Types.DOUBLE
            is DecimalColumnType -> Types.DECIMAL
            is CharacterColumnType -> Types.CHAR
            is CharColumnType -> Types.CHAR
            is VarCharColumnType -> Types.VARCHAR
            is TextColumnType -> Types.VARCHAR
            is BlobColumnType -> Types.BLOB
            is BooleanColumnType -> Types.BOOLEAN
            is EnumColumnType<*> -> Types.INTEGER
            is DateColumnType -> Types.DATE
            is DateTimeColumnType -> Types.TIME
            else -> Types.NULL
        }
    }
