package cn.imkarl.sqldsl.database.h2

import cn.imkarl.core.common.log.LogLevel
import cn.imkarl.core.common.log.LogUtils
import cn.imkarl.sqldsl.column.PrimaryKey
import cn.imkarl.sqldsl.database.Database
import cn.imkarl.sqldsl.database.DatabaseMetaData
import cn.imkarl.sqldsl.database.SqlColumnValue
import cn.imkarl.sqldsl.sql.SQLExpression
import cn.imkarl.sqldsl.sql.SQLWhere
import cn.imkarl.sqldsl.sql.eq
import cn.imkarl.sqldsl.table.SchemaTable
import cn.imkarl.sqldsl.table.Table
import com.alibaba.druid.filter.logging.Log4j2Filter
import com.alibaba.druid.pool.DruidDataSource
import cn.imkarl.sqldsl.sql.SQLAlterTable
import java.sql.SQLException
import java.util.*

/**
 * H2Database
 */
class H2Database(
    private val config: Properties,
    private val version: Int,
    onCreate: (database: Database) -> Unit,
    onUPgrade: ((database: Database, oldVersion: Int, newVersion: Int) -> Unit)? = null
) : Database() {

    companion object {
        internal const val defaultVersion = 0
    }

    private val dataSource by lazy {
        val dataSource = DruidDataSource()
        dataSource.driverClassName = config.getProperty("spring.datasource.driverClassName")
        dataSource.url = config.getProperty("spring.datasource.url")
        dataSource.username = config.getProperty("spring.datasource.username")
        dataSource.password = config.getProperty("spring.datasource.password")
        dataSource.isRemoveAbandoned = config.getProperty("druid.removeAbandoned")?.toLowerCase() == "true"
        dataSource.removeAbandonedTimeout = config.getProperty("druid.removeAbandonedTimeout")?.toInt() ?: dataSource.removeAbandonedTimeout
        dataSource.proxyFilters = listOf(
            object : Log4j2Filter() {
                override fun statementLog(message: String?) {
                    message ?: return

                    val feature = "} executed. "
                    val offset = message.indexOf(feature)
                    val logMessage = message.substring(offset + feature.length)
                    LogUtils.println(LogLevel.VERBOSE, logMessage)
                }
            }.apply {
                // 打开关闭 log
                isConnectionLogEnabled = false
                isConnectionCloseAfterLogEnabled = false
                isConnectionConnectBeforeLogEnabled = false
                isConnectionConnectAfterLogEnabled = false
                isConnectionCommitAfterLogEnabled = false
                isResultSetOpenAfterLogEnabled = false
                isResultSetCloseAfterLogEnabled = false
                isResultSetLogEnabled = false
                isResultSetNextAfterLogEnabled = false
                isStatementCreateAfterLogEnabled = false
                isStatementCloseAfterLogEnabled = false
                // 参数 log
                isStatementParameterClearLogEnable = false
                isStatementParameterSetLogEnabled = false
                isStatementPrepareAfterLogEnabled = false
                isStatementPrepareCallAfterLogEnabled = false

                //  执行之后的log
                isStatementExecuteAfterLogEnabled = false
                isStatementExecuteBatchAfterLogEnabled = false
                isStatementExecuteQueryAfterLogEnabled = false
                isStatementExecuteUpdateAfterLogEnabled = false

                // 可以打印完整的SQL
                isStatementExecutableSqlLogEnable = true
                isStatementLogEnabled = true
            }
        )
        dataSource.init()
        dataSource
    }

    init {
        // 获取当前版本号
        val currentVersion = if (tableExists(SchemaTable)) {
            selectAll(SchemaTable).firstOrNull()?.get(SchemaTable.version) ?: defaultVersion
        } else {
            defaultVersion
        }

        if (currentVersion == defaultVersion) {
            // 首次创建数据库
            this.createTable(SchemaTable)
            onCreate.invoke(this)
        } else {
            if (currentVersion > version) {
                // 降级版本号
                throw IllegalArgumentException("目标版本号(version: $version) 低于 当前版本号(version: $currentVersion)")
            }
            if (currentVersion != version) {
                // 升级版本号
                onUPgrade?.invoke(this, currentVersion, version)
            }
        }

        // 记录新的版本号
        this.insert(SchemaTable.newRow {
            it[SchemaTable.version] = version
        })
    }


    private var _metadata: DatabaseMetaData? = null
    override val metadata: DatabaseMetaData
        get() {
            if (_metadata == null) {
                _metadata = H2DatabaseMetaData(dataSource.connection.use { it.metaData })
            }
            return _metadata!!
        }


    override fun tableExists(table: Table): Boolean {
        return metadata.tableNames.find { it.equals(table.tableName, true) } != null
    }

    override fun executeUpdate(sql: String, bindArgs: Array<SqlColumnValue<*>>?): Int {
        if (sql.isEmpty()) {
            throw SQLException("sql is empty")
        }

        val execteSqlStartTime = System.currentTimeMillis()
        return dataSource.connection.use { connection ->
            return@use if (!bindArgs.isNullOrEmpty()) {
                connection.prepareStatement(sql).also { statement ->
                    bindArgs.forEachIndexed { index, value ->
                        statement.bindArg(index, value)
                    }
                }
            } else {
                connection.prepareStatement(sql)
            }
                .executeUpdate()
                .also {
                    val duration = System.currentTimeMillis() - execteSqlStartTime
                    LogUtils.println(LogLevel.INFO, "executeUpdate sql: $sql (${duration}ms)")
                    // 清空metadata，更新数据库结构
                    _metadata = null
                }
        }
    }

    override fun executeQuery(sql: String, selectionArgs: Array<SqlColumnValue<*>>?): List<Map<String, Any?>> {
        if (sql.isEmpty()) {
            throw SQLException("sql is empty")
        }

        val execteSqlStartTime = System.currentTimeMillis()
        return dataSource.connection.use { connection ->
            return@use if (selectionArgs != null && selectionArgs.isNotEmpty()) {
                connection.prepareStatement(sql).also { statement ->
                    selectionArgs.forEachIndexed { index, value ->
                        statement.bindArg(index, value)
                    }
                }.executeQuery()
            } else {
                connection.prepareStatement(sql).executeQuery()
            }.use { rs ->
                val duration = System.currentTimeMillis() - execteSqlStartTime
                LogUtils.println(LogLevel.INFO, "executeQuery sql: $sql (${duration}ms)")
                rs.toList()
            }
        }
    }

    override fun count(table: Table, where: SQLExpression?): Long {
        createTable(table)

        if (table.columns.isEmpty()) {
            throw SQLException("columns is empty")
        }

        val sql = buildString {
            append("SELECT COUNT(*) FROM ")
            append(table.tableName)

            where?.let {
                val whereSql = it.toSQLBuilder()
                if (whereSql.trim().isNotBlank()) {
                    append(" WHERE ")
                    append(whereSql)
                }
            }
        }

        return this.executeQuery(sql, where?.bindArgs())
            .firstOrNull()
            ?.values
            ?.firstOrNull() as Long? ?: -1L
    }


    @Throws(SQLException::class)
    override fun createTable(vararg tables: Table) {
        tables.forEach { table ->
            if (!tableExists(table)) {
                if (table.columns.isEmpty()) {
                    throw SQLException("columns is empty")
                }

                val sql = buildString {
                    append("CREATE TABLE IF NOT EXISTS ")
                    append(table.tableName)
                    table.columns.joinTo(buffer = this, prefix = " (", postfix = ")") { column ->
                        buildString {
                            append(column.columnName)
                            append(" ")
                            append(column.columnType.sqlType())

                            val isPKColumn = column is PrimaryKey
                            if (isPKColumn) {
                                append(" PRIMARY KEY")

                                val isAutoIncrement = (column as? PrimaryKey<*>)?.isAutoIncrement ?: false
                                if (isAutoIncrement) {
                                    append(" AUTO_INCREMENT")
                                }
                            } else {
                                if (column.columnType.nullable) {
                                    append(" NULL")
                                } else {
                                    append(" NOT NULL")
                                }
                            }
                        }
                    }
                }
                this.executeUpdate(sql)
            }
        }
    }

    @Throws(SQLException::class)
    override fun dropTable(vararg tables: Table) {
        tables.forEach { table ->
            if (tableExists(table)) {
                val sql = "DROP TABLE ${table.tableName}"
                this.executeUpdate(sql)
            }
        }
    }


    @Throws(SQLException::class)
    override fun insert(row: Table.Row): Int {
        if (row.values.isEmpty()) {
            throw SQLException("columns is empty")
        }

        val values = row.values.filter { !(it.key is PrimaryKey && (it.key as PrimaryKey).isAutoIncrement) }.toMutableMap()
        row.table.columns
            .filterNot { values.keys.contains(it) }
            .filter { column -> column !is PrimaryKey }
            .forEach { column ->
                values[column] = column.defaultValueFun.invoke()
            }

        val sql = buildString {
            append("INSERT INTO ")
            append(row.table.tableName)
            values.keys.joinTo(buffer = this, prefix = " (", postfix = ")") { it.columnName }

            append(" VALUES ")
            values.keys.joinTo(buffer = this, prefix = " (", postfix = ")") { "?" }
        }

        val conut = this.executeUpdate(
            sql = sql,
            bindArgs = values.map { SqlColumnValue(it.key.columnType, row[it.key]) }
        )
        return conut
    }


    @Throws(SQLException::class)
    override fun <T: Any> insertAndGetId(row: Table.Row): T {
        insert(row)

        val result = executeQuery("SELECT TOP 1 ID FROM ${row.table.tableName} ORDER BY ID DESC")
        val insertId = result.firstOrNull()?.values?.firstOrNull() as? T ?: throw SQLException()
        return insertId
    }

    @Throws(SQLException::class)
    override fun update(row: Table.Row, where: SQLExpression?): Int {
        val values = row.values.filter { !(it.key is PrimaryKey && (it.key as PrimaryKey).isAutoIncrement) }

        if (values.isEmpty()) {
            throw SQLException("columns is empty")
        }

        val sql = buildString {
            append("UPDATE ")
            append(row.table.tableName)

            values.entries.joinTo(this, prefix = " SET ") { (col, value) ->
                "${col.columnName}=?"
            }

            where?.let {
                val whereSql = it.toSQLBuilder()
                if (whereSql.trim().isNotBlank()) {
                    append(" WHERE ")
                    append(whereSql)
                }
            }
        }

        return this.executeUpdate(
            sql = sql,
            bindArgs = values.map { SqlColumnValue(it.key.columnType, row[it.key]) } + (where?.bindArgs() ?: emptyList())
        )
    }

    @Throws(SQLException::class)
    override fun deleteAll(table: Table): Int {
        return deleteWhere(table)
    }

    @Throws(SQLException::class)
    override fun deleteById(row: Table.Row): Int {
        val idColumn = row.table.primaryKey ?: throw SQLException("table `${row.table::class.java.name}` not primaryKey")
        val value = row[idColumn]
        return deleteWhere(row.table, idColumn.eq(value))
    }

    @Throws(SQLException::class)
    override fun deleteWhere(table: Table, where: SQLExpression?): Int {
        createTable(table)

        if (table.columns.isEmpty()) {
            throw SQLException("columns is empty")
        }

        val sql = buildString {
            append("DELETE FROM ")
            append(table.tableName)

            where?.let {
                val whereSql = it.toSQLBuilder()
                if (whereSql.trim().isNotBlank()) {
                    append(" WHERE ")
                    append(whereSql)
                }
            }
        }

        return this.executeUpdate(sql, where?.bindArgs())
    }

    @Throws(SQLException::class)
    override fun selectAll(table: Table): List<Table.Row> {
        return select(table)
    }

    @Throws(SQLException::class)
    override fun select(table: Table, where: SQLExpression?): List<Table.Row> {
        if (table.columns.isEmpty()) {
            throw SQLException("columns is empty")
        }

        val sql = buildString {
            append("SELECT * FROM ")
            append(table.tableName)

            where?.let {
                val whereSql = it.toSQLBuilder()
                if (whereSql.trim().isNotBlank()) {
                    append(" WHERE ")
                    append(whereSql)
                }
                if (where is SQLWhere) {
                    if (where.sorts.isNotEmpty()) {
                        append(
                            where.sorts.joinToString(prefix = " ORDER BY ", separator = ", ") { sort ->
                                "${sort.column.columnName} ${sort.order.name.toUpperCase()}"
                            }
                        )
                    }
                    if (where.limit > 0) {
                        append(" LIMIT ${where.offset}, ${where.limit}")
                    }
                }
            }
        }

        val bindArgs = where?.bindArgs()?.toTypedArray()
        return this.executeQuery(sql, bindArgs)
            .map { map ->
                table.newRow { row ->
                    map.forEach { columnName, columnValue ->
                        table.columns.find { it.columnName.equals(columnName, true) }?.let { column ->
                            row.set(column, columnValue?.let { column.columnType.valueFromDB(it) })
                        }
                    }
                }
            }
    }

    fun alter(table: Table, vararg sql: SQLAlterTable) {
        sql.forEach { item ->
            this.executeUpdate("ALTER TABLE ${table.tableName} ${item.toSQLBuilder()}")
        }
    }

}