package cn.imkarl.sqldsl.column

import java.io.InputStream
import java.math.BigDecimal
import java.math.RoundingMode
import java.sql.Blob
import java.sql.Clob
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.ZoneId
import javax.sql.rowset.serial.SerialBlob
import kotlin.reflect.KClass

abstract class ColumnType<T: Any?>(var nullable: Boolean = false) {

    abstract val defaultValue: T

    abstract fun sqlType(): String

    /**
     * Converts the specified [value] (from the database) to an object of the appropriated type, for this column type.
     * Default implementation returns the same instance.
     */
    abstract fun valueFromDB(value: Any): T

    /** Returns an object compatible with the database, from the specified [value], for this column type. */
    abstract fun valueToDB(value: T): Any?

    override fun toString(): String = sqlType()

}




// Numeric columns

/**
 * Numeric column for storing 1-byte integers.
 */
class ByteColumnType : ColumnType<Byte>() {

    override val defaultValue: Byte = -1

    override fun sqlType(): String = "TINYINT"

    override fun valueFromDB(value: Any): Byte {
        return when (value) {
            is Byte -> value
            is Number -> value.toByte()
            is String -> value.toByte()
            else -> error("Unexpected value of type Byte: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Byte): Byte {
        return valueFromDB(value)
    }

}

/**
 * Numeric column for storing 2-byte integers.
 */
class ShortColumnType : ColumnType<Short>() {

    override val defaultValue: Short = -1

    override fun sqlType(): String = "SMALLINT"

    override fun valueFromDB(value: Any): Short {
        return when (value) {
            is Short -> value
            is Number -> value.toShort()
            is String -> value.toShort()
            else -> error("Unexpected value of type Short: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Short): Short {
        return valueFromDB(value)
    }

}

/**
 * Numeric column for storing 4-byte integers.
 */
class IntegerColumnType : ColumnType<Int>() {

    override val defaultValue: Int = -1

    override fun sqlType(): String = "INT"

    override fun valueFromDB(value: Any): Int {
        return when (value) {
            is Int -> value
            is Number -> value.toInt()
            is String -> value.toInt()
            else -> error("Unexpected value of type Int: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Int): Int {
        return valueFromDB(value)
    }

}

/**
 * Numeric column for storing 8-byte integers.
 */
class LongColumnType : ColumnType<Long>() {

    override val defaultValue: Long = -1

    override fun sqlType(): String = "BIGINT"

    override fun valueFromDB(value: Any): Long {
        return when (value) {
            is Long -> value
            is Number -> value.toLong()
            is String -> value.toLong()
            else -> error("Unexpected value of type Long: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Long): Long {
        return valueFromDB(value)
    }

}

/**
 * Numeric column for storing 4-byte (single precision) floating-point numbers.
 */
class FloatColumnType : ColumnType<Float>() {

    override val defaultValue: Float = -1.0F

    override fun sqlType(): String = "FLOAT"

    override fun valueFromDB(value: Any): Float {
        return when (value) {
            is Float -> value
            is Number -> value.toFloat()
            is String -> value.toFloat()
            else -> error("Unexpected value of type Float: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Float): Float {
        return valueFromDB(value)
    }

}

/**
 * Numeric column for storing 8-byte (double precision) floating-point numbers.
 */
class DoubleColumnType : ColumnType<Double>() {

    override val defaultValue: Double = -1.0

    override fun sqlType(): String = "DOUBLE"

    override fun valueFromDB(value: Any): Double {
        return when (value) {
            is Double -> value
            is Number -> value.toDouble()
            is String -> value.toDouble()
            else -> error("Unexpected value of type Double: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Double): Double {
        return valueFromDB(value)
    }

}

/**
 * Numeric column for storing numbers with the specified [precision] and [scale].
 */
class DecimalColumnType(
    /** Total count of significant digits in the whole number. */
    val precision: Int,
    /** Count of decimal digits in the fractional part. */
    val scale: Int
) : ColumnType<BigDecimal>() {

    override val defaultValue: BigDecimal = BigDecimal(-1)

    override fun sqlType(): String = "DECIMAL($precision, $scale)"

    override fun valueFromDB(value: Any): BigDecimal {
        return when (value) {
            is BigDecimal -> value
            is Double -> value.toBigDecimal()
            is Float -> value.toBigDecimal()
            is Long -> value.toBigDecimal()
            is Int -> value.toBigDecimal()
            is String -> BigDecimal(value)
            else -> error("Unexpected value of type Double: $value of ${value::class.qualifiedName}")
        }.setScale(scale, RoundingMode.HALF_EVEN)
    }

    override fun valueToDB(value: BigDecimal): BigDecimal {
        return valueFromDB(value)
    }

}

// Character columns

/**
 * Character column for storing single characters.
 */
class CharacterColumnType : ColumnType<Char>() {

    override val defaultValue: Char = ' '

    override fun sqlType(): String = "CHAR"

    override fun valueFromDB(value: Any): Char {
        return when (value) {
            is Char -> value
            is Number -> value.toChar()
            is String -> value.single()
            else -> error("Unexpected value of type Char: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: Char): String = value.toString()

}

/**
 * Base character column for storing strings using the specified text [collate] type.
 */
abstract class StringColumnType : ColumnType<String>() {

    override val defaultValue: String = ""

    override fun valueFromDB(value: Any): String = when (value) {
        is Clob -> value.characterStream.readText()
        is ByteArray -> String(value)
        else -> value.toString()
    }

    override fun valueToDB(value: String): String = value

}

/**
 * Character column for storing strings with the exact [colLength] length using the specified [collate] type.
 */
open class CharColumnType(
    /** Returns the maximum length of this column. */
    val maxLength: Int = 255
) : StringColumnType() {
    override fun sqlType(): String = "CHAR($maxLength)"

    override fun valueToDB(value: String): String {
        require(value.codePointCount(0, value.length) <= maxLength) {
            "Value '$value' can't be stored to database column because exceeds length ($maxLength)"
        }
        return value
    }

}

/**
 * Character column for storing strings with the specified maximum [colLength] using the specified [collate] type.
 */
open class VarCharColumnType(
    /** Returns the maximum length of this column. */
    open val maxLength: Int = 255
) : StringColumnType() {
    override fun sqlType(): String = "VARCHAR($maxLength)"

    override fun valueToDB(value: String): String {
        require(value.codePointCount(0, value.length) <= maxLength) {
            "Value '$value' can't be stored to database column because exceeds length ($maxLength)"
        }
        return value
    }

}

/**
 * Character column for storing strings of arbitrary length using the specified [collate] type.
 * [eagerLoading] means what content will be loaded immediately when data loaded from database.
 */
open class TextColumnType : StringColumnType() {
    override fun sqlType(): String = "TEXT"
}

open class CustomColumnType<T: Any?, DBTYPE: Any>(
    val columnType: ColumnType<DBTYPE>,
    val valueFromDB: (value: DBTYPE) -> T,
    val valueToDB: (value: T) -> DBTYPE,
    _defaultValue: T
) : ColumnType<T>(columnType.nullable) {

    override val defaultValue: T = _defaultValue

    override fun sqlType(): String = columnType.sqlType()

    override fun valueFromDB(value: Any): T {
        val columnValue = columnType.valueFromDB(value)
        return valueFromDB.invoke(columnValue)
    }

    override fun valueToDB(value: T): Any? {
        val columnValue = valueToDB.invoke(value)
        return columnType.valueToDB(columnValue)
    }
}

/**
 * Binary column for storing BLOBs.
 */
class BlobColumnType : ColumnType<ByteArray>() {

    override val defaultValue: ByteArray = ByteArray(0)

    override fun sqlType(): String = "BLOB"

    override fun valueFromDB(value: Any): ByteArray {
        return when (value) {
            is ByteArray -> value
            is Blob -> value.binaryStream.use { it.readBytes() }
            is InputStream -> value.use { it.readBytes() }
            else -> error("Unexpected value of type Blob: $value of ${value::class.qualifiedName}")
        }
    }

    override fun valueToDB(value: ByteArray): Blob {
        return SerialBlob(value)
    }

}

// Boolean columns

/**
 * Boolean column for storing boolean values.
 */
class BooleanColumnType : ColumnType<Boolean>() {

    override val defaultValue: Boolean = false

    override fun sqlType(): String = "BOOLEAN"

    override fun valueFromDB(value: Any): Boolean {
        return when (value) {
            is Boolean -> value
            is Number -> value.toLong() != 0L
            is String -> try {
                value.toDouble().toInt() == 1
            } catch (e: Exception) {
                value.toBoolean()
            }
            else -> value.toString().toBoolean()
        }
    }

    override fun valueToDB(value: Boolean): Boolean {
        return valueFromDB(value)
    }

}

// Enumeration columns

/**
 * Enumeration column for storing enums of type [klass] by their ordinal.
 */
class EnumColumnType<T : Enum<T>>(
    /** Returns the enum class used in this column type. */
    val klass: KClass<T>,
    _defaultValue: T
) : ColumnType<T>() {

    override val defaultValue: T = _defaultValue

    override fun sqlType(): String = "INT"

    override fun valueFromDB(value: Any): T {
        return when (value) {
            is Number -> klass.java.enumConstants[value.toInt()]
            is Enum<*> -> value as T
            else -> error("$value of ${value::class.qualifiedName} is not valid for enum ${klass.simpleName}")
        }
    }

    override fun valueToDB(value: T): Int {
        return value.ordinal
    }

}



// Date Time columns

class DateColumnType : ColumnType<LocalDate>() {

    override val defaultValue: LocalDate = LocalDate.MIN

    override fun sqlType(): String = "DATE"

    override fun valueFromDB(value: Any): LocalDate {
        return when (value) {
            is LocalDate -> value
            is java.sql.Date -> longToLocalDate(value.time)
            is java.sql.Timestamp -> longToLocalDate(value.time)
            is Int -> longToLocalDate(value.toLong())
            is Long -> longToLocalDate(value)
            is String -> LocalDate.parse(value)
            else -> LocalDate.parse(value.toString())
        }
    }

    override fun valueToDB(value: LocalDate): java.sql.Date {
        return java.sql.Date(value.year, value.monthValue, value.dayOfMonth)
    }

    private fun longToLocalDate(instant: Long) = Instant.ofEpochMilli(instant).atZone(ZoneId.systemDefault()).toLocalDate()

}

open class DateTimeColumnType : ColumnType<LocalDateTime>() {

    override val defaultValue: LocalDateTime = LocalDateTime.MIN

    override fun sqlType(): String = "DATETIME"

    override fun valueFromDB(value: Any): LocalDateTime {
        return when (value) {
            is LocalDateTime -> value
            is java.sql.Date -> longToLocalDateTime(value.time)
            is java.sql.Timestamp -> longToLocalDateTime(value.time)
            is Int -> longToLocalDateTime(value.toLong())
            is Long -> longToLocalDateTime(value)
            is String -> LocalDateTime.parse(value)
            else -> LocalDateTime.parse(value.toString())
        }
    }

    override fun valueToDB(value: LocalDateTime): java.sql.Timestamp {
        return java.sql.Timestamp(value.year, value.monthValue, value.dayOfMonth, value.hour, value.minute, value.second, value.nano)
    }

    private fun longToLocalDateTime(millis: Long) = LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.systemDefault())

}
