package cn.imkarl.core.common.log

import cn.imkarl.core.common.app.AppUtils
import cn.imkarl.core.common.platform.Platform
import java.io.PrintWriter
import java.io.StringWriter
import java.net.UnknownHostException

/**
 * 日志相关工具类
 * @author imkarl
 */
object LogUtils {

    private var _maxOnceLength = 3000
    private var _globalTag = AppUtils.appName


    /**
     * 设置单次最大打印长度
     */
    @JvmStatic
    fun setMaxOnceLength(length: Int) {
        this._maxOnceLength = length
    }

    /**
     * 设置全局TAG
     */
    @JvmStatic
    fun setGlobalTag(tag: String) {
        this._globalTag = tag
    }



    @JvmStatic
    fun v(message: Any?) {
        log(LogLevel.VERBOSE, _globalTag, message)
    }
    @JvmStatic
    fun v(tag: String, message: Any?) {
        log(LogLevel.VERBOSE, tag, message)
    }

    @JvmStatic
    fun d(message: Any?) {
        log(LogLevel.DEBUG, _globalTag, message)
    }
    @JvmStatic
    fun d(tag: String, message: Any?) {
        log(LogLevel.DEBUG, tag, message)
    }

    @JvmStatic
    fun i(message: Any?) {
        log(LogLevel.INFO, _globalTag, message)
    }
    @JvmStatic
    fun i(tag: String, message: Any?) {
        log(LogLevel.INFO, tag, message)
    }

    @JvmStatic
    fun w(message: Any?) {
        log(LogLevel.WARN, _globalTag, message)
    }
    @JvmStatic
    fun w(tag: String, message: Any?) {
        log(LogLevel.WARN, tag, message)
    }

    @JvmStatic
    fun e(message: Any?) {
        log(LogLevel.ERROR, _globalTag, message)
    }
    @JvmStatic
    fun e(tag: String, message: Any?) {
        log(LogLevel.ERROR, tag, message)
    }


    @JvmStatic
    fun println(level: LogLevel, msg: String?) {
        if (!AppUtils.isDebug) {
            return
        }

        console(level, _globalTag, toString(msg))
    }
    @JvmStatic
    fun println(level: LogLevel, tag: String, msg: String?) {
        if (!AppUtils.isDebug) {
            return
        }

        console(level, tag, toString(msg))
    }


    private fun log(level: LogLevel, tag: String, message: Any?) {
        if (!AppUtils.isDebug) {
            return
        }

        val element = Throwable().stackTrace[2]
        var className = element.className
        className = className.substring(className.lastIndexOf(".") + 1)
        val codeLine = className + '.' + element.methodName + '(' + element.fileName + ':' + element.lineNumber + ')'
        console(level, tag, codeLine + "\n\t" + toString(message))
    }

    /**
     * 真实的打印方法
     */
    private fun console(level: LogLevel, tag: String, msg: String) {
        if (msg.length <= _maxOnceLength) {
            Platform.instance.console(level, tag, msg)
            return
        }

        // 超出目标长度，自动换行打印
        var startIndex = 0
        var endIndex = _maxOnceLength
        while (endIndex > startIndex) {
            Platform.instance.console(
                level = level,
                tag = tag,
                message = (if (startIndex == 0) "" else "\t↑↑↑↑\n") + msg.substring(startIndex, endIndex)
            )

            startIndex = endIndex
            endIndex = Math.min(msg.length, startIndex + _maxOnceLength)
        }
    }

    private fun toString(message: Any?): String {
        if (message == null) {
            return "[NULL]"
        }
        if (message is Throwable) {
            var t = message as Throwable?
            while (t != null) {
                if (t is UnknownHostException) {
                    return t.toString()
                }
                t = t.cause
            }

            val sw = StringWriter()
            val pw = PrintWriter(sw, false)
            message.printStackTrace(pw)
            pw.flush()
            return sw.toString()
        }
        return message.toString()
    }

}

enum class LogLevel {
    VERBOSE,
    INFO,
    DEBUG,
    WARN,
    ERROR
}
