package cn.iisme.starter.redis.common.utils;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.TimeoutUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import cn.iisme.starter.redis.common.IRedisKey;

/**
 * Redis缓存工具类
 *
 * @作者: yangguangjing
 * @创建时间: 2019年10月19日
 */
@Component
public class RedisJsonUtils {

	@Resource(name = "redisTemplateForJson")
	private RedisTemplate<String, Object> redisTemplate;
	
	private static RedisJsonUtils redisJsonUtils;


	@PostConstruct
	public void init() {
		redisJsonUtils = this;
		redisJsonUtils.redisTemplate = this.redisTemplate;
	}

	/** 一周 */
	private static final long SEVEN_DAYS = TimeoutUtils.toSeconds(7L, TimeUnit.DAYS);

	/**
	 * 设置缓存有效期
	 */
	public static void setExpireTime(String key, long seconds) {
		if (seconds < 1L)
			seconds = SEVEN_DAYS;
		redisJsonUtils.redisTemplate.expire(key, seconds, TimeUnit.SECONDS);
	}

	/**
	 * 设置缓存有效期
	 */
	public static void setExpireDate(String key, Date date) {
		if (null == date) {
			// 日期不传,默认7天有效
			Calendar calendar = Calendar.getInstance();
			calendar.setTime(date);
			calendar.add(Calendar.DAY_OF_MONTH, +7);
			date = calendar.getTime();
		}
		redisJsonUtils.redisTemplate.expireAt(key, date);
	}

	/**
	 * 判断是否存在key
	 */
	public static Boolean exists(String key) {
		return redisJsonUtils.redisTemplate.hasKey(key);
	}

	/**
	 * 删除key
	 */
	public static void remove(String key) {
		redisJsonUtils.redisTemplate.delete(key);
	}

	/**
	 * 批量删除keys
	 */
	public static void remove(List<String> keys) {
		redisJsonUtils.redisTemplate.delete(keys);
	}

	/**
	 * put缓存，永久有效
	 */
	public static void set(String key, Object value) {
		redisJsonUtils.redisTemplate.boundValueOps(key).set(value);
	}

	/**
	 * put缓存，并设置有效期
	 */
	public static void set(String key, Object value, long seconds) {
		Assert.hasText(key, "key is not null");
		redisJsonUtils.redisTemplate.boundValueOps(key).set(value);
		setExpireTime(key, seconds);
	}
	
	public static void set(String key, Object value, Date date) {
		Assert.hasText(key, "key is not null");
		redisJsonUtils.redisTemplate.boundValueOps(key).set(value);
		setExpireDate(key, date);
	}

	public static Object get(String key) {
		return redisJsonUtils.redisTemplate.boundValueOps(key).get();
	}

	private static Long leftPush(String key, Object value) {
		Assert.hasText(key, "key is not null");
		return redisJsonUtils.redisTemplate.opsForList().leftPush(key, value);
	}

	public static Long leftPush(String key, Object value, long seconds) {
		Long resultLong = leftPush(key, value);
		setExpireTime(key, seconds);
		return resultLong;
	}

	private static Long rightPush(String key, Object value) {
		Assert.hasText(key, "key is not null");
		return redisJsonUtils.redisTemplate.opsForList().rightPush(key, value);
	}

	public static Long rightPush(String key, Object value, long seconds) {
		Long resultLong = rightPush(key, value);
		setExpireTime(key, seconds);
		return resultLong;
	}

	public static Object leftPop(String key) {
		return redisJsonUtils.redisTemplate.opsForList().leftPop(key);
	}

	public static Object rightPop(String key) {
		return redisJsonUtils.redisTemplate.opsForList().rightPop(key);
	}

	public static Long listLength(String key) {
		return redisJsonUtils.redisTemplate.opsForList().size(key);
	}

	public static List<Object> listRange(String key, int start, int end) {
		return redisJsonUtils.redisTemplate.opsForList().range(key, start, end);
	}

	public static void listRemove(String key, long i, Object value) {
		redisJsonUtils.redisTemplate.opsForList().remove(key, i, value);
	}

	public static String index(String key, long index) {
		return (String) redisJsonUtils.redisTemplate.opsForList().index(key, index);
	}

	public static void listSet(String key, long index, Object value) {
		redisJsonUtils.redisTemplate.opsForList().set(key, index, value);
	}

	public static void listTrim(String key, long start, int end) {
		redisJsonUtils.redisTemplate.opsForList().trim(key, start, end);
	}

	/**
	 * set方式添加值
	 */
	public static void setAdd(String key, Object... values) {
		Assert.hasText(key, "key is not null");
		redisJsonUtils.redisTemplate.opsForSet().add(key, values);
	}

	/**
	 * set方式添加值，带有效期
	 */
	public static void setAdd(String key, long seconds, Object... values) {
		setAdd(key, values);
		setExpireTime(key, seconds);
	}

	/**
	 * 判断set是否包含value
	 */
	public static Boolean setContains(String key, Object value) {
		return redisJsonUtils.redisTemplate.opsForSet().isMember(key, value);
	}

	/**
	 * 删除set中key,value对应的元素
	 */
	public static long setRemove(String key, Object... values) {
		return redisJsonUtils.redisTemplate.opsForSet().remove(key, values);
	}

	/**
	 * 获取set中key的所有数据
	 */
	public static Set<Object> setMembers(String key) {
		return redisJsonUtils.redisTemplate.opsForSet().members(key);
	}

	/**
	 * hash方式设置
	 */
	public static void hashPut(String key, String hkey, Object hvalue) {
		Assert.hasText(key, "key is not null");
		Assert.hasText(key, "hkey is not null");
		redisJsonUtils.redisTemplate.opsForHash().put(key, hkey, hvalue);
	}

	/**
	 * hash方式设值，带有效期
	 */
	public static void hashPut(String key, String hkey, Object hvalue, long seconds) {
		hashPut(key, hkey, hvalue);
		setExpireTime(key, seconds);
	}

	/**
	 * 获取hash值
	 */
	public static Object hashGet(String key, String hkey) {
		Assert.hasText(key, "key is not null");
		Assert.hasText(key, "hkey is not null");
		return redisJsonUtils.redisTemplate.opsForHash().get(key, hkey);
	}

	/**
	 * 删除hash中key,value对应的元素
	 */
	public static long hashRemove(String key, Object... hkey) {
		Assert.hasText(key, "key is not null");
		return redisJsonUtils.redisTemplate.opsForHash().delete(key, hkey);
	}

	/**
	 * 获取key对应的hash的所有元素
	 */
	public static Map<String, Object> hashEntries(String key) {
		Assert.hasText(key, "key is not null");
		BoundHashOperations<String, String, Object> ops = redisJsonUtils.redisTemplate.boundHashOps(key);
		return ops.entries();
	}

	/**
	 * 获取set中key的所有HKey
	 */
	public static Set<String> hashKeys(String key) {
		return redisJsonUtils.redisTemplate.<String, Object>opsForHash().keys(key);
	}

	public static Boolean hashHasKey(String key, String hashKey) {
		return redisJsonUtils.redisTemplate.<String, Object>opsForHash().hasKey(key, hashKey);
	}

	public static Long hashSize(String key) {
		return redisJsonUtils.redisTemplate.<String, Object>opsForHash().size(key);
	}

	public static void hashPutAll(String key, Map<String, Object> m) {
		redisJsonUtils.redisTemplate.<String, Object>opsForHash().putAll(key, m);
	}

	/**
	 * 获取key的剩余有效期，单位秒
	 */
	public static long getExpire(String key) {
		return redisJsonUtils.redisTemplate.getExpire(key);
	}

	/**
	 * 根据pattern正则表达式获取keys
	 */
	public static Set<String> getKeys(String pattern) {
		return redisJsonUtils.redisTemplate.keys(pattern);
	}

	/**
	 * 对key进行原子累增
	 */
	public static long increment(String key) {
		return redisJsonUtils.redisTemplate.boundValueOps(key).increment(1l);
	}
	
	/**
	 * 查询缓存
	 * 
	 * @作者: yangguangjing
	 * @创建时间: 2020年1月10日  下午2:43:55
	 */
	public static <T> T queryWithCache(IRedisKey redisKey, Supplier<T> supplier) {
		return queryWithCache(redisKey, redisKey.getExpireSeconds(), supplier);
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T queryWithCache(IRedisKey redisKey, long seconds, Supplier<T> supplier) {
		T obj = (T) get(redisKey.getKey());
		if (obj == null) {
			obj = supplier.get();
			set(redisKey.getKey(), obj, seconds);
		}
		return obj;
	}
}