package cn.ibizlab.util.cache.cache;

import com.github.benmanes.caffeine.cache.Cache;
import org.springframework.cache.caffeine.CaffeineCache;

import javax.validation.constraints.NotNull;
import java.util.Set;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;

/**
 * 自定义的Caffeine缓存
 */
public class CusCaffeineCache extends CaffeineCache{

    public CusCaffeineCache(String name, Cache<Object, Object> cache) {
        super(name,cache);
    }

    public CusCaffeineCache(String name, Cache<Object, Object> cache, boolean allowNullValues) {
        super(name,cache,allowNullValues);
    }


    @Override
    public void evict(@NotNull Object key) {
        if (key instanceof String && ((String) key).startsWith("glob:")) {
            String globPattern = ((String)key).split("glob:")[1];
            // 将Glob匹配转换成正则匹配
            String regexPattern = Globs.toUnixRegexPattern(globPattern);
            Cache<Object,Object> cache = super.getNativeCache();
            Set<Object> keySet = cache.asMap().keySet();
            keySet = keySet.stream().filter(o->o.toString().matches(regexPattern)).collect(Collectors.toSet());

            cache.invalidateAll(keySet);
        }else{
            super.evict(key);
        }
    }

    public static class Globs {
        private Globs() { }

        private static final String REGEX_META_CHARS = ".^$+{[]|()";
        private static final String GLOB_META_CHARS = "\\*?[{";

        private static boolean isRegexMeta(char c) {
            return REGEX_META_CHARS.indexOf(c) != -1;
        }

        private static boolean isGlobMeta(char c) {
            return GLOB_META_CHARS.indexOf(c) != -1;
        }
        private static char EOL = 0;  //TBD

        private static char next(String glob, int i) {
            if (i < glob.length()) {
                return glob.charAt(i);
            }
            return EOL;
        }

        /**
         * Creates a regex pattern from the given glob expression.
         *
         * @throws PatternSyntaxException
         */
        private static String toRegexPattern(String globPattern, boolean isDos) {
            boolean inGroup = false;
            StringBuilder regex = new StringBuilder("^");

            int i = 0;
            while (i < globPattern.length()) {
                char c = globPattern.charAt(i++);
                switch (c) {
                    case '\\':
                        // escape special characters
                        if (i == globPattern.length()) {
                            throw new PatternSyntaxException("No character to escape",
                                    globPattern, i - 1);
                        }
                        char next = globPattern.charAt(i++);
                        if (isGlobMeta(next) || isRegexMeta(next)) {
                            regex.append('\\');
                        }
                        regex.append(next);
                        break;
                    case '/':
                        if (isDos) {
                            regex.append("\\\\");
                        } else {
                            regex.append(c);
                        }
                        break;
                    case '[':
                        // don't match name separator in class
                        if (isDos) {
                            regex.append("[[^\\\\]&&[");
                        } else {
                            regex.append("[[^/]&&[");
                        }
                        if (next(globPattern, i) == '^') {
                            // escape the regex negation char if it appears
                            regex.append("\\^");
                            i++;
                        } else {
                            // negation
                            if (next(globPattern, i) == '!') {
                                regex.append('^');
                                i++;
                            }
                            // hyphen allowed at start
                            if (next(globPattern, i) == '-') {
                                regex.append('-');
                                i++;
                            }
                        }
                        boolean hasRangeStart = false;
                        char last = 0;
                        while (i < globPattern.length()) {
                            c = globPattern.charAt(i++);
                            if (c == ']') {
                                break;
                            }
                            if (c == '/' || (isDos && c == '\\')) {
                                throw new PatternSyntaxException("Explicit 'name separator' in class",
                                        globPattern, i - 1);
                            }
                            // TBD: how to specify ']' in a class?
                            if (c == '\\' || c == '[' ||
                                    c == '&' && next(globPattern, i) == '&') {
                                // escape '\', '[' or "&&" for regex class
                                regex.append('\\');
                            }
                            regex.append(c);

                            if (c == '-') {
                                if (!hasRangeStart) {
                                    throw new PatternSyntaxException("Invalid range",
                                            globPattern, i - 1);
                                }
                                if ((c = next(globPattern, i++)) == EOL || c == ']') {
                                    break;
                                }
                                if (c < last) {
                                    throw new PatternSyntaxException("Invalid range",
                                            globPattern, i - 3);
                                }
                                regex.append(c);
                                hasRangeStart = false;
                            } else {
                                hasRangeStart = true;
                                last = c;
                            }
                        }
                        if (c != ']') {
                            throw new PatternSyntaxException("Missing ']", globPattern, i - 1);
                        }
                        regex.append("]]");
                        break;
                    case '{':
                        if (inGroup) {
                            throw new PatternSyntaxException("Cannot nest groups",
                                    globPattern, i - 1);
                        }
                        regex.append("(?:(?:");
                        inGroup = true;
                        break;
                    case '}':
                        if (inGroup) {
                            regex.append("))");
                            inGroup = false;
                        } else {
                            regex.append('}');
                        }
                        break;
                    case ',':
                        if (inGroup) {
                            regex.append(")|(?:");
                        } else {
                            regex.append(',');
                        }
                        break;
                    case '*':
                        if (next(globPattern, i) == '*') {
                            // crosses directory boundaries
                            regex.append(".*");
                            i++;
                        } else {
                            // within directory boundary
                            if (isDos) {
                                regex.append("[^\\\\]*");
                            } else {
                                regex.append("[^/]*");
                            }
                        }
                        break;
                    case '?':
                        if (isDos) {
                            regex.append("[^\\\\]");
                        } else {
                            regex.append("[^/]");
                        }
                        break;

                    default:
                        if (isRegexMeta(c)) {
                            regex.append('\\');
                        }
                        regex.append(c);
                }
            }

            if (inGroup) {
                throw new PatternSyntaxException("Missing '}", globPattern, i - 1);
            }

            return regex.append('$').toString();
        }

        public static String toUnixRegexPattern(String globPattern) {
            return toRegexPattern(globPattern, false);
        }

        public static String toWindowsRegexPattern(String globPattern) {
            return toRegexPattern(globPattern, true);
        }
    }
}
