package cn.ibizlab.util.security;

import cn.ibizlab.util.service.AuthenticationUserService;
import io.jsonwebtoken.ExpiredJwtException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.PathMatcher;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

@Slf4j
@Component
public class AuthorizationTokenFilter extends OncePerRequestFilter {

    private final UserDetailsService userDetailsService;

    private final String tokenHeader;


    @Value("${ibiz.auth.excludesPattern:}")
    private String[] excludesPattens;

    private Set<String> excludesPattern ;
    public Set<String> getExcludesPattern() {
        if(excludesPattern==null) {
            excludesPattern=new HashSet<>(Arrays.asList(excludesPattens));
        }
        return excludesPattern;
    }
    private PathMatcher pathMatcher = new AntPathMatcher();

    public AuthorizationTokenFilter(UserDetailsService userDetailsService, @Value("${ibiz.jwt.header:Authorization}") String tokenHeader) {
        this.userDetailsService = userDetailsService;
        this.tokenHeader = tokenHeader;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws ServletException, IOException {
        if (isExclusion(request.getRequestURI())) {
            chain.doFilter(request, response);
            return;
        }

        if(SecurityContextHolder.getContext().getAuthentication() != null){
            chain.doFilter(request, response);
            return;
        }

        final String requestHeader = request.getHeader(this.tokenHeader);

        String username = null;
        String authToken = null;
        if (requestHeader != null && requestHeader.startsWith("Bearer ")) {
            authToken = requestHeader.substring(7);
            try {
                username = ((AuthenticationUserService)userDetailsService).getUsernameFromToken(authToken);
            } catch (ExpiredJwtException e) {
                log.error(e.getMessage());
            }
        }

        if (username != null && SecurityContextHolder.getContext().getAuthentication() == null) {
            UserDetails userDetails = this.userDetailsService.loadUserByUsername(username);
            boolean validate = true;
            if(userDetailsService instanceof  AuthenticationUserService) {
                validate = ((AuthenticationUserService)userDetailsService).validateToken(authToken, userDetails);
                if(userDetails instanceof AuthenticationUser)
                    ((AuthenticationUser)userDetails).readHeader(false);
            }
            if (validate) {
                UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(userDetails, null, userDetails.getAuthorities());
                authentication.setDetails(new WebAuthenticationDetailsSource().buildDetails(request));
                SecurityContextHolder.getContext().setAuthentication(authentication);
            }
        }
        chain.doFilter(request, response);
    }

    public void addExcludePattern(String excludePattern) {
        getExcludesPattern().add(excludePattern);
    }

    public boolean isExclusion(String requestURI) {
        if (this.getExcludesPattern() == null) {
            return false;
        } else {
            for(String pattern:this.getExcludesPattern()) {
                if(pathMatcher.match(pattern, requestURI))
                    return true;
            }
            return false;
        }
    }
}
