package net.ibizsys.model.engine;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;

import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import net.ibizsys.model.IPSModelObject;
import net.ibizsys.model.IPSSystem;
import net.ibizsys.model.IPSSystemService;
import net.ibizsys.model.PSModelEnums.SysUtilType;
import net.ibizsys.model.PSModelUtils;
import net.ibizsys.model.dataentity.dataflow.IPSDEDataFlowNode;
import net.ibizsys.model.dataentity.logic.IPSDELogicNode;
import net.ibizsys.model.dataentity.service.IPSDEMethodDTO;
import net.ibizsys.model.engine.dataentity.dataflow.IPSDEDataFlowNodeAddin;
import net.ibizsys.model.engine.dataentity.logic.IPSDELogicNodeAddin;
import net.ibizsys.model.engine.dataentity.service.IPSDEMethodDTOEngine;
import net.ibizsys.model.engine.eai.IPSSysDataSyncAgentEngine;
import net.ibizsys.model.engine.plugin.IPSModelEngineAddin;
import net.ibizsys.model.engine.res.IPSSysSFPluginEngine;
import net.ibizsys.model.engine.service.IPSSubSysServiceAPIEngine;
import net.ibizsys.model.engine.sysutil.IPSSysUtilEngine;
import net.ibizsys.model.res.IPSSysDataSyncAgent;
import net.ibizsys.model.res.IPSSysSFPlugin;
import net.ibizsys.model.res.IPSSysSFPluginSupportable;
import net.ibizsys.model.res.IPSSysUtil;
import net.ibizsys.model.service.IPSSubSysServiceAPI;
import net.ibizsys.model.util.DataTypeUtils;

public abstract class PSModelEngineHolderBase implements IPSModelEngineHolder{

	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(PSModelEngineHolderBase.class);
	
	private final IPSSystemService iPSSystemService;
	private final IPSSystem iPSSystem;
	private final IPSModelEngineFactory iPSModelEngineFactory;
	private Map<String, Object> paramMap = new HashMap<String, Object>();
	private Map<String, IPSModelEngine<?>> psModelEngineMap = new ConcurrentHashMap<String, IPSModelEngine<?>>();
	
	private Map<String, BiFunction<IPSModelEngineHolder, IPSModelObject, String>> psModelSubTypeFunctionMap = new HashMap<>();
	
	private static Map<String, BiFunction<IPSModelEngineHolder, IPSModelObject, String>> PSModelSubTypeFunctionMap = new HashMap<>();
	
//	private static Map<String, Class<? extends IPSModelEngine>> PSModelEngineAliasMap = new HashMap<String, Class<? extends IPSModelEngine>>();
//	private static Map<String, Class<? extends IPSModelEngineAddin>> PSModelEngineAddinAliasMap = new HashMap<String, Class<? extends IPSModelEngineAddin>>();
	
	
	private IPSSystemEngine<?> iPSSystemEngine = null;
	private IPSSystemLogger iPSSystemLogger = null;
	private IPSSystemUtil iPSSystemUtil = null;
//	private static Map<String, BiFunction<IPSModelEngineHolder, IPSModelObject, String>> GlobalPSModelTagFunctionMap = new HashMap<>();
	
	static {
		PSModelSubTypeFunctionMap.put(IPSSysDataSyncAgentEngine.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
			@Override
			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
				IPSSysDataSyncAgent iPSSysDataSyncAgent = (IPSSysDataSyncAgent)iPSModelObject;
				String configFolder = IPSModelEngineHolder.CONFIGFOLDER_SYSEAIAGENT + "." + PSModelUtils.calcUniqueTag(iPSSysDataSyncAgent.getPSSystemModule(), iPSSysDataSyncAgent.getCodeName());
				return iPSModelEngineHolder.getParam(configFolder + ".agenttype", iPSSysDataSyncAgent.getAgentType());
			}
		});
		
		PSModelSubTypeFunctionMap.put(IPSSubSysServiceAPIEngine.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
			@Override
			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
				IPSSubSysServiceAPI iPSSubSysServiceAPI = (IPSSubSysServiceAPI)iPSModelObject;
				return iPSSubSysServiceAPI.getServiceType();
			}
		});
		
		PSModelSubTypeFunctionMap.put(IPSSysUtilEngine.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
			@Override
			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
				IPSSysUtil iPSSysUtil = (IPSSysUtil)iPSModelObject;
				
				if(SysUtilType.USER.value.equals(iPSSysUtil.getUtilType())) {
					return String.format("%1$s:%2$s", iPSSysUtil.getUtilType(), iPSSysUtil.getUtilTag());
				}
				
				return iPSSysUtil.getUtilType();
			}
		});
		
		PSModelSubTypeFunctionMap.put(IPSDELogicNodeAddin.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
			@Override
			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
				IPSDELogicNode iPSDELogicNode = (IPSDELogicNode)iPSModelObject;
				return iPSDELogicNode.getLogicNodeType();
			}
		});
		
		PSModelSubTypeFunctionMap.put(IPSDEDataFlowNodeAddin.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
			@Override
			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
				IPSDEDataFlowNode iPSDEDataFlowNode = (IPSDEDataFlowNode)iPSModelObject;
				return iPSDEDataFlowNode.getNodeType();
			}
		});
		
		
		PSModelSubTypeFunctionMap.put(IPSDEMethodDTOEngine.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
			@Override
			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
				IPSDEMethodDTO iPSDEMethodDTO = (IPSDEMethodDTO)iPSModelObject;
				return iPSDEMethodDTO.getType();
			}
		});
		
		
//		GlobalPSModelTagFunctionMap.put(IPSSysDataSyncAgentEngine.class.getSimpleName(), new BiFunction<IPSModelEngineHolder, IPSModelObject, String>() {
//			@Override
//			public String apply(IPSModelEngineHolder iPSModelEngineHolder, IPSModelObject iPSModelObject) {
//				IPSSysDataSyncAgent iPSSysDataSyncAgent = (IPSSysDataSyncAgent)iPSModelObject;
//				return String.format("%1$s.%2$s",  IPSSysDataSyncAgentEngine.class.getSimpleName(), PSModelUtils.calcUniqueTag(iPSSysDataSyncAgent.getPSSystemModule(), iPSSysDataSyncAgent.getCodeName()));
//			}
//		});
	}
	
	
	public PSModelEngineHolderBase(IPSModelEngineFactory iPSModelEngineFactory, IPSSystemService iPSSystemService, Map<String, Object> map) {
		Assert.notNull(iPSModelEngineFactory, "传入模型引擎工厂对象无效");
		Assert.notNull(iPSSystemService, "传入系统模型服务对象无效");
		this.iPSModelEngineFactory = iPSModelEngineFactory;
		this.iPSSystemService = iPSSystemService;
		this.iPSSystem = iPSSystemService.getPSSystem();
		reload(map);
		
		//准备默认
		try {
			this.iPSSystemEngine = this.getPSModelEngine(this.iPSSystem, IPSSystemEngine.class);
			this.iPSSystemLogger = this.getPSModelEngineAddin(this.getPSSystemEngine(), null, IPSSystemLogger.class);
			this.iPSSystemUtil = this.getPSModelEngineAddin(this.getPSSystemEngine(), null, IPSSystemUtil.class);
			this.iPSSystemEngine.start();
		}
		catch (Exception ex) {
			throw new RuntimeException(ex);
		}
	}
	
	@Override
	public IPSSystemService getPSSystemService() {
		return this.iPSSystemService;
	}
	
	@Override
	public IPSModelEngineFactory getPSModelEngineFactory() {
		return this.iPSModelEngineFactory;
	}
	
	protected void reload(Map<String, Object> map) {
		this.paramMap.clear();
		if(map!=null) {
			this.load("", map, true);
		}
		
	}

	protected void load(String pkey, Map<String, Object> map, boolean bOverwrite) {
		for (java.util.Map.Entry<String, Object> entry : map.entrySet()) {
			String strKey = null;
			if (StringUtils.hasLength(pkey)) {
				strKey = String.format("%1$s.%2$s", pkey, entry.getKey()).toLowerCase();
			} else {
				strKey = entry.getKey().toLowerCase();
			}
			if (entry.getValue() instanceof Map) {
				load(strKey, (Map) entry.getValue(), bOverwrite);
			} else {
				if (!bOverwrite) {
					if(containsParam(strKey)) {
						continue;
					}
				}
				setParam(strKey, entry.getValue());
			}
		}
	}

	@Override
	public void setParams(String pkey, Map<String, Object> params) {
		this.load(pkey, params, true);
	}

	@Override
	public void setParamsIf(String pkey, Map<String, Object> params) {
		this.load(pkey, params, false);
	}
	
	
	@Override
	public void setParam(String key, Object objValue) {
		Assert.hasLength(key, "传入键名为空");
		if (objValue != null) {
			this.paramMap.put(key.toLowerCase(), objValue);
		} else {
			this.paramMap.remove(key.toLowerCase());
		}

	}

	@Override
	public String getParam(String key, String strDefault) {
		Object objValue = this.getParam(key);
		if (objValue != null) {
			if (objValue instanceof String) {
				if (!ObjectUtils.isEmpty(objValue)) {
					return (String) objValue;
				}
			} else {
				return objValue.toString();
			}
		}
		return strDefault;
	}
	
	

	@Override
	public int getParam(String key, int defaultValue) throws Exception{
		Object objValue = this.getParam(key);
		if (objValue != null) {
			return DataTypeUtils.getIntegerValue(objValue, defaultValue);
		}
		return defaultValue;
	}

	@Override
	public long getParam(String key, long defaultValue) throws Exception{
		Object objValue = this.getParam(key);
		if (objValue != null) {
			return DataTypeUtils.getLongValue(objValue, defaultValue);
		}
		return defaultValue;
	}
	

	@Override
	public double getParam(String key, double defaultValue) throws Exception{
		Object objValue = this.getParam(key);
		if (objValue != null) {
			return DataTypeUtils.getDoubleValue(objValue, defaultValue);
		}
		return defaultValue;
	}
	
	@Override
	public boolean getParam(String key, boolean bDefault) throws Exception{
		Object objValue = this.getParam(key);
		if (objValue != null) {
			return DataTypeUtils.getBooleanValue(objValue, bDefault);
		}
		return bDefault;
	}
	

	@Override
	public Object getParam(String key) {
		Assert.hasLength(key, "传入键名为空");
		return this.paramMap.get(key.toLowerCase());
	}

	@Override
	public boolean containsParam(String key) {
		Assert.hasLength(key, "传入键名为空");
		return this.paramMap.containsKey(key.toLowerCase());
	}

	@Override
	public Map<String, Object> getParams(String pkey, Map<String, Object> params) {
		String pkeyPrefix = (pkey + ".").toLowerCase();
		int nPKeyLength = pkeyPrefix.length();
		for (java.util.Map.Entry<String, Object> entry : this.paramMap.entrySet()) {
			int nPos = entry.getKey().indexOf(pkeyPrefix);
			if (nPos != 0) {
				continue;
			}
			String key = entry.getKey().substring(nPKeyLength);
			if (params == null) {
				params = new HashMap<>();
			}
			params.put(key, entry.getValue());
		}
		return params;
	}

	@Override
	public void removeParam(String key) {
		Assert.hasLength(key, "传入键名为空");
		this.paramMap.remove(key.toLowerCase());
	}

	@Override
	public void removeParams(String pkey) {
		Map<String, Object> params = this.getParams(pkey, null);
		if(!ObjectUtils.isEmpty(params)) {
			this.removeParams(params.keySet());
		}
	}
	
	@Override
	public void removeParams(Set<String> keys) {
		Assert.notEmpty(keys, "传入键名集合为空");
		for(String key : keys) {
			this.removeParam(key);
		}
	}
	
	@Override
	public IPSSystemEngine<?> getPSSystemEngine() {
		return this.iPSSystemEngine;
	}
	
//	@Override
//	public <T extends IPSSystemEngine> T getPSSystemEngine(Class<T> psModelEngineCls) throws Exception {
//		return this.getPSModelEngine(iPSSystem, psModelEngineCls);
//	}

	@Override
	public IPSSystemLogger getPSSystemLogger() {
		return this.iPSSystemLogger;
	}
	
	@Override
	public IPSSystemUtil getPSSystemUtil() {
		return this.iPSSystemUtil;
	}
	
	@Override
	public <T extends IPSModelEngine<?>> T getPSModelEngine(IPSModelObject iPSModelObject, Class<T> psModelEngineCls) throws Exception {
		return this.getPSModelEngine(iPSModelObject, psModelEngineCls, null);
	}

	@Override
	public <T extends IPSModelEngine<?>> T getPSModelEngine(IPSModelObject iPSModelObject, Class<T> psModelEngineCls, String subType) throws Exception {
		
		String psModelEngineTag = getPSModelEngineTag(iPSModelObject, psModelEngineCls);
		Object engine = psModelEngineMap.get(psModelEngineTag); 
		if(engine != null) {
			return (T)engine;
		}
		
		if(iPSModelObject instanceof IPSSysSFPluginSupportable) {
			IPSSysSFPlugin iPSSysSFPlugin = ((IPSSysSFPluginSupportable)iPSModelObject).getPSSysSFPlugin();
			if(iPSSysSFPlugin != null) {
				IPSSysSFPluginEngine iPSSysSFPluginEngine = this.getPSModelEngine(iPSSysSFPlugin, IPSSysSFPluginEngine.class, null);
				engine = iPSSysSFPluginEngine.getRuntimeObject(psModelEngineCls, true);
			}
		}
		
		if(engine == null) {
			if(!StringUtils.hasLength(subType)) {
				subType = getPSModelSubType(iPSModelObject, psModelEngineCls);
			}
			
			engine = this.iPSModelEngineFactory.getObject(psModelEngineCls, subType);
			if(engine == null) {
				if(StringUtils.hasLength(subType)) {
					engine = this.iPSModelEngineFactory.getObject(psModelEngineCls, null);
				}
				if(engine == null) {
					if(StringUtils.hasLength(subType)) {
						throw new Exception(String.format("无法建立模型对象[%1$s][%2$s]引擎对象", iPSModelObject, subType));
					}
					else {
						throw new Exception(String.format("无法建立模型对象[%1$s]引擎对象", iPSModelObject));
					}
				}
			}
		}
		
		
		IPSModelEngine iPSModelEngine = (IPSModelEngine)engine;
		try {
			iPSModelEngine.init(this, psModelEngineTag, iPSModelObject);
		}
		catch (Exception ex) {
			throw new Exception(String.format("初始化模型对象[%1$s]引擎对象发生异常，%2$s", iPSModelObject, ex.getMessage()));
		}
		psModelEngineMap.put(psModelEngineTag, iPSModelEngine);
		return (T)iPSModelEngine;
	}
	
	
	protected String getPSModelEngineTag(IPSModelObject iPSModelObject, Class<?> psModelEngineCls) throws Exception{
		String modelTag = PSModelUtils.calcFullUniqueTag(iPSModelObject, true);
		if(!StringUtils.hasLength(modelTag)) {
			throw new Exception(String.format("无法计算模型对象[%1$s]标记", iPSModelObject));
		}
		return String.format("%1$s.%2$s",  psModelEngineCls.getSimpleName(), modelTag);
	}

	@Override
	public <T extends IPSModelEngineAddin<?, ?>> T getPSModelEngineAddin(IPSModelEngine<?> iPSModelEngine, Object data, Class<T> psModelEngineCls) throws Exception {
		return this.getPSModelEngineAddin(iPSModelEngine, data, psModelEngineCls, null);
	}

	@Override
	public <T extends IPSModelEngineAddin<?, ?>> T getPSModelEngineAddin(IPSModelEngine<?> iPSModelEngine, Object data, Class<T> psModelEngineCls, String subType) throws Exception {
		
		IPSModelEngineAddin addin = null;
		if(data instanceof IPSSysSFPluginSupportable) {
			IPSSysSFPlugin iPSSysSFPlugin = ((IPSSysSFPluginSupportable)data).getPSSysSFPlugin();
			if(iPSSysSFPlugin != null) {
				IPSSysSFPluginEngine<?> iPSSysSFPluginEngine = this.getPSModelEngine(iPSSysSFPlugin, IPSSysSFPluginEngine.class, null);
				addin = iPSSysSFPluginEngine.getRuntimeObject(psModelEngineCls, true);
			}
		}
		
		if(addin == null) {
			if(!StringUtils.hasLength(subType) && data instanceof IPSModelObject) {
				subType = getPSModelSubType((IPSModelObject)data, psModelEngineCls);
			}
			
			addin = this.iPSModelEngineFactory.getObject(psModelEngineCls, subType);
			if(addin == null) {
				if(StringUtils.hasLength(subType)) {
					addin = this.iPSModelEngineFactory.getObject(psModelEngineCls, null);
				}
				if(addin == null) {
					if(StringUtils.hasLength(subType)) {
						throw new Exception(String.format("无法建立模型引擎插件[%1$s][%2$s]", psModelEngineCls.getSimpleName(), subType));
					}
					else {
						throw new Exception(String.format("无法建立模型引擎插件[%1$s]引擎对象", psModelEngineCls.getSimpleName()));
					}
				}
			}
		}
		
		try {
			addin.init(this, iPSModelEngine, data);
			return (T)addin;
		}
		catch (Exception ex) {
			throw new Exception(String.format("初始化模型引擎插件[%1$s]发生异常，%2$s", psModelEngineCls.getSimpleName(), ex.getMessage()));
		}
	}

	
	protected String getPSModelSubType(IPSModelObject iPSModelObject, Class<?> psModelEngineCls) {
		BiFunction<IPSModelEngineHolder, IPSModelObject, String> biFunction = PSModelSubTypeFunctionMap.get(psModelEngineCls.getSimpleName());
		if(biFunction!=null) {
			return biFunction.apply(this, iPSModelObject);
		}
		return null;
	}

	
}
