package net.ibizsys.model.engine;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import net.ibizsys.model.PSModelEnums.DEMethodDTOType;
import net.ibizsys.model.engine.dataentity.action.PSDEActionInputDTOEngine;
import net.ibizsys.model.engine.dataentity.ds.PSDEFilterDTOEngine;
import net.ibizsys.model.engine.dataentity.service.IPSDEMethodDTOEngine;
import net.ibizsys.model.engine.dataentity.service.PSDEMethodDTOEngine;
import net.ibizsys.model.engine.service.IPSSubSysServiceAPIEngine;
import net.ibizsys.model.engine.service.PSSubSysServiceAPIEngine;
import net.ibizsys.model.engine.service.client.IWebClient;
import net.ibizsys.model.engine.service.client.WebClient;
import net.ibizsys.model.engine.sysutil.IPSSysUtilEngine;
import net.ibizsys.model.engine.sysutil.PSSysUtilEngine;

/**
 * 模型引擎工厂对象
 * @author lionlau
 *
 */
public abstract class PSModelEngineFactoryBase implements IPSModelEngineFactory {

	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(PSModelEngineFactoryBase.class);

	static class EngineObject{
		public Object object = null;
		public int order = 100;
		
		public EngineObject() {
			
		}
	}
	
	/**
	 * 注册数据为静态变量
	 */
	private Map<String, EngineObject> engineObjectMap = new ConcurrentHashMap<>();
	
	private Map<String, Map> engineObjectListMap = new ConcurrentHashMap<>();
	
	public PSModelEngineFactoryBase() {
		prepareEngineObjects();
	}
	
	protected void prepareEngineObjects() {
		
		this.registerObjectIf(IPSDEMethodDTOEngine.class, DEMethodDTOType.DEFAULT.value, PSDEMethodDTOEngine.class);
		this.registerObjectIf(IPSDEMethodDTOEngine.class, DEMethodDTOType.DEFILTER.value, PSDEFilterDTOEngine.class);
		this.registerObjectIf(IPSDEMethodDTOEngine.class, DEMethodDTOType.DEDATASETINPUT.value, PSDEFilterDTOEngine.class);
		this.registerObjectIf(IPSDEMethodDTOEngine.class, DEMethodDTOType.DEACTIONINPUT.value, PSDEActionInputDTOEngine.class);
		
		
		this.registerObjectIf(IPSSystemLogger.class, null, PSSystemLogger.class);
		this.registerObjectIf(IPSSystemUtil.class, null, PSSystemUtil.class);
		this.registerObjectIf(IPSSysUtilEngine.class, null, PSSysUtilEngine.class);
		this.registerObjectIf(IPSSubSysServiceAPIEngine.class, null, PSSubSysServiceAPIEngine.class);
		this.registerObjectIf(IWebClient.class, null, WebClient.class);
		
	}

	protected boolean registerEngineObject(Class<?> cls, String type, Object objObject) {
		return registerEngineObject(cls, type, objObject, ORDER_DEFAULT);
	}
	
	protected boolean registerEngineObject(Class<?> cls, String type, Object objObject, int order) {
		String strTag = null;
		if (StringUtils.hasLength(type)) {
			strTag = String.format("%1$s|%2$s", getObjectType(cls), type);
		} else {
			strTag = String.format("%1$s|", getObjectType(cls));
		}
		
		if(objObject instanceof String) {
			try {
				objObject = Class.forName((String)objObject);
			} catch (Exception ex) {
				//log.debug(String.format("忽略注册引擎对象[%1$s][%2$s]，对象不存在", objObject, strTag));
				return false;
			}
		}
		
		EngineObject lastObject = engineObjectMap.get(strTag);
		if(lastObject != null) {
			if(order > lastObject.order){
				log.warn(String.format("无法注册引擎对象[%1$s][%2$s]，优先级不足", objObject, strTag));
				return false;
			}
			else {
				log.warn(String.format("注册引擎对象[%1$s][%2$s]，替换[%3$s]", objObject, strTag, lastObject.object));
			}
		}
		else {
			log.debug(String.format("注册引擎对象[%1$s][%2$s]", objObject, strTag));
		}
		
		EngineObject runtimeObject = new EngineObject();
		
		runtimeObject.object = objObject;
		runtimeObject.order = order;
		
		engineObjectMap.put(strTag, runtimeObject);
		
		return true;
		
//		if (StringUtils.hasLength(type)) {
//			engineObjectMap.put(String.format("%1$s|%2$s", cls.getCanonicalName(), type), objObject);
//		} else {
//			engineObjectMap.put(String.format("%1$s|", cls.getCanonicalName()), objObject);
//		}
	}
	
	
	@Override
	public boolean registerObject(Class<?> cls, String type, Object objectType) {
		return registerObject(cls, type, objectType, ORDER_DEFAULT);
	}
	
	@Override
	public boolean registerObjectPrimary(Class<?> cls, String type, Object objectType) {
		return registerObject(cls, type, objectType, ORDER_PRIMARY);
	}
	
	@Override
	public boolean registerObject(Class<?> cls, String type, Object objectType, int orderValue) {
		return registerEngineObject(cls, type, objectType, orderValue);
	}
	
	@Override
	public boolean registerObjectIf(Class<?> cls, String type, Object objectType) {
		if(this.containsObject(cls, type)) {
			return false;
		}
		return this.registerObject(cls, type, objectType);
	}
	
	@Override
	public boolean containsObject(Class<?> cls, String type) {
		Object runtimeObject = null;
		if (StringUtils.hasLength(type)) {
			runtimeObject = engineObjectMap.get(String.format("%1$s|%2$s", getObjectType(cls) , type));
		} else {
			runtimeObject = engineObjectMap.get(String.format("%1$s|", getObjectType(cls)));
		}
		return !ObjectUtils.isEmpty(runtimeObject);
	}
	
	
	@Override
	public <T> T getObject(Class<T> cls){
		return this.getObject(cls, "");
	}
	
	@Override
	public <T> T getObject(Class<T> cls, String type){
		EngineObject runtimeObject = null;
		if (StringUtils.hasLength(type)) {
			runtimeObject = engineObjectMap.get(String.format("%1$s|%2$s", getObjectType(cls), type));
		} else {
			runtimeObject = engineObjectMap.get(String.format("%1$s|", getObjectType(cls)));
		}

		if (ObjectUtils.isEmpty(runtimeObject)) {
			return null;
		}
		
		return getObject(cls, runtimeObject, false);
	}
	
	
	@Override
	public Class<?> getObjectClass(Class<?> cls) {
		return this.getObjectClass(cls, "");
	}

	@Override
	public Class<?> getObjectClass(Class<?> cls, String type) {
		EngineObject runtimeObject = null;
		if (StringUtils.hasLength(type)) {
			runtimeObject = engineObjectMap.get(String.format("%1$s|%2$s", getObjectType(cls), type));
		} else {
			runtimeObject = engineObjectMap.get(String.format("%1$s|", getObjectType(cls)));
		}

		if (ObjectUtils.isEmpty(runtimeObject)) {
			return null;
		}
		
		Object obj = runtimeObject.object;
		if(obj == null) {
			return null;
		}
		
		if(obj instanceof Class) {
			return (Class)obj;
		}
		else
			if(obj instanceof String) {
				try {
					return Class.forName((String)obj);
				} catch (Exception ex) {
					log.warn(String.format("引擎对象[%1$s]不存在", obj));
					return null;
				}
			}
		else
			return obj.getClass();
	}

	

	@Override
	public <T> Map<String, T> getObjects(Class<T> cls){
		return getObjects(cls, null);
	}

	@Override
	public <T> Map<String, T> getObjects(Class<T> cls, String typePrefix){
		String strTag = null;
		if(StringUtils.hasLength(typePrefix)) {
			strTag = String.format("%1$s|%2$s", getObjectType(cls), typePrefix);
		} else {
			strTag = String.format("%1$s|", getObjectType(cls));
		}
		Object objMap = engineObjectListMap.get(strTag);
		if(objMap != null) {
			return (Map<String, T>)objMap;
		}
		Map<String, T> map = new HashMap<String, T>();
		for(java.util.Map.Entry<String, EngineObject> entry : engineObjectMap.entrySet()) {
			if(entry.getKey().indexOf(strTag)!=0) {
				continue;
			}
			
			T t = getObject(cls, entry.getValue(), true);
			if(t != null) {
				map.put(entry.getKey().substring( getObjectType(cls).length() + 1), t);
			}
		}
		
		engineObjectListMap.put(strTag, map);
		return map;
	}
	
	protected <T> T getObject(Class<T> cls, EngineObject runtimeObject, boolean bTryMode){
		
		Object obj = runtimeObject.object;
		
		if(obj instanceof Class) {
			Object objInstance;
			try {
				objInstance = createObject(obj);
			} catch (Exception e) {
				if(bTryMode) {
					log.warn(String.format("无法建立引擎对象[%1$s]", obj));
					return null;
				}
				throw new RuntimeException(e);
			}
			if (cls.isAssignableFrom(objInstance.getClass())) {
				return (T) objInstance;
			}
		}
		else
			if(obj instanceof String) {
				Object objInstance;
				try {
					objInstance =  createObject(obj);
				} catch (Exception e) {
					if(bTryMode) {
						log.warn(String.format("无法建立引擎对象[%1$s]", obj));
						return null;
					}
					throw new RuntimeException(e);
				}
				if (cls.isAssignableFrom(objInstance.getClass())) {
					return (T) objInstance;
				}
			}
		else
			if (cls.isAssignableFrom(obj.getClass())) {
				return (T) obj;
			}
			
		if(bTryMode) {
			return null;
		}
		throw new RuntimeException(String.format("[%1$s]类型不正确", obj));
	}
	
	protected String getObjectType(Class<?> cls) {
		//return cls.getCanonicalName();
		return cls.getSimpleName();
	}
	
	protected Object createObject(Object objCls) throws Exception{
		Assert.notNull(objCls, "传入对象类型无效");
		if(objCls instanceof Class) {
			 return ((Class)objCls).newInstance();
		}
		else {
			return Class.forName(objCls.toString()).newInstance();
		}
	}
	
}
