package net.ibizsys.model.engine;

import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import net.ibizsys.model.IPSModelObject;
import net.ibizsys.model.engine.util.IAction;

public abstract class PSModelEngineBase<T extends IPSModelObject> implements IPSModelEngine<T>{

	private IPSModelEngineHolder iPSModelEngineHolder = null;
	private T iPSModelObject  = null;
	private String configFolder = null;
	private String id = null;
	
	@Override
	public void init(IPSModelEngineHolder iPSModelEngineHolder, String id, T iPSModelObject) throws Exception {
		Assert.notNull(iPSModelEngineHolder, "模型引擎上下文对象无效");
		Assert.hasLength(id, "模型引擎标识无效");
		Assert.notNull(iPSModelObject, "模型对象无效");
		this.id = id;
		this.iPSModelEngineHolder = iPSModelEngineHolder;
		this.iPSModelObject = iPSModelObject;
		this.prepareSetting();
		this.onInit();		
	}
	
	protected void onInit() throws Exception {
		
	}
	
	/**
	 * 准备默认设置
	 * @throws Exception
	 */
	protected void prepareSetting() throws Exception{
		this.onPrepareSetting();
	}
	
	protected void onPrepareSetting() throws Exception{
		
	}
	
	
	@Override
	public String getId() {
		return this.id;
	}
	
	@Override
	public String getName() {
		return this.getPSModelObject().getName();
	}
	
	@Override
	public String getFullName() {
		return this.getName();
	}
	
//	protected <T> T execute(IAction<T> iAction, Class<T> cls) {
//		return this.execute(iAction, cls, null);
//	}
//
//	protected <T> T execute(IAction<T> iAction, Class<T> cls, Object[] args) {
//		try {
//			return (T)iAction.execute(args);
//		} catch (Throwable e) {
//			String message = String.format("", args)
//		}
//	}
	
//	protected Object execute(String name, IAction<T> iAction) {
//		return this.execute(name, iAction, cls, null);
//	}
	
	protected Object executeAction(String name, IAction<Object> iAction) {
		return this.executeAction(name, iAction, null, Object.class);
	}
	
	protected Object executeAction(String name, IAction<Object> iAction, Object[] args) {
		return this.executeAction(name, iAction, args, Object.class);
	}
	
	protected <T> T executeAction(String name, IAction<T> iAction, Class<T> cls) {
		return this.executeAction(name, iAction, null, cls);
	}
	
	protected <T> T executeAction(String name, IAction<T> iAction, Object[] args, Class<T> cls) {
		try {
			return (T)iAction.execute(args);
		} catch (Throwable ex) {
			PSModelEngineException.rethrow(this, ex);
			if(StringUtils.hasLength(name)) {
				String message = String.format("%1$s发生错误，%2$s", name, ex.getMessage());
				throw new PSModelEngineException(this, message, ex);
			}
			throw new PSModelEngineException(this, ex.getMessage(), ex);
		}
	}

//	//@Override
//	public Object execute(Object[] args) throws Throwable {
//		return onExecute(args);
//	}
//	
//	protected Object onExecute(Object[] args) throws Throwable {
//		throw new Exception("没有实现");
//	}

	@Override
	public T getPSModelObject() {
		return this.iPSModelObject;
	}
	
	@Override
	public IPSModelEngineHolder getPSModelEngineHolder() {
		return this.iPSModelEngineHolder;
	}

	protected void setConfigFolder(String configFolder) {
		this.configFolder = configFolder;
	}
	
	
	/**
	 * 获取模型的运行时配置目录
	 * @return
	 */
	protected String getConfigFolder() {
		return this.configFolder;
	}
	
	
	protected String getLogCat() {
		return getFullName();
	}
	
	protected IPSSystemLogger getLogger(){
		return this.getPSModelEngineHolder().getPSSystemLogger();
	}
	
	protected IPSSystemUtil getUtil(){
		return this.getPSModelEngineHolder().getPSSystemUtil();
	}
}
