package net.ibizsys.central.cloud.oss.core.spring.controller;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.fasterxml.jackson.databind.JsonNode;

import net.ibizsys.central.SystemGatewayException;
import net.ibizsys.central.cloud.core.IServiceHub;
import net.ibizsys.central.cloud.core.cloudutil.ICloudOSSUtilRuntime;
import net.ibizsys.central.cloud.core.cloudutil.ICloudUtilRuntime;
import net.ibizsys.central.cloud.core.security.AuthenticationUser;
import net.ibizsys.central.cloud.core.util.RestUtils;
import net.ibizsys.central.cloud.oss.core.IOSSUtilSystemRuntime;
import net.ibizsys.central.cloud.oss.core.cloudutil.ISimpleFileStorageService;
import net.ibizsys.central.cloud.oss.core.util.domain.FileItem;

@RestController()
@RequestMapping("")
public class OSSRestController {

	private static final Log log = LogFactory.getLog(OSSRestController.class);

	@Autowired
	IServiceHub iServiceHub;
	
	@Value("${ibiz.cloud.oss.ignoreauth:true}")
	private boolean ignoreAuth;
	
	@PostConstruct
	protected void postConstruct() {
		log.debug(String.format("CloudOSS服务已经启动"));
		
		if(this.ignoreAuth) {
			iServiceHub.registerIgnoreAuthPattern("/ibizutil/download/**");
		}
		
		
		iServiceHub.registerNamingService(ICloudUtilRuntime.CLOUDSERVICEURL_OSS);
		if(this.iCloudOSSUtilRuntime == null) {
			//通过服务网关需要注册OSS功能组件
			iServiceHub.requireCloudUtilRuntime(IOSSUtilSystemRuntime.class, ICloudOSSUtilRuntime.class);
		}
	}
	

	@Autowired(required=false)
	private ICloudOSSUtilRuntime iCloudOSSUtilRuntime = null;

	protected ICloudOSSUtilRuntime getCloudOSSUtilRuntime() {
		if (this.iCloudOSSUtilRuntime == null) {
			try {
				this.iCloudOSSUtilRuntime = iServiceHub.getCloudUtilRuntime(IOSSUtilSystemRuntime.class, ICloudOSSUtilRuntime.class, false);
			}
			catch(Throwable ex) {
				log.debug(String.format("未指定Cloud服务OSS功能模块"), ex);
				throw new SystemGatewayException(iServiceHub, String.format("未指定Cloud服务OSS功能模块"), ex);
			}
		}
		return this.iCloudOSSUtilRuntime;
	}

	private ISimpleFileStorageService iSimpleFileStorageService = null;
	protected ISimpleFileStorageService getSimpleFileStorageService() {
		if (this.iSimpleFileStorageService == null) {
			if(this.getCloudOSSUtilRuntime() instanceof ISimpleFileStorageService) {
				this.iSimpleFileStorageService = (ISimpleFileStorageService)this.getCloudOSSUtilRuntime();
			}
			if (this.iSimpleFileStorageService == null) {
				throw new SystemGatewayException(iServiceHub, String.format("Cloud服务OSS功能模块未提供简单文件存储服务功能"));
			}
		}
		return this.iSimpleFileStorageService;
	}
	
	@RequestMapping(method = RequestMethod.POST, value = {"/oss/object/{db}/{table}/create", "/oss/object/{db}/{table}"})
	public ResponseEntity<String> createObject(@PathVariable("db") String db, @PathVariable("table") String table, @RequestBody Map params) {
		if(!AuthenticationUser.isApiuser(AuthenticationUser.getCurrentMust()) && !AuthenticationUser.isSuperuser(AuthenticationUser.getCurrentMust())) {
			throw new RuntimeException("仅限API用户访问");
		}
		return ResponseEntity.ok(this.getCloudOSSUtilRuntime().createObject(db, table, params));
	}
	
	
	@RequestMapping(method = RequestMethod.PUT, value = {"/oss/object/{db}/{table}/{key}/update", "/oss/object/{db}/{table}/{key}"})
	public ResponseEntity<Integer> updateObject(@PathVariable("db") String db, @PathVariable("table") String table, @PathVariable("key") String key, @RequestBody Map params) {
		if(!AuthenticationUser.isApiuser(AuthenticationUser.getCurrentMust()) && !AuthenticationUser.isSuperuser(AuthenticationUser.getCurrentMust())) {
			throw new RuntimeException("仅限API用户访问");
		}
		return ResponseEntity.ok(this.getCloudOSSUtilRuntime().updateObject(db, table, key, params, true));
	}

	@RequestMapping(method = RequestMethod.DELETE, value = {"/oss/object/{db}/{table}/{key}/delete", "/oss/object/{db}/{table}/{key}"})
	public ResponseEntity<Integer> deleteObject(@PathVariable("db") String db, @PathVariable("table") String table, @PathVariable("key") String key) {
		if(!AuthenticationUser.isApiuser(AuthenticationUser.getCurrentMust()) && !AuthenticationUser.isSuperuser(AuthenticationUser.getCurrentMust())) {
			throw new RuntimeException("仅限API用户访问");
		}
		return ResponseEntity.ok(this.getCloudOSSUtilRuntime().deleteObject(db, table, key));
	}
	
	@RequestMapping(method = RequestMethod.GET, value = {"/oss/object/{db}/{table}/{key}/get", "/oss/object/{db}/{table}/{key}"})
	public ResponseEntity<Map> getObject(@PathVariable("db") String db, @PathVariable("table") String table, @PathVariable("key") String key) {
		if(!AuthenticationUser.isApiuser(AuthenticationUser.getCurrentMust()) && !AuthenticationUser.isSuperuser(AuthenticationUser.getCurrentMust())) {
			throw new RuntimeException("仅限API用户访问");
		}
		return ResponseEntity.ok(this.getCloudOSSUtilRuntime().getObject(db, table, key));
	}
	
	
	@RequestMapping(method = RequestMethod.POST, value = {"/oss/object/{db}/{table}/fetch"})
	public ResponseEntity<Object> fetchObjects(@PathVariable("db") String db, @PathVariable("table") String table, @RequestBody Map params) {
		if(!AuthenticationUser.isApiuser(AuthenticationUser.getCurrentMust()) && !AuthenticationUser.isSuperuser(AuthenticationUser.getCurrentMust())) {
			throw new RuntimeException("仅限API用户访问");
		}
		
		Page page = this.getCloudOSSUtilRuntime().fetchObjects(db, table, params);
		return RestUtils.sendBackPage(page, Map.class);
	}
	

//	@Autowired
//	private FileService fileService;
//                                              
	@PostMapping(value = "${ibiz.cloud.oss.uploadpath:/ibizutil/upload}")
	public ResponseEntity<FileItem> upload(@RequestParam("file") MultipartFile multipartFile){
		return ResponseEntity.ok().body(getSimpleFileStorageService().uploadFile(null, multipartFile));
	}

	@GetMapping(value = "${ibiz.cloud.oss.downloadpath:/ibizutil/download/{id}}")
	@ResponseStatus(HttpStatus.OK)
	public void download(@PathVariable String id, HttpServletResponse response){
//		File file= getSimpleFileStorageService().getFile(null, id);
//		response.setHeader("Content-Disposition", "attachment;filename="+getFileName(file.getName()));
//		this.sendResponse(response, file);
		
		this.getSimpleFileStorageService().donwloadFile(null, id, response);
	}
	
	@PostMapping(value = "${ibiz.cloud.oss.uploadpath2:/ibizutil/upload/{cat}}")
	public ResponseEntity<FileItem> upload(@PathVariable String cat, @RequestParam("file") MultipartFile multipartFile){
		return ResponseEntity.ok().body(getSimpleFileStorageService().uploadFile(cat, multipartFile));
	}

	@GetMapping(value = "${ibiz.cloud.oss.downloadpath2:/ibizutil/download/{cat}/{id}}")
	@ResponseStatus(HttpStatus.OK)
	public void download(@PathVariable String cat, @PathVariable String id, HttpServletResponse response){
//		File file= getSimpleFileStorageService().getFile(cat, id);
//		response.setHeader("Content-Disposition", "attachment;filename="+getFileName(file.getName()));
//		this.sendResponse(response, file);
		this.getSimpleFileStorageService().donwloadFile(cat, id, response);
	}
	
	@PostMapping(value = "${ibiz.cloud.oss.downloadpath3:/ibizutil/download}")
	@ResponseStatus(HttpStatus.OK)
	public void download(@RequestBody List<JsonNode> list, HttpServletResponse response, @RequestParam(value = "packmode", required = false, defaultValue=ISimpleFileStorageService.PACKMODE_DEFAULT) String packmode){
		File file= getSimpleFileStorageService().getFile(null, list, packmode);
		response.setHeader("Content-Disposition", "attachment;filename="+getFileName(file.getName()));
		this.sendResponse(response, file);
	}
	
	
	@PostMapping(value = "${ibiz.cloud.oss.downloadpath4:/ibizutil/download/{cat}}")
	@ResponseStatus(HttpStatus.OK)
	public void download(@PathVariable String cat, @RequestBody List<JsonNode> list, HttpServletResponse response, @RequestParam(value = "packmode", required = false, defaultValue=ISimpleFileStorageService.PACKMODE_DEFAULT) String packmode){
		File file= getSimpleFileStorageService().getFile(cat, list, packmode);
		response.setHeader("Content-Disposition", "attachment;filename="+getFileName(file.getName()));
		this.sendResponse(response, file);
	}
	
//
//
	protected void sendResponse(HttpServletResponse response, File file){
		BufferedInputStream bis = null;
		BufferedOutputStream bos = null;
		try {
			bis = new BufferedInputStream(new FileInputStream(file));
			bos = new BufferedOutputStream(response.getOutputStream());
			byte[] buff = new byte[2048];
			int bytesRead;
			while (-1 != (bytesRead = bis.read(buff, 0, buff.length))) {
				bos.write(buff, 0, bytesRead);
			}
		}
		catch (Exception e) {
			log.error(e);
		}
		finally {
			if (bis != null) {
				try {
					bis.close();
				}
				catch (IOException e) {
					log.error(e);
				}
			}
			if (bos != null) {
				try {
					bos.close();
				}
				catch (IOException e) {
					log.error(e);
				}
			}
		}
	}
//
	protected String getFileName(String fileName){
		try {
			return new String(fileName.getBytes("utf-8"),"iso8859-1");//防止中文乱码
		}
		catch (UnsupportedEncodingException ex) {
			log.error(ex);
		}
		return fileName;
	}
}
