package net.ibizsys.central.cloud.core.spring.configuration;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.yaml.snakeyaml.Yaml;

import com.alibaba.cloud.nacos.NacosConfigManager;
import com.alibaba.cloud.nacos.NacosConfigProperties;
import com.alibaba.nacos.api.config.ConfigService;

import net.ibizsys.central.ISystemRuntimeSetting;
import net.ibizsys.central.cloud.core.ServiceHubSettingBase;
import net.ibizsys.central.cloud.core.util.ConfigEntity;
import net.ibizsys.central.cloud.core.util.NacosConfigUtils;
import net.ibizsys.central.cloud.core.util.domain.DataSource;
import net.ibizsys.central.cloud.core.util.domain.DeploySystem;
import net.ibizsys.runtime.util.DataTypeUtils;

/**
 * Nacos服务总线配置基类
 * @author lionlau
 *
 */
public abstract class NacosServiceHubSettingBase extends ServiceHubSettingBase {

	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(NacosServiceHubSettingBase.class);

	/**
	 * 数据项：服务总线前缀
	 */
	public final static String DATAID_SERVICEHUB_PREFIX = "servicehub-";
	
	/**
	 * 数据项：应用总线前缀
	 */
	public final static String DATAID_APPHUB_PREFIX = "apphub-";
	

	/**
	 * 数据项：部署系统前缀
	 */
	public final static String DATAID_DEPLOYSYSTEM_PREFIX = "deploysystem-";
	
	/**
	 * 数据项：部署应用前缀
	 */
	public final static String DATAID_DEPLOYAPP_PREFIX = "deployapp-";
	
	
	/**
	 * 数据项：数据库实例前缀
	 */
	public final static String DATAID_DBINST_PREFIX = "dbinst-";
	
	
	/**
	 * 数据项：大数据库实例前缀
	 */
	public final static String DATAID_BDINST_PREFIX = "bdinst-";
	
	
	
	@Autowired
	private NacosConfigManager nacosConfigManager;

	@Autowired
	private NacosConfigProperties nacosConfigProperties;


	private boolean bLoadConfig = false;

	public synchronized void loadConfig() {
		if (bLoadConfig) {
			return;
		}
		try {
			onLoadConfig();
			bLoadConfig = true;
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	protected void onLoadConfig() throws Exception {

		ConfigService configService = nacosConfigManager.getConfigService();
		
		// 本方法启动的时候获取内容
		boolean bLocal = false;
		String content = configService.getConfig(DATAID_SERVICEHUB_PREFIX + this.getId(), this.nacosConfigProperties.getGroup(), 5000);
		if(!StringUtils.hasLength(content)) {
			log.warn(String.format("无法获取远程配置[%1$s]，使用本地配置文件", DATAID_SERVICEHUB_PREFIX + this.getId()));
			//优先获取远程配置，没有则尝试从本地获取
			content = this.getLocalConfig();
			
			if(!StringUtils.hasLength(content)) {
				throw new Exception("未定义配置");
			}
			bLocal = true;
		}
		
		Yaml yaml = new Yaml();
		Map config = yaml.loadAs(content, Map.class);

		this.onLoadConfig(config, bLocal);
		
	}
	
	
	/**
	 * 加载配置
	 * @param config
	 * @param bLocal 配置是否来自本地定义
	 * @throws Exception
	 */
	protected void onLoadConfig(Map config, boolean bLocal) throws Exception {
		
		Object value = config.get("enableappgateway");
		if (value != null) {
			this.setEnableAppGateway(value.toString().equalsIgnoreCase("true"));
		}
		
		if(!StringUtils.hasLength(this.getTempFolder())) {
			value = config.get("tempfolder");
			if (value != null) {
				this.setTempFolder(value.toString());
			}
		}
		
		if(!StringUtils.hasLength(this.getFileFolder())) {
			value = config.get("filefolder");
			if (value != null) {
				this.setFileFolder(value.toString());
			}
		}
		
		if(!StringUtils.hasLength(this.getFontFolder())) {
			value = config.get("fontfolder");
			if (value != null) {
				this.setFontFolder(value.toString());
			}
		}
		
		if(!StringUtils.hasLength(this.getReportFolder())) {
			value = config.get("reportfolder");
			if (value != null) {
				this.setReportFolder(value.toString());
			}
		}
		
		if(!StringUtils.hasLength(this.getCloudPluginServiceId())) {
			value = config.get("cloudpluginserviceid");
			if (value != null) {
				this.setCloudPluginServiceId(value.toString());
			}
		}
		
		
		value = config.get("enablereloadsystem");
		if (value != null) {
			this.setEnableReloadSystem(DataTypeUtils.getBooleanValue(value, this.isEnableReloadSystem()));
		}
		
		
		if(this.getNode() == null) {
			value = config.get("node");
			if (value != null) {
				this.setNode(Long.valueOf(value.toString()));
			}
		}
		

		Map<String, Object> systemSettings = null;
		if (config.get("systemsettings") instanceof Map) {
			systemSettings = (Map) config.get("systemsettings");
		}
		
		
		ConfigService configService = nacosConfigManager.getConfigService();
		Yaml yaml = new Yaml();
		
		Object objDataSources = config.get("datasources");
		Map<String, DataSource> dataSourceMap = new HashMap<String, DataSource>();
		if (objDataSources instanceof Map) {
			List<DataSource> dataSourceList = new ArrayList<DataSource>();
			Map<String, Object> map = (Map<String, Object>) objDataSources;
			for (java.util.Map.Entry<String, Object> entry : map.entrySet()) {
				DataSource dataSource = new DataSource();
				Object objDataSource = entry.getValue();
				if(objDataSource instanceof Map) {
					
					Map<String, Object> source = (Map<String, Object>)objDataSource;
					dataSource.putAll(source);
					//进一步处理属性，这些属性将完全复制之数据源配置
					for (java.util.Map.Entry<String, Object> item : source.entrySet()) {
						String strKey = item.getKey().replace("-", "");
						if(strKey.equals(item.getKey())) {
							continue;
						}
						dataSource.set(strKey, item.getValue());
					}
					
					if (ObjectUtils.isEmpty(dataSource.getDriverClassName())) {
						dataSource.setDriverClassName((String) dataSource.get("driver-class-name"));
					}
				}
				else {
					String strDBInstConfigKey = String.format("%1$s%2$s", DATAID_DBINST_PREFIX, objDataSource);
					String strBackupConfigId = strDBInstConfigKey;
					strDBInstConfigKey = NacosConfigUtils.getDataId(strBackupConfigId);
					if(!strBackupConfigId.equals(strDBInstConfigKey)) {
						log.warn(String.format("键名[%1$s]包含非法字符，转换至[%2$s]，可能会出现键名冲突", strBackupConfigId, strDBInstConfigKey));
					}
					
					String strDBInstConfig = configService.getConfig(strDBInstConfigKey, this.nacosConfigProperties.getGroup(), 5000);
					if(!StringUtils.hasLength(strDBInstConfig)) {
						log.warn(String.format("未定义数据库实例[%1$s]配置，访问可能会出现问题", strDBInstConfigKey));
						continue;
					}
					
					ConfigEntity configEntity = new ConfigEntity(strDBInstConfig);
					String strDBInstConfigExKey = String.format("%1$s%2$s-ex", DATAID_DBINST_PREFIX, objDataSource);
					strDBInstConfigExKey = NacosConfigUtils.getDataId(strDBInstConfigExKey);
					String strDBInstConfigEx = configService.getConfig(strDBInstConfigExKey, this.nacosConfigProperties.getGroup(), 5000);
					if(StringUtils.hasLength(strDBInstConfigExKey)) {
						ConfigEntity configexEntity = new ConfigEntity(strDBInstConfigEx);
						configexEntity.copyTo(configEntity);
					}
					
					configEntity.copyTo(dataSource);
				}
				
				if (ObjectUtils.isEmpty(dataSource.getJdbcUrl())) {
					dataSource.setJdbcUrl((String) dataSource.get("url"));
				}
				dataSource.setDataSourceId(entry.getKey());
				
				dataSourceList.add(dataSource);
				
				dataSourceMap.put(dataSource.getDataSourceId(), dataSource);
			}
			this.setDataSources(dataSourceList);
		}
		

		Object objDeploySystems = config.get("deploysystems");
		if (objDeploySystems instanceof List) {
			List<DeploySystem> deploySystemList = new ArrayList<DeploySystem>();
			List list = (List) objDeploySystems;
			for (Object item : list) {
				if (ObjectUtils.isEmpty(item)) {
					continue;
				}
				
				//判断是否指定接口
				String strEnableAPIs = "";
				String strItem = item.toString();
				int nPos = strItem.indexOf(":");
				if(nPos != -1) {
					item = strItem.substring(0, nPos);
					strEnableAPIs = strItem.substring(nPos + 1);
				}

				String deploySystemConfig = configService.getConfig(DATAID_DEPLOYSYSTEM_PREFIX + item.toString(), this.nacosConfigProperties.getGroup(), 5000);
				if (!StringUtils.hasLength(deploySystemConfig)) {
					throw new Exception(String.format("部署系统[%1$s]配置无效", item.toString()));
				}
				Map map2 = yaml.loadAs(deploySystemConfig, Map.class);
				DeploySystem deploySystem = new DeploySystem();
				deploySystem.putAll(map2);
				deploySystem.setDeploySystemId(item.toString());
				
				
				// 合并设置
				if (systemSettings != null) {
					Map<String, Object> settings = new HashMap<String, Object>();
					settings.putAll(systemSettings);
					if (deploySystem.getSettings() != null) {
						settings.putAll(deploySystem.getSettings());
					}
					deploySystem.setSettings(settings);
				}
				
				//放入启动接口
				if(StringUtils.hasLength(strEnableAPIs)) {
					if(deploySystem.getSettings()==null) {
						deploySystem.setSettings(new HashMap<String, Object>());
					}
					deploySystem.getSettings().put(ISystemRuntimeSetting.PARAM_ENABLEAPIS, strEnableAPIs);
					//deploySystem.setAPIs(Arrays.asList(strInterfaces.split("[;]")));
				}
				
				//判断是否存在默认数据源
				String strDefaultDBInstTag = String.format("%1$s__default", deploySystem.getDeploySystemId());
				if(dataSourceMap.containsKey(strDefaultDBInstTag)) {
					if(deploySystem.getSettings()==null) {
						deploySystem.setSettings(new HashMap<String, Object>());
					}
					deploySystem.getSettings().put(ISystemRuntimeSetting.PARAM_DEFAULTDBINSTTAG, strDefaultDBInstTag);
				}
				
				
				
				deploySystemList.add(deploySystem);
			}
			this.setDeploySystems(deploySystemList);
		}
		
	}

	
	/**
	 * 获取本地配置
	 * @return
	 * @throws Exception
	 */
	protected String getLocalConfig() throws Exception{
		//优先获取远程配置，没有则尝试从本地获取
		try(InputStream is = this.getClass().getResourceAsStream("/servicehub.yml")){
			if(is != null) {
				return IOUtils.toString(is, "UTF-8");
			}
		}
		
		return null;
	}
	
}
