package net.ibizsys.central.cloud.core;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;

import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.client.loadbalancer.reactive.ReactorLoadBalancerExchangeFilterFunction;
import org.springframework.core.io.Resource;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.PathMatcher;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;
import org.yaml.snakeyaml.Yaml;

import com.alibaba.cloud.nacos.NacosConfigManager;
import com.alibaba.cloud.nacos.NacosConfigProperties;
import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.nacos.api.config.ConfigService;
import com.alibaba.nacos.api.config.listener.Listener;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.fasterxml.jackson.databind.node.ObjectNode;

import net.ibizsys.central.ISystemRuntime;
import net.ibizsys.central.SystemGateway;
import net.ibizsys.central.SystemGatewayException;
import net.ibizsys.central.Version;
import net.ibizsys.central.app.IApplicationRuntime;
import net.ibizsys.central.cloud.core.app.IProxyAppRuntime;
import net.ibizsys.central.cloud.core.app.IServletAppRuntime;
import net.ibizsys.central.cloud.core.cloudutil.ICloudDevOpsUtilRuntime;
import net.ibizsys.central.cloud.core.cloudutil.ICloudUtilRuntime;
import net.ibizsys.central.cloud.core.cloudutil.client.ICloudDevOpsClient;
import net.ibizsys.central.cloud.core.database.ISysDBSchemeSyncAdapter;
import net.ibizsys.central.cloud.core.security.AuthenticationUser;
import net.ibizsys.central.cloud.core.security.IAuthenticationUser;
import net.ibizsys.central.cloud.core.service.ISysServiceAPIDocAdapter;
import net.ibizsys.central.cloud.core.service.ISysServiceAPIRequestMappingAdapter;
import net.ibizsys.central.cloud.core.service.StandardSysServiceAPIRequestMappingAdapter;
import net.ibizsys.central.cloud.core.servlet.IServiceHubFilter;
import net.ibizsys.central.cloud.core.spring.configuration.NacosServiceHubSetting;
import net.ibizsys.central.cloud.core.spring.configuration.NacosServiceHubSettingBase;
import net.ibizsys.central.cloud.core.spring.util.GatewayUtils;
import net.ibizsys.central.cloud.core.sysutil.IHubSysDevOpsUtilRuntime;
import net.ibizsys.central.cloud.core.sysutil.ISysCloudClientUtilRuntime;
import net.ibizsys.central.cloud.core.sysutil.ISysDevOpsUtilRuntime;
import net.ibizsys.central.cloud.core.sysutil.ISysUtilContainerOnly;
import net.ibizsys.central.cloud.core.util.ConfigEntity;
import net.ibizsys.central.cloud.core.util.IConfigListener;
import net.ibizsys.central.cloud.core.util.NacosConfigUtils;
import net.ibizsys.central.cloud.core.util.domain.AppData;
import net.ibizsys.central.cloud.core.util.domain.DataSource;
import net.ibizsys.central.cloud.core.util.domain.DeploySystem;
import net.ibizsys.central.cloud.core.util.domain.ServiceInstance;
import net.ibizsys.central.database.ISysDBSchemeRuntime;
import net.ibizsys.central.plugin.spring.service.client.WebFluxClient;
import net.ibizsys.central.service.ISysServiceAPIRuntime;
import net.ibizsys.model.IPSSystemService;
import net.ibizsys.model.PSModelServiceImpl;
import net.ibizsys.model.app.IPSApplication;
import net.ibizsys.model.database.IPSSysDBScheme;
import net.ibizsys.model.service.IPSSysServiceAPI;
import net.ibizsys.runtime.res.SysSFPluginRuntime;
import net.ibizsys.runtime.security.IUserContext;
import net.ibizsys.runtime.security.UserContext;
import net.ibizsys.runtime.sysutil.ISysFileUtilRuntime;
import net.ibizsys.runtime.util.DataTypeUtils;
import net.ibizsys.runtime.util.DataTypes;
import net.ibizsys.runtime.util.Errors;
import net.ibizsys.runtime.util.ZipUtils;

public abstract class ServiceHubBase extends SystemGateway implements IServiceHub, IServiceHubFilter {

	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(ServiceHubBase.class);
	private IServiceHubSetting iServiceHubSetting;

	private IServiceHubFilter realServiceHubFilter = null;
	private Set<String> ignoreAuthPatternSet = null;
	private PathMatcher pathMatcher = new AntPathMatcher();
	private Map<String, IServletAppRuntime> servletAppRuntimeMap = new ConcurrentHashMap<String, IServletAppRuntime>();

	private IHubSystemRuntime hubSystemRuntime = null;

	private RequestMappingHandlerMapping requestMappingHandlerMapping = null;

	private ISysServiceAPIDocAdapter sysServiceAPIDocAdapter = null;

	private ISysDBSchemeSyncAdapter sysDBSchemeSyncAdapter = null;

	@Autowired
	private NacosServiceManager nacosServiceManager;

	@Autowired
	private NacosDiscoveryProperties nacosDiscoveryProperties;

	@Autowired
	private NacosConfigManager nacosConfigManager;

	@Autowired
	private NacosConfigProperties nacosConfigProperties;

	@Autowired
	private ReactorLoadBalancerExchangeFilterFunction reactorLoadBalancerExchangeFilterFunction;

	private NamingService namingService = null;

	private String strIPAddress = null;

	private Map<Class<? extends ICloudUtilRuntime>, Class<?>> requiredCloudUtilRuntimeMap = new LinkedHashMap<Class<? extends ICloudUtilRuntime>, Class<?>>();

	private Map<IConfigListener, Listener> configListenerMap = new ConcurrentHashMap<IConfigListener, Listener>();

	private String strDeploySystemModelPath = null;

	private boolean bInstalled = false;

	private ISysServiceAPIRequestMappingAdapter iSysServiceAPIRequestMappingAdapter = null;
	
	private ICloudPluginService cloudPluginService = null;
	
	private Map<String, String> deploySystemVerMap = new ConcurrentHashMap<String, String>();
	
	

	/**
	 * 数据库实例配置Map
	 */
	private Map<String, ConfigEntity> dbInstConfigMap = new HashMap<String, ConfigEntity>();

	private Map<String, DataSource> dataSourceMap = new HashMap<String, DataSource>();

	@Value("${server.port}")
	private int port;

	// private Map<Class, Object> sysUtilRuntimeMap = new
	// ConcurrentHashMap<Class, Object>();

	public static IServiceHub getInstance() {
		return (IServiceHub) getInstance(false);
	}

	public ServiceHubBase() {
		if (getInstance(true) == null) {
			setInstance(this);
		}
	}

	protected void onPostConstruct() {

		WebFluxClient.setExchangeFilterFunction(reactorLoadBalancerExchangeFilterFunction);
		this.namingService = nacosServiceManager.getNamingService(nacosDiscoveryProperties.getNacosProperties());
		this.strIPAddress = nacosDiscoveryProperties.getIp();
		if (!StringUtils.hasLength(this.strIPAddress)) {
			try {
				this.strIPAddress = InetAddress.getLocalHost().getHostAddress();
			} catch (UnknownHostException e) {
				log.debug(e);
			}
		}
		log.info(String.format("服务网关版本[%1$s]", Version.toVersionString()));
		log.debug(String.format("服务网关开始初始化[%1$s:%2$s]", this.strIPAddress, this.port));
	}

	public IServiceHubSetting getServiceHubSetting() {
		return this.iServiceHubSetting;
	}

	public void setServiceHubSetting(IServiceHubSetting iServiceHubSetting) {
		this.iServiceHubSetting = iServiceHubSetting;
	}

	protected RequestMappingHandlerMapping getRequestMappingHandlerMapping() {
		return this.requestMappingHandlerMapping;
	}

	protected void setRequestMappingHandlerMapping(RequestMappingHandlerMapping requestMappingHandlerMapping) {
		this.requestMappingHandlerMapping = requestMappingHandlerMapping;
	}

	protected ISysServiceAPIDocAdapter getSysServiceAPIDocAdapter() {
		return sysServiceAPIDocAdapter;
	}

	protected void setSysServiceAPIDocAdapter(ISysServiceAPIDocAdapter sysServiceAPIDocAdapter) {
		this.sysServiceAPIDocAdapter = sysServiceAPIDocAdapter;
	}

	protected ISysDBSchemeSyncAdapter getSysDBSchemeSyncAdapter() {
		return sysDBSchemeSyncAdapter;
	}

	protected void setSysDBSchemeSyncAdapter(ISysDBSchemeSyncAdapter sysDBSchemeSyncAdapter) {
		this.sysDBSchemeSyncAdapter = sysDBSchemeSyncAdapter;
	}

	public void install() {
		try {
			if (getServiceHubSetting() == null) {
				throw new Exception("无效的配置对象");
			}

			log.debug(String.format("服务总线[%1$s]开始安装部署", getServiceHubSetting().getId()));
			onInstall();
			this.bInstalled = true;

		} catch (Exception ex) {
			throw new SystemGatewayException(this, String.format("安装服务网关发生异常，%1$s", ex.getMessage()), ex);
		}
	}

	protected void onInstall() throws Exception {

		registerIgnoreAuthPatterns();

		this.setEnableAppGateway(this.getServiceHubSetting().isEnableAppGateway());
		this.setTempFolder(this.getServiceHubSetting().getTempFolder());
		this.setFileFolder(this.getServiceHubSetting().getFileFolder());
		this.setReportFolder(this.getServiceHubSetting().getReportFolder());
		this.setFontFolder(this.getServiceHubSetting().getFontFolder());
		
		/**
		 * 准备Cloud环境插件服务
		 */
		this.prepareCloudPluginService();
		

		// 安装动态数据源
		List<DataSource> dataSourceList = this.getServiceHubSetting().getDataSources();
		if (dataSourceList != null) {
			for (DataSource ds : dataSourceList) {
				try {
					dataSourceMap.put(ds.getDataSourceId(), ds);
					this.registerDataSource(ds);
				} catch (Exception ex) {
					throw new Exception(String.format("注册数据源[%1$s]发生异常，%2$s", ds.getDataSourceId(), ex.getMessage()), ex);
				}
			}
		}

		if (this.getSysDBSchemeSyncAdapter() == null) {
			log.warn(String.format("未配置数据库模型同步适配器，无法同步数据库模型"));
		}

		List<DeploySystem> deploySystemList = this.getServiceHubSetting().getDeploySystems();
		if (deploySystemList != null) {
			for (DeploySystem deploySystem : deploySystemList) {
				try {
					registerDeploySystem(deploySystem);
				} catch (Exception ex) {
					throw new Exception(String.format("注册系统[%1$s]发生异常，%2$s", deploySystem.getDeploySystemId(), ex.getMessage()), ex);
				}
			}

			if (this.getServiceHubSetting().isEnableReloadSystem() && !this.isEnableDebug()) {
				for (DeploySystem deploySystem : deploySystemList) {

					if (DEPLOYSYSYTEMID_GATEWAY.equalsIgnoreCase(deploySystem.getDeploySystemId())) {
						continue;
					}

					if (!DataTypeUtils.getBooleanValue(deploySystem.getEnableReload(), true)) {
						continue;
					}

					String strConfigId = String.format("%1$s%2$s-ver", NacosServiceHubSettingBase.DATAID_DEPLOYSYSTEM_PREFIX, deploySystem.getDeploySystemId());
					String strConfig = this.getConfig(strConfigId);
					if(strConfig == null) {
						strConfig = "";
					}
					this.deploySystemVerMap.put(strConfigId, strConfig);
					this.addConfigListener(strConfigId, new IConfigListener() {

						@Override
						public Executor getExecutor() {
							return null;
						}

						@Override
						public void receiveConfigInfo(String configInfo) {
							
							String strConfig = deploySystemVerMap.get(strConfigId);
							if(configInfo == null) {
								configInfo = "";
							}
							if(configInfo.equals(strConfig)) {
								log.debug(String.format("部署系统[%1$s]接收到版本变化，但配置内容一致，忽略", deploySystem.getDeploySystemId()));
								return;
							}
							deploySystemVerMap.put(strConfigId, configInfo);
							log.debug(String.format("部署系统[%1$s]版本发生变化", deploySystem.getDeploySystemId()));
							if (isInstalled()) {
								reloadDeploySystem(deploySystem, configInfo);
								
							} else {
								log.warn(String.format("服务总线未安装完成，忽略部署系统版本变化"));
							}
						}
					});
				}
			}
		}

		// 部署需要的插件
		if (!ObjectUtils.isEmpty(this.requiredCloudUtilRuntimeMap)) {
			for (java.util.Map.Entry<Class<? extends ICloudUtilRuntime>, Class<?>> entry : this.requiredCloudUtilRuntimeMap.entrySet()) {
				this.getSysUtilRuntime(entry.getValue(), entry.getKey(), false);
			}
		}
		
		if(this.isEnableDebug() && this.getServiceHubSetting().isStartDebugger()) {
			this.startCloudDebugger();
		}
	}
	
	/**
	 * 准备Cloud环境插件服务
	 * @throws Exception
	 */
	protected void prepareCloudPluginService() throws Exception{
		String cloudPluginServiceId = this.getServiceHubSetting().getCloudPluginServiceId();
		if(!StringUtils.hasLength(cloudPluginServiceId)) {
			log.warn(String.format("未指定Cloud插件服务标识，未启用插件服务"));
			return;
		}
		
		DeploySystem deploySystem = new DeploySystem();
		deploySystem.setDeploySystemId(cloudPluginServiceId);
		deploySystem.setModelPath(DEPLOYSYSYTEMMODELPATH_REMOTE);
		IPSSystemService iPSSystemService = null;
		try {
			iPSSystemService = this.getPSSystemService(deploySystem, true);
		} catch (Exception ex) {
			log.error(String.format("获取Cloud插件服务发生异常，%1$s", ex.getMessage()), ex);
		}
		
		if(iPSSystemService == null) {
			log.warn(String.format("Cloud插件服务[%1$s]无效，未启用插件服务", cloudPluginServiceId));
			return;
		}
		
		ICloudPluginService cloudPluginService = this.createCloudPluginService();
		cloudPluginService.init(iPSSystemService);
		this.setCloudPluginService(cloudPluginService);
	}
	
	protected ICloudPluginService createCloudPluginService() {
		return new CloudPluginService();
	}
	

	/**
	 * 启动Debugger
	 * @throws Exception
	 */
	protected void startCloudDebugger() throws Exception {
		
		String strDebuggerCmd = this.getServiceHubSetting().getDebuggerCmd();
		if(!StringUtils.hasLength(strDebuggerCmd)) {
			//自动构建mingl
			String strDebugTag = String.format("%1$s_%2$s", this.strIPAddress, this.port).replace(".", "_");
			
			strDebuggerCmd = String.format("java -Dibiz.debug.config.debugIpAddr=127.0.0.1:12345 -Dibiz.debug.mqttmessenger.serviceUrl=tcp://mqtt.ibizcloud.cn:31883 -Dibiz.debug.mqttmessenger.debugSystemTag=%1$s -jar /ibiztools/clouddebugger.jar", strDebugTag);
			log.debug(String.format("CloudDebugger command: %1$s", strDebuggerCmd));
			
		}
		
		Runtime.getRuntime().exec(strDebuggerCmd);
		log.info("CloudDebugger启动成功");
	}
	
	@Override
	public boolean isInstalled() {
		return this.bInstalled;
	}


	@Override
	public boolean isEnableProxyApp() {
		return this.getServiceHubSetting().isEnableProxyApp();
	}
	

	@Override
	public boolean isEnableServletApp() {
		return this.getServiceHubSetting().isEnableServletApp();
	}
	

	@Override
	public boolean isEnableDebug() {
		if(isEnableProdMode()) {
			return false;
		}
		return this.getServiceHubSetting().isEnableDebug();
	}

	@Override
	public boolean isEnableRTCodeMode() {
		return this.getServiceHubSetting().isEnableRTCodeMode();
	}
	
	@Override
	public boolean isEnableProdMode() {
		return this.getServiceHubSetting().isEnableProdMode();
	}
	
	@Override
	public int getWorkThreadCorePoolSize() {
		return this.getServiceHubSetting().getWorkThreadCorePoolSize();
	}

	@Override
	public int getWorkThreadMaximumPoolSize() {
		return this.getServiceHubSetting().getWorkThreadMaximumPoolSize();
	}

	@Override
	public int getWorkThreadBlockingQueueSize() {
		return this.getServiceHubSetting().getWorkThreadBlockingQueueSize();
	}

	@Override
	public String getServletAppBaseUrl() {
		return BASEURL_SERVLETAPP;
	}

	@Override
	public String getProxyAppBaseUrl() {
		return BASEURL_PROXYAPP;
	}
	
	@Override
	public ICloudPluginService getCloudPluginService() {
		return this.cloudPluginService;
	}
	
	
	/**
	 * 设置Cloud插件服务对象
	 * @param cloudPluginService
	 */
	protected void setCloudPluginService(ICloudPluginService cloudPluginService) {
		this.cloudPluginService = cloudPluginService;
	}

	/**
	 * 准备忽略认证的路径集合
	 * 
	 * @throws Exception
	 */
	protected void registerIgnoreAuthPatterns() throws Exception {
		this.registerIgnoreAuthPattern("/swagger-ui/**");
		this.registerIgnoreAuthPattern("/swagger**/**");
		this.registerIgnoreAuthPattern("/webjars/**");
		this.registerIgnoreAuthPattern("/v3/**");
		if (isEnableServletApp()) {
			this.registerIgnoreAuthPattern(String.format("/%1$s/**", getServletAppBaseUrl()));
		}

	}

	protected javax.sql.DataSource registerDataSource(DataSource ds) throws Exception {
		throw new Exception("未提供注册动态数据源能力");
	}

	protected ISystemRuntime registerDeploySystem(DeploySystem deploySystem) throws Exception {

		IPSSystemService iPSSystemService = this.getPSSystemService(deploySystem);
		if (iPSSystemService == null) {
			String strModelPath = deploySystem.getModelPath();
			if (StringUtils.hasLength(strModelPath)) {
				PSModelServiceImpl psModelServiceImpl = new PSModelServiceImpl();
				psModelServiceImpl.setPSModelFolderPath(strModelPath);
				iPSSystemService = psModelServiceImpl;
			}
		}

		String strRuntimeClassName = deploySystem.getRuntimeClassName();
		if (!StringUtils.hasLength(strRuntimeClassName) && iPSSystemService != null) {
			strRuntimeClassName = iPSSystemService.getPSSystem().getRTObjectName();
		}

		ISystemRuntime iSystemRuntime = null;
		if (StringUtils.hasLength(strRuntimeClassName)) {
			iSystemRuntime = (ISystemRuntime) SysSFPluginRuntime.createObject(strRuntimeClassName);
		} else {
			if (DEPLOYSYSYTEMID_GATEWAY.equalsIgnoreCase(deploySystem.getDeploySystemId())) {
				iSystemRuntime = new HubSystemRuntime();
			} else {
				iSystemRuntime = new ServiceSystemRuntime();
			}
		}

		iSystemRuntime.init(getSystemGatewayContext(), iPSSystemService, deploySystem.getDeploySystemId(), deploySystem.getSettings());
		this.registerSystemRuntime(deploySystem, iSystemRuntime);

		return iSystemRuntime;
	}

	protected String getDeploySystemModelPath() {
		if (!StringUtils.hasLength(this.strDeploySystemModelPath)) {
			String strPath = String.format("%1$s%2$sdeploysystem", this.getFileFolder(), File.separator);
			File folder = new File(strPath);
			if (!folder.exists()) {
				folder.mkdirs();
			}
			this.strDeploySystemModelPath = strPath;
		}
		return this.strDeploySystemModelPath;
	}

	protected IPSSystemService getPSSystemService(DeploySystem deploySystem) throws Exception {
		return this.getPSSystemService(deploySystem, false);
	}

	protected IPSSystemService getPSSystemService(DeploySystem deploySystem, boolean tryMode) throws Exception {

		if (DEPLOYSYSYTEMID_GATEWAY.equalsIgnoreCase(deploySystem.getDeploySystemId())) {
			return null;
		}

		String strModelPath = deploySystem.getModelPath();
		if (!DEPLOYSYSYTEMMODELPATH_REMOTE.equals(strModelPath)) {
			return null;
		}

		// 获取系统模型路径
		ISysCloudClientUtilRuntime iSysCloudClientRuntime = this.getHubSystemRuntime(false).getSysUtilRuntime(ISysCloudClientUtilRuntime.class, false);
		ICloudDevOpsClient iCloudDevOpsClient = iSysCloudClientRuntime.getServiceClient(ICloudUtilRuntime.CLOUDSERVICE_DEVOPS).getProxyClient(ICloudDevOpsClient.class);
		String strModelDigest = iCloudDevOpsClient.getSystemModelDigest(deploySystem.getDeploySystemId());

		if (!StringUtils.hasLength(strModelDigest)) {
			if (tryMode) {
				return null;
			}
			throw new Exception(String.format("未定义远程模型配置"));
		}

		String[] items = strModelDigest.split("[|]");
		if (items.length < 2) {
			throw new Exception(String.format("远程模型配置不正确"));
		}

		String strFileId = items[1];
		// 判断本地路径是否存在
		String strDynaModelPath = String.format("%1$s%2$s%3$s", this.getDeploySystemModelPath(), File.separator, strFileId);

		File systemModelFile = new File(strDynaModelPath + File.separator + "PSSYSTEM.json");
		if (!systemModelFile.exists()) {

			ISysFileUtilRuntime iSysFileUtilRuntime = this.getHubSystemRuntime(false).getSysUtilRuntime(ISysFileUtilRuntime.class, false);

			net.ibizsys.runtime.util.domain.File ossFile = iSysFileUtilRuntime.getOSSFile(strFileId, ICloudDevOpsUtilRuntime.OSSCAT_DYNAMODEL, true);
			if (ossFile == null) {
				ossFile = iSysFileUtilRuntime.getOSSFile(strFileId, null, true);
				if (ossFile == null) {
					throw new Exception(String.format("指定模型文件[%1$s]不存在", strFileId));
				}
			}
			File dynaModelFolder = new File(strDynaModelPath);
			if (!systemModelFile.exists()) {
				ZipUtils.unzip(new File(ossFile.getLocalPath()), dynaModelFolder);
			}
		}

		log.debug(String.format("系统[%1$s]使用远程模型[%2$s][%3$s]", deploySystem.getDeploySystemId(), strModelDigest, strDynaModelPath));

		PSModelServiceImpl psModelServiceImpl = new PSModelServiceImpl();
		psModelServiceImpl.setPSModelFolderPath(strDynaModelPath);
		psModelServiceImpl.setModelDigest(items[0]);
		return psModelServiceImpl;
	}

	protected void registerSystemRuntime(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {
		String strDeploySystemId = deploySystem.getDeploySystemId();
		this.registerSystemRuntime(strDeploySystemId, iSystemRuntime);
		if (!DEPLOYSYSYTEMID_GATEWAY.equalsIgnoreCase(strDeploySystemId)) {

			boolean bEnableServiceHubAPI = true;
			if (iSystemRuntime instanceof IServiceSystemRuntime) {
				bEnableServiceHubAPI = ((IServiceSystemRuntime) iSystemRuntime).isEnableServiceHubAPI();
			}

			if (bEnableServiceHubAPI) {
				String strServiceName = String.format("%1$s-%2$s", "servicehub", iSystemRuntime.getDeploySystemId());
				this.registerNamingService(strServiceName);
			}
		}

		// 注册数据源
		registerDataSources(deploySystem, iSystemRuntime);
		// 注册服务
		registerServiceAPIRequestMapping(deploySystem, iSystemRuntime);
		// 注册应用
		if (isEnableServletApp()) {
			registerServletAppRequestMapping(deploySystem, iSystemRuntime);
		}
		if (isEnableProxyApp()) {
			registerProxyAppRequestMapping(deploySystem, iSystemRuntime);
		}
		
		// 注册应用网关路由
		registerAppGatewayRoutes(deploySystem, iSystemRuntime);

	}

	@Override
	public void registerSystemRuntime(String strDeploySystemId, ISystemRuntime iSystemRuntime) throws Exception {
		super.registerSystemRuntime(strDeploySystemId, iSystemRuntime);
		if (this.getServiceHubFilter() == null) {
			this.setServiceHubFilter(iSystemRuntime.getSysUtilRuntime(IServiceHubFilter.class, true));
		}

		if (DEPLOYSYSYTEMID_GATEWAY.equalsIgnoreCase(strDeploySystemId)) {
			if (iSystemRuntime instanceof IHubSystemRuntime) {
				this.setHubSystemRuntime((IHubSystemRuntime) iSystemRuntime);
			}
		}
	}

	protected void registerServiceAPIRequestMapping(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {
		if (this.getRequestMappingHandlerMapping() == null) {
			log.warn(String.format("接口请求映射处理对象无效"));
			return;
		}

		java.util.List<IPSSysServiceAPI> psSysServiceAPIs = iSystemRuntime.getPSSystem().getAllPSSysServiceAPIs();
		if (psSysServiceAPIs != null) {
			for (IPSSysServiceAPI iPSSysServiceAPI : psSysServiceAPIs) {

				ISysServiceAPIRuntime iSysServiceAPIRuntime = iSystemRuntime.getSysServiceAPIRuntime(iPSSysServiceAPI.getCodeName(), true);
				if (iSysServiceAPIRuntime == null) {
					log.warn(String.format("系统[%1$s]服务接口[%2$s]运行时对象无效，忽略请求处理映射", deploySystem.getDeploySystemId(), iPSSysServiceAPI.getName()));
					continue;
				}

				ISysServiceAPIRequestMappingAdapter iSysServiceAPIRequestMappingAdapter = getSysServiceAPIRequestMappingAdapter(iSysServiceAPIRuntime);
				if (iSysServiceAPIRequestMappingAdapter == null) {
					continue;
				}

				if (iSysServiceAPIRuntime instanceof net.ibizsys.central.cloud.core.service.ISysServiceAPIRuntime) {
					((net.ibizsys.central.cloud.core.service.ISysServiceAPIRuntime) iSysServiceAPIRuntime).registerMapping(iSysServiceAPIRequestMappingAdapter);
				} else {
					iSysServiceAPIRequestMappingAdapter.registerMapping(iSysServiceAPIRuntime);
				}

				// 注册服务
				String strServiceName = String.format("%1$s-%2$s", iSystemRuntime.getDeploySystemId(), iPSSysServiceAPI.getCodeName()).toLowerCase();
				this.registerNamingService(strServiceName);
			}
		}
	}

	protected void registerServletAppRequestMapping(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {

		String prefix = String.format("/%1$s/", this.getServletAppBaseUrl());

		java.util.List<IPSApplication> psApplications = iSystemRuntime.getPSSystem().getAllPSApps();
		if (psApplications != null) {
			for (IPSApplication iPSApplication : psApplications) {

				IApplicationRuntime iApplicationRuntime = iSystemRuntime.getApplicationRuntime(iPSApplication.getCodeName(), true);
				if (iApplicationRuntime == null) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象无效，忽略请求处理映射", deploySystem.getDeploySystemId(), iPSApplication.getName()));
					continue;
				}

				if (!(iApplicationRuntime instanceof IServletAppRuntime)) {
					continue;
				}

				// log.info(String.format("系统[%1$s]前端应用[%2$s]运行时对象类型不支持，忽略请求处理映射",
				// deploySystem.getDeploySystemId(), iPSApplication.getName()));

				IServletAppRuntime iServletAppRuntime = (IServletAppRuntime) iApplicationRuntime;
				String url = iServletAppRuntime.getBaseUrl();
				if (url.indexOf(prefix) != 0) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]服务前缀[%3$s]不一致，忽略注册", deploySystem.getDeploySystemId(), iApplicationRuntime.getName(), url));
					continue;
				}

				url = url.substring(prefix.length());
				servletAppRuntimeMap.put(url + "/**", iServletAppRuntime);

				// 注册服务
				String strServiceName = String.format("%1$s-app-%2$s", iSystemRuntime.getDeploySystemId(), iPSApplication.getCodeName()).toLowerCase();
				this.registerNamingService(strServiceName);
			}
		}
	}

	protected void registerProxyAppRequestMapping(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {
		if (this.getRequestMappingHandlerMapping() == null) {
			log.warn(String.format("接口请求映射处理对象无效"));
			return;
		}

		String prefix = String.format("/%1$s/", this.getProxyAppBaseUrl());

		java.util.List<IPSApplication> psApplications = iSystemRuntime.getPSSystem().getAllPSApps();
		if (psApplications != null) {
			for (IPSApplication iPSApplication : psApplications) {

				IApplicationRuntime iApplicationRuntime = iSystemRuntime.getApplicationRuntime(iPSApplication.getCodeName(), true);
				if (iApplicationRuntime == null) {
					continue;
				}

				// log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象无效，忽略请求处理映射",
				// deploySystem.getDeploySystemId(), iPSApplication.getName()));

				if (!(iApplicationRuntime instanceof IProxyAppRuntime)) {
					// log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象类型不支持，忽略请求处理映射",
					// deploySystem.getDeploySystemId(),
					// iPSApplication.getName()));
					continue;
				}

				IProxyAppRuntime iProxyAppRuntime = (IProxyAppRuntime) iApplicationRuntime;
				String url = iProxyAppRuntime.getBaseUrl();
				if (url.indexOf(prefix) != 0) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]服务前缀[%3$s]不一致，忽略注册", deploySystem.getDeploySystemId(), iApplicationRuntime.getName(), url));
					continue;
				}

				ISysServiceAPIRequestMappingAdapter iSysServiceAPIRequestMappingAdapter = this.getApplicationRequestMappingAdapter(iProxyAppRuntime);
				iProxyAppRuntime.registerMapping(iSysServiceAPIRequestMappingAdapter);

				// 注册服务
				String strServiceName = String.format("%1$s-app-%2$s", iSystemRuntime.getDeploySystemId(), iPSApplication.getCodeName()).toLowerCase();
				this.registerNamingService(strServiceName);
			}
		}
	}


	
	protected void unregisterSystemRuntime(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {
		String strDeploySystemId = deploySystem.getDeploySystemId();
		if (DEPLOYSYSYTEMID_GATEWAY.equalsIgnoreCase(strDeploySystemId)) {
			throw new Exception(String.format("无法注销[%1$s]系统运行时", strDeploySystemId));
		}

		try {
			unregisterServiceAPIRequestMapping(deploySystem, iSystemRuntime);
		} catch (Throwable ex) {
			log.error(String.format("注销服务接口路径映射发生异常，%1$s", ex.getMessage()), ex);
		}

		try {
			if (this.isEnableServletApp()) {
				unregisterServletAppRequestMapping(deploySystem, iSystemRuntime);
			}
		} catch (Throwable ex) {
			log.error(String.format("注销前端应用路径映射发生异常，%1$s", ex.getMessage()), ex);
		}

		try {
			if (this.isEnableProxyApp()) {
				unregisterProxyAppRequestMapping(deploySystem, iSystemRuntime);
			}
		} catch (Throwable ex) {
			log.error(String.format("注销前端应用路径映射发生异常，%1$s", ex.getMessage()), ex);
		}


		try {
			boolean bEnableServiceHubAPI = true;
			if (iSystemRuntime instanceof IServiceSystemRuntime) {
				bEnableServiceHubAPI = ((IServiceSystemRuntime) iSystemRuntime).isEnableServiceHubAPI();
			}

			if (bEnableServiceHubAPI) {
				String strServiceName = String.format("%1$s-%2$s", "servicehub", iSystemRuntime.getDeploySystemId());
				this.unregisterNamingService(strServiceName);
			}
		} catch (Throwable ex) {
			log.error(String.format("注销系统命名服务发生异常，%1$s", ex.getMessage()), ex);
		}

		this.unregisterSystemRuntime(strDeploySystemId, iSystemRuntime);

		// 注销数据源
		// unregisterDataSources(deploySystem, iSystemRuntime);
		// 注销服务

		// 注销应用网关路由
		// unregisterAppGatewayRoutes(deploySystem, iSystemRuntime);

	}

	protected void unregisterSystemRuntime(String strDeploySystemId, ISystemRuntime iSystemRuntime) throws Exception {
		if (iSystemRuntime instanceof IServiceSystemRuntime) {
			((IServiceSystemRuntime) iSystemRuntime).shutdown();
		}
		super.unregisterSystemRuntime(strDeploySystemId);
	}

	protected void unregisterServiceAPIRequestMapping(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {
		if (this.getRequestMappingHandlerMapping() == null) {
			log.warn(String.format("接口请求映射处理对象无效"));
			return;
		}

		java.util.List<IPSSysServiceAPI> psSysServiceAPIs = iSystemRuntime.getPSSystem().getAllPSSysServiceAPIs();
		if (psSysServiceAPIs != null) {
			for (IPSSysServiceAPI iPSSysServiceAPI : psSysServiceAPIs) {

				ISysServiceAPIRuntime iSysServiceAPIRuntime = iSystemRuntime.getSysServiceAPIRuntime(iPSSysServiceAPI.getCodeName(), true);
				if (iSysServiceAPIRuntime == null) {
					log.warn(String.format("系统[%1$s]服务接口[%2$s]运行时对象无效，忽略请求处理映射", deploySystem.getDeploySystemId(), iPSSysServiceAPI.getName()));
					continue;
				}

				ISysServiceAPIRequestMappingAdapter iSysServiceAPIRequestMappingAdapter = getSysServiceAPIRequestMappingAdapter(iSysServiceAPIRuntime);
				if (iSysServiceAPIRequestMappingAdapter == null) {
					continue;
				}

				if (iSysServiceAPIRuntime instanceof net.ibizsys.central.cloud.core.service.ISysServiceAPIRuntime) {
					((net.ibizsys.central.cloud.core.service.ISysServiceAPIRuntime) iSysServiceAPIRuntime).unregisterMapping(iSysServiceAPIRequestMappingAdapter);
				} else {
					iSysServiceAPIRequestMappingAdapter.unregisterMapping(iSysServiceAPIRuntime);
				}

				// 注销服务
				String strServiceName = String.format("%1$s-%2$s", iSystemRuntime.getDeploySystemId(), iPSSysServiceAPI.getCodeName()).toLowerCase();
				this.unregisterNamingService(strServiceName);
			}
		}
	}

	protected void unregisterServletAppRequestMapping(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {

		String prefix = String.format("/%1$s/", this.getServletAppBaseUrl());
		java.util.List<IPSApplication> psApplications = iSystemRuntime.getPSSystem().getAllPSApps();
		if (psApplications != null) {
			for (IPSApplication iPSApplication : psApplications) {

				IApplicationRuntime iApplicationRuntime = iSystemRuntime.getApplicationRuntime(iPSApplication.getCodeName(), true);
				if (iApplicationRuntime == null) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象无效，忽略请求处理映射", deploySystem.getDeploySystemId(), iPSApplication.getName()));
					continue;
				}

				if (!(iApplicationRuntime instanceof IServletAppRuntime)) {
					// log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象类型不支持，忽略请求处理映射",
					// deploySystem.getDeploySystemId(),
					// iPSApplication.getName()));
					continue;
				}

				IServletAppRuntime iServletAppRuntime = (IServletAppRuntime) iApplicationRuntime;

				String url = iServletAppRuntime.getBaseUrl();
				if (url.indexOf(prefix) != 0) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]服务前缀[%3$s]不一致，忽略注销", deploySystem.getDeploySystemId(), iApplicationRuntime.getName(), url));
					continue;
				}

				url = url.substring(prefix.length());

				servletAppRuntimeMap.remove(url + "/**", iServletAppRuntime);

				// 注销
				String strServiceName = String.format("%1$s-app-%2$s", iSystemRuntime.getDeploySystemId(), iPSApplication.getCodeName()).toLowerCase();
				this.unregisterNamingService(strServiceName);
			}
		}
	}

	protected void unregisterProxyAppRequestMapping(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {

		if (this.getRequestMappingHandlerMapping() == null) {
			log.warn(String.format("接口请求映射处理对象无效"));
			return;
		}

		String prefix = String.format("/%1$s/", this.getProxyAppBaseUrl());
		java.util.List<IPSApplication> psApplications = iSystemRuntime.getPSSystem().getAllPSApps();
		if (psApplications != null) {
			for (IPSApplication iPSApplication : psApplications) {

				IApplicationRuntime iApplicationRuntime = iSystemRuntime.getApplicationRuntime(iPSApplication.getCodeName(), true);
				if (iApplicationRuntime == null) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象无效，忽略请求处理映射", deploySystem.getDeploySystemId(), iPSApplication.getName()));
					continue;
				}

				if (!(iApplicationRuntime instanceof IProxyAppRuntime)) {
					// log.warn(String.format("系统[%1$s]前端应用[%2$s]运行时对象类型不支持，忽略请求处理映射",
					// deploySystem.getDeploySystemId(),
					// iPSApplication.getName()));
					continue;
				}

				IProxyAppRuntime iProxyAppRuntime = (IProxyAppRuntime) iApplicationRuntime;
				String url = iProxyAppRuntime.getBaseUrl();
				if (url.indexOf(prefix) != 0) {
					log.warn(String.format("系统[%1$s]前端应用[%2$s]服务前缀[%3$s]不一致，忽略注册", deploySystem.getDeploySystemId(), iApplicationRuntime.getName(), url));
					continue;
				}

				ISysServiceAPIRequestMappingAdapter iSysServiceAPIRequestMappingAdapter = this.getApplicationRequestMappingAdapter(iProxyAppRuntime);
				iProxyAppRuntime.unregisterMapping(iSysServiceAPIRequestMappingAdapter);

				// 注销
				String strServiceName = String.format("%1$s-app-%2$s", iSystemRuntime.getDeploySystemId(), iPSApplication.getCodeName()).toLowerCase();
				this.unregisterNamingService(strServiceName);
			}
		}
	}


	protected ISysServiceAPIRequestMappingAdapter getSysServiceAPIRequestMappingAdapter(ISysServiceAPIRuntime iSysServiceAPIRuntime) throws Exception {
		if (this.iSysServiceAPIRequestMappingAdapter == null) {
			StandardSysServiceAPIRequestMappingAdapter standardSysServiceAPIRequestMappingAdapter = new StandardSysServiceAPIRequestMappingAdapter();
			standardSysServiceAPIRequestMappingAdapter.init(this.getSystemGatewayContext(), this.getRequestMappingHandlerMapping(), this.getSysServiceAPIDocAdapter());
			this.iSysServiceAPIRequestMappingAdapter = standardSysServiceAPIRequestMappingAdapter;
		}
		return this.iSysServiceAPIRequestMappingAdapter;

	}

	protected ISysServiceAPIRequestMappingAdapter getApplicationRequestMappingAdapter(IApplicationRuntime iApplicationRuntime) throws Exception {
		if (this.iSysServiceAPIRequestMappingAdapter == null) {
			StandardSysServiceAPIRequestMappingAdapter standardSysServiceAPIRequestMappingAdapter = new StandardSysServiceAPIRequestMappingAdapter();
			standardSysServiceAPIRequestMappingAdapter.init(this.getSystemGatewayContext(), this.getRequestMappingHandlerMapping(), this.getSysServiceAPIDocAdapter());
			this.iSysServiceAPIRequestMappingAdapter = standardSysServiceAPIRequestMappingAdapter;
		}
		return this.iSysServiceAPIRequestMappingAdapter;

	}

	@Override
	public boolean fillDataSourceInfo(ISysDBSchemeRuntime iSysDBSchemeRuntime) {
		// String strDBType = configEntity.getString("dbtype", null);
		// if(StringUtils.hasLength(strDBType)) {
		// iSysDBSchemeRuntime.setDBType(strDBType);
		// }

		// 设置数据源属性
		// iSysDBSchemeRuntime.setDataSourceProperties(configEntity.any());
		return super.fillDataSourceInfo(iSysDBSchemeRuntime);
	}

	/**
	 * 注册部署系统数据源
	 * 
	 * @param deploySystem
	 * @param iSystemRuntime
	 * @throws Exception
	 */
	protected void registerDataSources(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {

		List<IPSSysDBScheme> psSysDBSchemes = iSystemRuntime.getPSSystem().getAllPSSysDBSchemes();
		if (ObjectUtils.isEmpty(psSysDBSchemes)) {
			log.warn(String.format("系统[%1$s]未定义数据库体系结构", deploySystem.getDeploySystemId()));
			return;
		}

		for (IPSSysDBScheme iPSSysDBScheme : psSysDBSchemes) {
			ISysDBSchemeRuntime iSysDBSchemeRuntime = iSystemRuntime.getSysDBSchemeRuntime(iPSSysDBScheme);
			String strDBInstTag = iPSSysDBScheme.getDBInstTag();
			if (!StringUtils.hasLength(strDBInstTag)) {
				continue;
			}

			if (!strDBInstTag.equals(iSysDBSchemeRuntime.getDataSourceTag())) {
				log.warn(String.format("系统[%1$s]数据库[%2$s]指定数据源[%3$s]，忽略默认指定[%4$s]", deploySystem.getDeploySystemId(), iPSSysDBScheme.getName(), iSysDBSchemeRuntime.getDataSourceTag(), strDBInstTag));
				continue;
			}

			if (this.registerDBInst(strDBInstTag) == null) {
				log.warn(String.format("未定义数据库实例[%1$s]配置，系统[%2$s]数据库[%3$s]访问可能会出现问题", strDBInstTag, deploySystem.getDeploySystemId(), iPSSysDBScheme.getName()));
				continue;
			}
		}

		boolean bUpdateDBScheme = DataTypeUtils.getBooleanValue(deploySystem.getUpdateDBSchema(), false);
		if(!bUpdateDBScheme) {
			log.warn(String.format("系统[%1$s]忽略发布数据库结构", deploySystem.getDeploySystemId()));
			return;
		}
		if (this.getSysDBSchemeSyncAdapter() != null && bUpdateDBScheme) {
			for (IPSSysDBScheme iPSSysDBScheme : psSysDBSchemes) {
				ISysDBSchemeRuntime iSysDBSchemeRuntime = iSystemRuntime.getSysDBSchemeRuntime(iPSSysDBScheme);
				DataSource dataSource = this.dataSourceMap.get(iSysDBSchemeRuntime.getDataSourceTag());
				if (dataSource == null) {
					log.warn(String.format("系统[%1$s]数据库[%2$s]指定数据源[%3$s]不存在，忽略发布结构", deploySystem.getDeploySystemId(), iPSSysDBScheme.getName(), iSysDBSchemeRuntime.getDataSourceTag()));
					continue;
				}

				try {
					this.getSysDBSchemeSyncAdapter().sync(iSystemRuntime, iSysDBSchemeRuntime, dataSource, null);
				} catch (Throwable ex) {
//					throw new Exception(String.format("同步系统[%1$s]数据库[%2$s]结构发生异常，%3$s", iSystemRuntime.getDeploySystemId(), iSysDBSchemeRuntime.getDSLink(), ex.getMessage()), ex);
					log.error(String.format("同步系统[%1$s]数据库[%2$s]结构发生异常，%3$s", iSystemRuntime.getDeploySystemId(), iSysDBSchemeRuntime.getDSLink(), ex.getMessage()), ex);
				}
			}
		}

	}

	protected javax.sql.DataSource registerDBInst(String strDBInstTag) throws Exception {
		ConfigEntity configEntity = this.dbInstConfigMap.get(strDBInstTag);
		if (configEntity == null) {

			// 注册数据源
			String strDBInstConfigKey = String.format("%1$s%2$s", NacosServiceHubSetting.DATAID_DBINST_PREFIX, strDBInstTag);
			String strConfig = this.getConfig(strDBInstConfigKey);
			if (!StringUtils.hasLength(strConfig)) {
				log.warn(String.format("未定义数据库实例[%1$s]配置，访问可能会出现问题", strDBInstConfigKey));
				return null;
			}

			configEntity = new ConfigEntity(strConfig);
			String strDBInstConfigExKey = String.format("%1$s%2$s-ex", NacosServiceHubSetting.DATAID_DBINST_PREFIX, strDBInstTag);
			String strConfigEx = this.getConfig(strDBInstConfigExKey);
			if (StringUtils.hasLength(strConfigEx)) {
				ConfigEntity configexEntity = new ConfigEntity(strConfigEx);
				configexEntity.copyTo(configEntity);
			}

			DataSource dataSource = new DataSource();
			configEntity.copyTo(dataSource);

			if (ObjectUtils.isEmpty(dataSource.getJdbcUrl())) {
				dataSource.setJdbcUrl((String) dataSource.get("url"));
			}

			dataSource.setDataSourceId(strDBInstTag);
			dataSourceMap.put(dataSource.getDataSourceId(), dataSource);
			this.registerDataSource(dataSource);

			this.dbInstConfigMap.put(strDBInstTag, configEntity);
		}

		return this.getDataSource(strDBInstTag, true);
	}

	/**
	 * 注册应用网关路由信息
	 * 
	 * @param iSystemRuntime
	 * @throws Exception
	 */
	protected void registerAppGatewayRoutes(DeploySystem deploySystem, ISystemRuntime iSystemRuntime) throws Exception {

		java.util.List<IPSApplication> psApplications = iSystemRuntime.getPSSystem().getAllPSApps();
		if (psApplications != null) {
			for (IPSApplication iPSApplication : psApplications) {

				ObjectNode objectNode = GatewayUtils.getAppGatewayNode(iSystemRuntime.getDeploySystemId(), iSystemRuntime.getDeploySystemId(), iPSApplication);
				if (objectNode == null) {
					log.warn(String.format("部署系统[%1$s][%2$s]未提供路由配置"));
					continue;
				}

				String strConfigId = String.format("deployapp-%1$s-%2$s", iSystemRuntime.getDeploySystemId(), iPSApplication.getCodeName()).toLowerCase();
				this.publishConfig(strConfigId, objectNode);
			}
		}
	}

	@Override
	public <T extends ICloudUtilRuntime> T getCloudUtilRuntime(Class<?> systemCls, Class<T> cls, boolean bTryMode) {
		return this.getSysUtilRuntime(systemCls, cls, bTryMode);
	}

	@Override
	public <T> T getSysUtilRuntime(Class<?> systemCls, Class<T> cls, boolean bTryMode) {
		// 获取指定系统
		if (systemCls != null) {
			ISystemRuntime iSystemRuntime = (ISystemRuntime) this.getSystemRuntime(systemCls, true);
			if (iSystemRuntime != null) {
				return iSystemRuntime.getSysUtilRuntime(cls, bTryMode);
			}
		}

		if (this.getHubSystemRuntime(true) != null) {
			T t = this.getHubSystemRuntime(false).getSysUtilRuntime(cls, true);
			if (t != null && (!(t instanceof ISysUtilContainerOnly))) {
				return t;
			}
		}

		// 循环系统
		for (ISystemRuntime iSystemRuntime : this.getSystemRuntimes()) {
			if (iSystemRuntime == this.getHubSystemRuntime(true)) {
				continue;
			}
			if (iSystemRuntime instanceof IServiceSystemRuntime) {
				T t = ((IServiceSystemRuntime) iSystemRuntime).getSysUtilRuntime(cls, true, true);
				if (t != null && (!(t instanceof ISysUtilContainerOnly))) {
					return t;
				}
			} else {
				T t = iSystemRuntime.getSysUtilRuntime(cls, true);
				if (t != null && (!(t instanceof ISysUtilContainerOnly))) {
					return t;
				}
			}
		}

		if (!bTryMode) {
			if (this.getHubSystemRuntime(true) != null) {
				// 强行从网关系统中获取
				return this.getHubSystemRuntime(false).getSysUtilRuntime(cls, false);
			}
		}

		if (bTryMode)
			return null;
		throw new SystemGatewayException(this, String.format("无法获取指定系统功能组件运行时对象[%1$s]", cls.getCanonicalName()));
	}

	@Override
	public boolean doFilter(ServletRequest request, ServletResponse response) throws IOException, ServletException {

		// 置空当前认证信息
		UserContext.setCurrent(null);
		SecurityContextHolder.getContext().setAuthentication(null);
		if (this.getServiceHubFilter() == null) {
			return true;
		}

		if (this.ignoreAuthPatternSet != null) {
			String strRequestURI = ((HttpServletRequest) request).getRequestURI();
			Iterator<String> it = this.ignoreAuthPatternSet.iterator();
			while (it.hasNext()) {
				if (pathMatcher.match(it.next(), strRequestURI)) {
					if (this.getSysServiceAPIDocAdapter() != null) {
						if (strRequestURI.equals("/v3/api-docs")) {
							return this.getSysServiceAPIDocAdapter().outputOpenAPIDoc((HttpServletRequest) request, (HttpServletResponse) response);
						}
					}

					return true;
				}
			}
		}
		return this.getServiceHubFilter().doFilter(request, response);
	}

	@Override
	public boolean redoFilter() throws IOException, ServletException {
		// UserContext.setCurrent(null);
		if (this.getServiceHubFilter() == null) {
			return false;
		}
		return this.getServiceHubFilter().redoFilter();
	}

	@Override
	public void registerIgnoreAuthPattern(String ignoreAuthPattern) {
		if (this.ignoreAuthPatternSet == null) {
			this.ignoreAuthPatternSet = new HashSet<String>();
		}
		this.ignoreAuthPatternSet.addAll(Arrays.asList(ignoreAuthPattern.split("\\s*,\\s*")));
	}

	protected void setServiceHubFilter(IServiceHubFilter iServiceHubFilter) {
		this.realServiceHubFilter = iServiceHubFilter;
	}

	protected IServiceHubFilter getServiceHubFilter() {
		return this.realServiceHubFilter;
	}

	@Override
	public IHubSystemRuntime getHubSystemRuntime() {
		return this.getHubSystemRuntime(false);
	}

	@Override
	public IHubSystemRuntime getHubSystemRuntime(boolean bTryMode) {
		if (this.hubSystemRuntime != null || bTryMode) {
			return this.hubSystemRuntime;
		}
		throw new SystemGatewayException(this, String.format("未定义服务总线系统"));
	}

	protected void setHubSystemRuntime(IHubSystemRuntime hubSystemRuntime) {
		this.hubSystemRuntime = hubSystemRuntime;
	}

	@Override
	public void registerNamingService(String strServiceId) {
		this.registerNamingService(strServiceId, this.strIPAddress, this.port);
	}

	@Override
	public void registerNamingService(String strServiceId, String strIp, int nPort) {
		if (!getServiceHubSetting().isRegisterNamingService()) {
			log.warn(String.format("忽略注册命名服务[%1$s]", strServiceId));
			return;
		}

		if (this.namingService == null) {
			throw new SystemGatewayException(this, "命名服务对象无效");
		}

		try {
			if (this.isEnableDebug()) {
				// 强行移除
				// namingService.deregisterInstance(strServiceId,
				// nacosDiscoveryProperties.getGroup(), strIp, nPort);
				log.warn(String.format("调试模式调整命名服务[%1$s] ==> [%1$s-debug]", strServiceId));
				strServiceId = String.format("%1$s-debug", strServiceId);
				namingService.registerInstance(strServiceId, nacosDiscoveryProperties.getGroup(), strIp, nPort);
			} else {
				namingService.registerInstance(strServiceId, nacosDiscoveryProperties.getGroup(), strIp, nPort);
				// strServiceId = String.format("%1$s-debug", strServiceId);
				// namingService.deregisterInstance(strServiceId,
				// nacosDiscoveryProperties.getGroup(), strIp, nPort);
			}

		} catch (Exception ex) {
			log.error(String.format("注册命名服务[%1$s]发生异常，%2$s", strServiceId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("注册命名服务[%1$s][%3$s:%4$s]发生异常，%2$s", strServiceId, ex.getMessage(), strIp, nPort), ex);
		}
	}

	@Override
	public void unregisterNamingService(String strServiceId) {
		this.unregisterNamingService(strServiceId, this.strIPAddress, this.port);
	}

	@Override
	public void unregisterNamingService(String strServiceId, String strIp, int nPort) {
		if (!getServiceHubSetting().isRegisterNamingService()) {
			log.warn(String.format("忽略注销命名服务[%1$s]", strServiceId));
			return;
		}

		if (this.namingService == null) {
			throw new SystemGatewayException(this, "命名服务对象无效");
		}

		try {
			if (this.isEnableDebug()) {
				strServiceId = String.format("%1$s-debug", strServiceId);
			}
			namingService.deregisterInstance(strServiceId, nacosDiscoveryProperties.getGroup(), strIp, nPort);
		} catch (Exception ex) {
			log.error(String.format("注册命名服务[%1$s]发生异常，%2$s", strServiceId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("注销命名服务[%1$s][%3$s:%4$s]发生异常，%2$s", strServiceId, ex.getMessage(), strIp, nPort), ex);
		}
	}

	@Override
	public void publishConfig(String strConfigId, ObjectNode objectNode) {
		Assert.hasLength(strConfigId, "传入键名无效");
		if (!getServiceHubSetting().isPublishConfig()) {
			log.warn(String.format("忽略发布配置[%1$s]", strConfigId));
			return;
		}

		String strBackupConfigId = strConfigId;
		strConfigId = NacosConfigUtils.getDataId(strBackupConfigId);
		if (!strBackupConfigId.equals(strConfigId)) {
			log.warn(String.format("键名[%1$s]包含非法字符，转换至[%2$s]，可能会出现键名冲突", strBackupConfigId, strConfigId));
		}

		ConfigService configService = nacosConfigManager.getConfigService();
		try {
			String strContent = objectNode.toPrettyString();
			String strLastContent = configService.getConfig(strConfigId, nacosConfigProperties.getGroup(), 5000);
			if (DataTypeUtils.compare(DataTypes.VARCHAR, strContent, strLastContent) == 0) {
				log.warn(String.format("配置项[%1$s]没有发生变化，忽略发布", strConfigId));
				return;
			}
			configService.publishConfig(strConfigId, nacosConfigProperties.getGroup(), strContent, com.alibaba.nacos.api.config.ConfigType.JSON.toString());
		} catch (Exception ex) {
			log.error(String.format("发布配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("发布配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
		}
	}

	@Override
	public void publishConfig(String strConfigId, Map map) {
		Assert.hasLength(strConfigId, "传入键名无效");

		if (!getServiceHubSetting().isPublishConfig()) {
			log.warn(String.format("忽略发布配置[%1$s]", strConfigId));
			return;
		}

		String strBackupConfigId = strConfigId;
		strConfigId = NacosConfigUtils.getDataId(strBackupConfigId);
		if (!strBackupConfigId.equals(strConfigId)) {
			log.warn(String.format("键名[%1$s]包含非法字符，转换至[%2$s]，可能会出现键名冲突", strBackupConfigId, strConfigId));
		}

		ConfigService configService = nacosConfigManager.getConfigService();
		try {
			Yaml yaml = new Yaml();// 将Map转换成 YAML 字符串
			String strContent = yaml.dumpAsMap(map);

			String strLastContent = configService.getConfig(strConfigId, nacosConfigProperties.getGroup(), 5000);
			if (DataTypeUtils.compare(DataTypes.VARCHAR, strContent, strLastContent) == 0) {
				log.warn(String.format("配置项[%1$s]没有发生变化，忽略发布", strConfigId));
				return;
			}

			configService.publishConfig(strConfigId, nacosConfigProperties.getGroup(), strContent, com.alibaba.nacos.api.config.ConfigType.YAML.toString());
		} catch (Exception ex) {
			log.error(String.format("发布配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("发布配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
		}
	}

	@Override
	public void publishConfig(String strConfigId, String strContent) {
		Assert.hasLength(strConfigId, "传入键名无效");

		if (!getServiceHubSetting().isPublishConfig()) {
			log.warn(String.format("忽略发布配置[%1$s]", strConfigId));
			return;
		}

		String strBackupConfigId = strConfigId;
		strConfigId = NacosConfigUtils.getDataId(strBackupConfigId);
		if (!strBackupConfigId.equals(strConfigId)) {
			log.warn(String.format("键名[%1$s]包含非法字符，转换至[%2$s]，可能会出现键名冲突", strBackupConfigId, strConfigId));
		}

		ConfigService configService = nacosConfigManager.getConfigService();
		try {

			String strLastContent = configService.getConfig(strConfigId, nacosConfigProperties.getGroup(), 5000);
			if (DataTypeUtils.compare(DataTypes.VARCHAR, strContent, strLastContent) == 0) {
				log.warn(String.format("配置项[%1$s]没有发生变化，忽略发布", strConfigId));
				return;
			}
			if (strContent == null) {
				strContent = "";
			}

			if (strContent.indexOf("{") == 0) {
				configService.publishConfig(strConfigId, nacosConfigProperties.getGroup(), strContent, com.alibaba.nacos.api.config.ConfigType.JSON.toString());
			} else {
				configService.publishConfig(strConfigId, nacosConfigProperties.getGroup(), strContent, com.alibaba.nacos.api.config.ConfigType.YAML.toString());
			}

		} catch (Exception ex) {
			log.error(String.format("发布配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("发布配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
		}
	}

	@Override
	public String getConfig(String strConfigId) {

		Assert.hasLength(strConfigId, "传入键名无效");

		String strBackupConfigId = strConfigId;
		strConfigId = NacosConfigUtils.getDataId(strBackupConfigId);
		if (!strBackupConfigId.equals(strConfigId)) {
			log.warn(String.format("键名[%1$s]包含非法字符，转换至[%2$s]，可能会出现键名冲突", strBackupConfigId, strConfigId));
		}

		ConfigService configService = nacosConfigManager.getConfigService();
		try {
			return configService.getConfig(strConfigId, nacosConfigProperties.getGroup(), 5000);
		} catch (Exception ex) {
			log.error(String.format("获取配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("获取配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
		}
	}

	@Override
	public void addConfigListener(String strConfigId, final IConfigListener listener) {
		Listener l = new Listener() {
			@Override
			public Executor getExecutor() {
				return listener.getExecutor();
			}

			@Override
			public void receiveConfigInfo(String configInfo) {
				log.debug(String.format("配置项[%1$s]发生变化", strConfigId));
				listener.receiveConfigInfo(configInfo);
			}
		};
		ConfigService configService = nacosConfigManager.getConfigService();
		try {
			configService.addListener(strConfigId, nacosConfigProperties.getGroup(), l);
			this.configListenerMap.put(listener, l);
		} catch (Exception ex) {
			log.error(String.format("侦听配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("侦听配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
		}
	}

	@Override
	public void removeConfigListener(String strConfigId, IConfigListener listener) {

		Listener l = this.configListenerMap.get(listener);
		if (listener == null) {
			log.warn(String.format("未注册侦听器[%1$s]", listener));
			return;
		}

		ConfigService configService = nacosConfigManager.getConfigService();
		try {

			configService.removeListener(strConfigId, nacosConfigProperties.getGroup(), l);
			this.configListenerMap.remove(listener);
		} catch (Exception ex) {
			log.error(String.format("取消侦听配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("取消侦听配置[%1$s]发生异常，%2$s", strConfigId, ex.getMessage()), ex);
		}
	}

	@Override
	public void requireCloudUtilRuntime(Class<?> systemRuntimeCls, Class<? extends ICloudUtilRuntime> cloudUtilCls) {
		this.requiredCloudUtilRuntimeMap.put(cloudUtilCls, systemRuntimeCls);
	}

	@Override
	public AppData invokeGetAppData(String strDeploySystemId, String strSystemTag, String strOrgId, IAuthenticationUser iAuthenticationUser, Object objTag) throws Throwable {
		// 获取系统运行时
		ISystemRuntime iSystemRuntime = this.getSystemRuntime(strDeploySystemId);
		if (iSystemRuntime instanceof IServiceSystemRuntime) {
			return ((IServiceSystemRuntime) iSystemRuntime).invokeGetAppData(strSystemTag, strOrgId, iAuthenticationUser, objTag);
		}
		throw new SystemGatewayException(this, String.format("部署系统[%1$s]不支持此调用", strDeploySystemId));
	}

	@Override
	public Object invokeDevOpsAction(String strDeploySystemId, String strAction, Object objBody, String strKey, IUserContext iUserContext, Object objTag) throws Throwable {

		IAuthenticationUser authenticationUser = AuthenticationUser.getCurrentMust();
		if (!AuthenticationUser.isSuperuser(authenticationUser) && !AuthenticationUser.isDevuser(authenticationUser)) {
			if (!iUserContext.isSuperuser()) {
				throw new SystemGatewayException(this, "必须管理员才能进行此操作", Errors.ACCESSDENY);
			}
		}

		// 获取系统运行时
		ISystemRuntime iSystemRuntime = this.getSystemRuntime(strDeploySystemId);
		// 优先使用指定系统DevOps组件执行作业
		ISysDevOpsUtilRuntime iSysDevOpsUtilRuntime = iSystemRuntime.getSysUtilRuntime(ISysDevOpsUtilRuntime.class, true);
		if (iSysDevOpsUtilRuntime != null) {
			if (!(iSysDevOpsUtilRuntime instanceof ISysUtilContainerOnly) && iSysDevOpsUtilRuntime instanceof IHubSysDevOpsUtilRuntime) {
				return ((IHubSysDevOpsUtilRuntime) iSysDevOpsUtilRuntime).executeAction(iSystemRuntime, strAction, objBody, strKey, objTag);
			}
			return iSysDevOpsUtilRuntime.executeAction(strAction, objBody, strKey, objTag);
		}
		IHubSysDevOpsUtilRuntime iHubSysDevOpsUtilRuntime = this.getHubSystemRuntime().getSysUtilRuntime(IHubSysDevOpsUtilRuntime.class, false);
		return iHubSysDevOpsUtilRuntime.executeAction(iSystemRuntime, strAction, objBody, strKey, objTag);
	}

	protected ISystemRuntime reloadDeploySystem(DeploySystem deploySystem, String strConfigInfo) {

		ISystemRuntime iSystemRuntime = this.getLoadedSystemRuntime(deploySystem.getDeploySystemId(), false);

		try {
			this.unregisterSystemRuntime(deploySystem, iSystemRuntime);
		} catch (Throwable ex) {
			throw new SystemGatewayException(this, String.format("注销系统[%1$s]运行时对象发生异常，%2$s", deploySystem.getDeploySystemId(), ex.getMessage()), ex);
		}

		try {
			this.registerDeploySystem(deploySystem);
		} catch (Throwable ex) {
			throw new SystemGatewayException(this, String.format("注册系统[%1$s]运行时对象发生异常，%2$s", deploySystem.getDeploySystemId(), ex.getMessage()), ex);
		}

		log.info(String.format("重新加载系统[%1$s]运行时对象完成", deploySystem.getDeploySystemId()));
		return iSystemRuntime;
	}

	@Override
	public Resource resolveWebResource(HttpServletRequest request, String requestPath) {

		for (java.util.Map.Entry<String, IServletAppRuntime> entry : this.servletAppRuntimeMap.entrySet()) {
			if (pathMatcher.match(entry.getKey(), requestPath)) {
				return entry.getValue().resolveResource(request, requestPath);
			}
		}

		return null;
	}

	@Override
	public IPSSystemService getPSSystemService(String serviceId, boolean tryMode) {
		DeploySystem deploySystem = new DeploySystem();
		deploySystem.setDeploySystemId(serviceId);
		deploySystem.setModelPath(DEPLOYSYSYTEMMODELPATH_REMOTE);
		try {
			return this.getPSSystemService(deploySystem, tryMode);
		} catch (Exception ex) {
			throw new SystemGatewayException(this, String.format("获取指定服务系统[%1$s]模型发生异常，%2$s", serviceId, ex.getMessage()), ex);
		}
	}
	
	
	

	@Override
	public List<ServiceInstance> getNamingServiceInstances(String strServiceId) {
		if (this.namingService == null) {
			throw new SystemGatewayException(this, "命名服务对象无效");
		}

		try {
			 List<Instance> instanceList = this.namingService.selectInstances(strServiceId, nacosDiscoveryProperties.getGroup(), true);
			 if(!ObjectUtils.isEmpty(instanceList)) {
				 List<ServiceInstance> list = new ArrayList<ServiceInstance>();
				 for(Instance instance : instanceList) {
					 ServiceInstance serviceInstance = new ServiceInstance();
					 serviceInstance.setServiceName(strServiceId);
					 serviceInstance.setIp(instance.getIp());
					 serviceInstance.setPort(instance.getPort());
					 list.add(serviceInstance);
				 }
				 return list;
			 }
			return null;

		} catch (Exception ex) {
			log.error(String.format("获取命名服务[%1$s]实例清单发生异常，%2$s", strServiceId, ex.getMessage()), ex);
			throw new SystemGatewayException(this, String.format("获取命名服务[%1$s]实例清单发生异常，%2$s", strServiceId, ex.getMessage()), ex);
		}
	}

	@Override
	public ISystemRuntime reloadSystemRuntime(String serviceId) {

		Assert.hasLength(serviceId, "未传入系统标识");
		
		DeploySystem reloadDeploySystem = null;
		List<DeploySystem> deploySystemList = this.getServiceHubSetting().getDeploySystems();
		if (deploySystemList != null) {
			for (DeploySystem deploySystem : deploySystemList) {
				if(serviceId.equalsIgnoreCase(deploySystem.getDeploySystemId())) {
					reloadDeploySystem = deploySystem;
					break;
				}
			}
		}
		
		if(reloadDeploySystem == null) {
			throw new SystemGatewayException(this, String.format("无法获取指定部署系统[%1$s]", serviceId));
		}
			
		return this.reloadDeploySystem(reloadDeploySystem, null);
	}
	
	@Override
	public String getServiceUrl() {
		return String.format("http://%1$s:%2$s", this.strIPAddress, this.port);
	}
	
	
	@Override
	public String getIPAddress() {
		return this.strIPAddress;
	}

	@Override
	public int getPort() {
		return this.port;
	}
}
