package com.jsmframe.service;

import com.alibaba.fastjson.TypeReference;
import com.jsmframe.base.BaseService;
import com.jsmframe.context.AppContext;
import com.jsmframe.context.SpringContext;
import com.jsmframe.jedis.JedisService;
import com.jsmframe.utils.StringUtil;
import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

/**
 * EhCache + Redis(异步)
 */
@Service
public class CacheService extends BaseService {

    private ExecutorService executorService = null;

    private Cache queryCache;
    @Resource
    private JedisService jedisService;

    private void init() {
        Integer threadNumber = AppContext.getAsInteger("cache.service.async.thread");
        if (threadNumber == null) {
            threadNumber = 1;
        }
        logger.info("CacheService Redis use async thread number:{}", threadNumber);
        executorService = Executors.newFixedThreadPool(threadNumber, new ThreadFactory() {
            byte index = 0;

            @Override
            public Thread newThread(Runnable runnable) {
                return new Thread(runnable, "cache-pool" + (++index));
            }
        });
    }

    public Cache getQueryCache() {
        if (queryCache == null) {
            String ehcacheName = AppContext.get("cache.service.cache.name");
            if (StringUtil.isEmpty(ehcacheName)) {
                ehcacheName = "queryCache";
            }
            queryCache = SpringContext.getBean(ehcacheName);
            logger.info("CacheService EhCache use cache name:{}", ehcacheName);
        }
        return queryCache;
    }

    public void close() {
        if (executorService != null) {
            executorService.shutdown();
        }
    }

    private boolean check() {
        try {
            if (executorService == null) {
                init();
            }
            return true;
        } catch (Exception e) {
            logger.error("CacheService init error.", e);
            return false;
        }
    }

    public void remove(String key){
        queryCache.remove(key);
        jedisService.del(key);
    }

    public <T> T get(String key, Class<T> clazz) {
        if (!check()) {
            return null;
        }
        T result = getFromL1(key);
        if (result == null) {
            result = getFromL2(key, clazz);
        }
        return result;
    }

    public <T> T get(String key, TypeReference<T> typeReference) {
        if (!check()) {
            return null;
        }
        T result = getFromL1(key);
        if (result == null) {
            result = getFromL2(key, typeReference);
        }
        return result;
    }

    public <T> T get(String key, Type type) {
        if (!check()) {
            return null;
        }
        T result = getFromL1(key);
        if (result == null) {
            result = getFromL2(key, type);
        }
        return result;
    }

    public void set(String key, int seconds, Serializable object) {
        if (!check()) {
            return;
        }
        Cache ehQueryCache = getQueryCache();
        //设置一级缓存
        ehQueryCache.put(new Element(key, object));
        //设置二级缓存
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                jedisService.setex(key, seconds, StringUtil.toJson(object));
            }
        });

    }

    private <T> T getFromL1(String key) {
        if (!check()) {
            return null;
        }
        Cache ehQueryCache = getQueryCache();
        Element element = ehQueryCache.get(key);
        if (element != null) {
            logger.debug("from cache l1 key:{}", key);
            return (T) element.getObjectValue();
        }
        return null;
    }

    private <T> T getFromL2(String key, Class<T> clazz) {
        if (!check()) {
            return null;
        }
        Cache ehQueryCache = getQueryCache();
        String resultStr = jedisService.get(key);
        if (!StringUtil.isEmpty(resultStr)) {
            T result = StringUtil.parseObject(resultStr, clazz);
            if (result != null) {//保存到一级缓存
                logger.debug("from cache l2 key:{}", key);
                ehQueryCache.put(new Element(key, (Serializable) result));
                return result;
            }
        }
        return null;
    }

    private <T> T getFromL2(String key, TypeReference<T> typeReference) {
        if (!check()) {
            return null;
        }
        Cache ehQueryCache = getQueryCache();
        String resultStr = jedisService.get(key);
        if (!StringUtil.isEmpty(resultStr)) {
            T result = StringUtil.parseObject(resultStr, typeReference);
            if (result != null) {//保存到一级缓存
                logger.debug("from cache l2 key:{}", key);
                ehQueryCache.put(new Element(key, (Serializable) result));
                return result;
            }
        }
        return null;
    }

    private <T> T getFromL2(String key, Type type) {
        if (!check()) {
            return null;
        }
        Cache ehQueryCache = getQueryCache();
        String resultStr = jedisService.get(key);
        if (!StringUtil.isEmpty(resultStr)) {
            T result = StringUtil.parseObject(resultStr, type);
            if (result != null) {//保存到一级缓存
                logger.debug("from cache l2 key:{}", key);
                ehQueryCache.put(new Element(key, (Serializable) result));
                return result;
            }
        }
        return null;
    }

}
