package com.jsmframe.log4j2;

import com.alibaba.fastjson.JSONObject;
import com.jsmframe.context.AppContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.disruptor.JsmDisruptor;
import com.jsmframe.elastic.ElasticClient;
import com.jsmframe.stat.model.Alarm;
import com.jsmframe.utils.DateUtil;
import com.jsmframe.utils.StringUtil;
import com.jsmframe.utils.SystemUtil;
import com.lmax.disruptor.EventHandler;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.impl.ThrowableProxy;
import org.apache.logging.log4j.core.layout.PatternLayout;

import java.io.Serializable;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicLong;

@Plugin(name = "ElasticAppender", category = "Core", elementType = "appender", printObject = true)
public class ElasticAppender extends AbstractAppender {
    private static ElasticClient elasticClient;
    private static String address;
    private static AtomicLong idAi = new AtomicLong(0);
    private static String clusterId;
    private static String jsonOption = "{\"mappings\":{\"properties\":{\"time\":{\"type\":\"date\",\"format\":\"yyyyMMdd HH:mm:ss.SSS||epoch_millis\"},\"logName\":{\"type\":\"keyword\"},\"methodName\":{\"type\":\"keyword\"},\"reqId\":{\"type\":\"keyword\"},\"level\":{\"type\":\"keyword\"},\"thread\":{\"type\":\"keyword\"}}}}";
    private boolean hasAddress = true;

    private JsmDisruptor<String> stringDisruptor = new JsmDisruptor<String>(new EventHandler<JsmDisruptor<String>.MessageEvent<String>>() {
        @Override
        public void onEvent(JsmDisruptor<String>.MessageEvent<String> stringMessageEvent, long sequence, boolean endOfBatch) throws Exception {
            try {
                elasticClient.post(getId(), stringMessageEvent.message);
            } catch (Exception e) {
                LOGGER.error("ElasticAppender onEvent error:", e);
                SystemUtil.addAlarm(new Alarm("ElasticAppender", "onEvent error."));
            }

        }
    }, 1024);

    protected ElasticAppender(String name, Filter filter, Layout<? extends Serializable> layout, boolean ignoreExceptions) {
        super(name, filter, layout, ignoreExceptions);
    }

    @Override
    public void append(LogEvent event) {
        if (!hasAddress) {
            return;
        }
        String reqId = WebContext.getRequestId(event.getThreadName());
        if (reqId == null) {
//            LOGGER.error("ElasticAppender reqId is null,threadName:" + event.getThreadName());
            return;
        }
        if (elasticClient == null) {
            boolean ok = tryCreateElasticClient();
            if (ok) {
                LOGGER.info("ElasticAppender create elasticClient ok:" + address);
            } else {
                return;
            }
        }
        JSONObject jsonObject = new JSONObject();
        ThrowableProxy thrownProxy = event.getThrownProxy();
        jsonObject.put("time", event.getTimeMillis());
        jsonObject.put("logName", event.getLoggerName());
        if (event.getSource() != null) {
            jsonObject.put("methodName", event.getSource().getMethodName());
        }
        if (event.getMessage() != null) {
            jsonObject.put("logMsg", event.getMessage().getFormattedMessage());
        }
        jsonObject.put("reqId", reqId);
        jsonObject.put("level", event.getLevel().name());
        jsonObject.put("thread", event.getThreadName());
        if (thrownProxy != null) {
            jsonObject.put("exMsg", thrownProxy.getMessage());
            jsonObject.put("exName", thrownProxy.getName());
            jsonObject.put("exTrace", parseException(thrownProxy.getStackTrace()));
        }
        stringDisruptor.pushMessage(jsonObject.toJSONString());
    }

    private String getId() {
        try {
            long intCount = 0;
            intCount = idAi.getAndIncrement();
            if (intCount + 1000 > Long.MAX_VALUE) {
                idAi.set(0);
            }
            return String.format("%s-%s-%s", clusterId, DateUtil.format(DateUtil.currentDate(), DateUtil.COMPACT_DATE_PATTERN), intCount);
        } catch (Exception e) {
            LOGGER.error("getId error.", e);
            return StringUtil.uuid();
        }
    }

    private boolean tryCreateElasticClient() {
        if (StringUtil.isEmpty(address)) {
            LOGGER.debug("ElasticAppender no address attribute. find elastic.address in AppContext");
            address = AppContext.get("elastic.address");
            if (StringUtil.isEmpty(address)) {
                LOGGER.error("tryCreateElasticClient give up, address is empty.");
                if (hasAddress) {
                    hasAddress = false;
                }
                return false;
            }
        }
        LOGGER.info("elastic.address:" + address);
        clusterId = AppContext.get("cluster.id");
        elasticClient = new ElasticClient(String.format("%s/app_%s", address, clusterId));
        elasticClient.createIndexIfNotExists(jsonOption);
        return true;
    }

    private String parseException(StackTraceElement[] stackTrace) {
        StringBuffer sb = new StringBuffer();
        sb.append("\n");
        Arrays.stream(stackTrace).forEach((e) -> sb.append(e.getClassName()).append(".").append(e.getMethodName()).append("(").append(e.getFileName()).append(":").append(e.getLineNumber()).append(")").append("\n")
        );
        return sb.toString();
    }

    /**
     * @param name
     * @param filter
     * @param layout
     * @param ignoreExceptions
     * @param address
     * @return
     */
    @PluginFactory
    public static ElasticAppender createAppender(@PluginAttribute("name") String name,
                                                 @PluginElement("Filter") final Filter filter,
                                                 @PluginElement("Layout") Layout<? extends Serializable> layout,
                                                 @PluginAttribute("ignoreExceptions") boolean ignoreExceptions,
                                                 @PluginAttribute("address") String address
    ) {
        if (name == null) {
            name = "elastic";
            LOGGER.warn("ElasticAppender no name attribute,use default:" + name);
        }
        if (layout == null) {
            layout = PatternLayout.createDefaultLayout();
        }
        return new ElasticAppender(name, filter, layout, ignoreExceptions);
    }

    public static void recreateDb() {
        if (elasticClient != null) {
            elasticClient.recreateIndex(jsonOption);
        }
    }
}
