package com.jsmframe.interceptor;

import com.jsmframe.context.AppContext;
import com.jsmframe.context.SpringContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.rest.resp.RestResp;
import com.jsmframe.service.DefaultApiStatService;
import com.jsmframe.stat.model.ApiStat;
import com.jsmframe.utils.*;
import org.slf4j.Logger;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author longzl / @createOn 2010-6-25
 */
public class JsmFilter extends OncePerRequestFilter {
    private Logger logger = LogUtil.log(this.getClass());
    private String encoding = "UTF-8";
    private boolean apiStat = AppContext.getAsBoolean("api.stat", false);
    private String[] excludeArr = null;

    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    @Override
    protected void initFilterBean() throws ServletException {
        super.initFilterBean();
        String exclude = getFilterConfig().getInitParameter("exclude");
        if (!StringUtil.isEmpty(exclude)) {
            excludeArr = exclude.split(";");
        }
    }

    private boolean isExclude(HttpServletRequest request) {
        if (excludeArr != null) {
            String servletPath = request.getServletPath();
            for (String pattern : excludeArr) {
                if (servletPath.startsWith(pattern)) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain fc) {
        if (isExclude(request)) {
            try {
                fc.doFilter(request, response);
            } catch (Exception e) {
                logger.error("", e);
            }
            return;
        }
        ThreadLocalUtil.cleanAll();
        WebContext.setRequest(request);
        WebContext.setResponse(response);
        String reqMethod = request.getMethod();
        final long pid = Thread.currentThread().getId();
        final String uri = String.format("%s;%s;%s", pid, reqMethod, request.getRequestURI(), WebContext.getRemoteIp(request));
        String reqId = WebContext.getRequestId();
        logger.info(uri);
        long startTime = System.currentTimeMillis();
        response.setCharacterEncoding(this.encoding);
        try {
            if (reqMethod.equals(RequestMethod.OPTIONS.name())) {
                String error = "request method not support.";
                logger.warn("request method not support {}", reqMethod);
                throw new RuntimeException(error);
            } else {
                logger.debug("before fc.doFilter");
                fc.doFilter(request, response);// FileUploadBase$SizeLimitExceededException
                logger.debug("after fc.doFilter");
            }
        } catch (Exception e) {
            String error = "request error url:" + request.getRequestURL().toString();
            logger.error(error, e);
            RestResp<String> errorResp = JsmFrameUtil.dealException(e);
            ResponseUtil.outputRestResp(response, errorResp);
        } finally {
            if (apiStat) {
                statApi(request, response, reqMethod, startTime, reqId);
            }
            WebContext.removeRequest();
            WebContext.removeResponse();
            WebContext.clearRequestId();
            response.setHeader("Access-Control-Allow-Origin", "*");
            response.setHeader("Access-Control-Allow-Methods", "POST, GET, OPTIONS, DELETE, HEAD");
            response.setHeader("Access-Control-Max-Age", "3600");
            response.setHeader("Access-Control-Allow-Headers", "access-control-allow-origin, authority, content-type, version-info, X-Requested-With");
        }
    }


    private void statApi(HttpServletRequest request, HttpServletResponse response, String reqMethod, long startTime, String reqId) {
        DefaultApiStatService defaultApiStatService = SpringContext.getBean(DefaultApiStatService.class);
        long endTime = System.currentTimeMillis();
        long spendTime = endTime - startTime;
        ApiStat apiStat = new ApiStat();
        apiStat.name = AppContext.get("api.stat.name");
        apiStat.reqId = reqId;
        apiStat.url = reqMethod + " " + request.getRequestURI();
        apiStat.beginTime = startTime;
        apiStat.endTime = endTime;
        apiStat.spendTime = spendTime;
        apiStat.httpStatus = response.getStatus();
        apiStat.respCode = (String) request.getAttribute(WebContext.JSM_RESP_CODE);
        apiStat.createTime = DateUtil.currentTime();
        defaultApiStatService.add(apiStat);
    }

}
