package com.jsmframe.auth;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.jsmframe.annotation.RestAnn;
import com.jsmframe.consts.BasePairConsts;
import com.jsmframe.context.AppContext;
import com.jsmframe.http.HttpClient;
import com.jsmframe.oauth.password.OauthContext;
import com.jsmframe.rest.resp.RestResp;
import com.jsmframe.service.CacheService;
import com.jsmframe.utils.*;
import org.apache.http.Header;
import org.apache.http.message.BasicHeader;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;

@Component
public class PasswordAuth implements Auth {

    private Logger logger = LogUtil.log(PasswordAuth.class);

    @Resource
    private CacheService cacheService;

    private String tokenName = AppContext.get("oauth.password.tokenName");

    private String verifyUrl = AppContext.get("oauth.password.verifyUrl");

    private HttpClient httpClient = new HttpClient();

    @Override
    public String checkAuth(HttpServletRequest request, RestAnn resetAnn, boolean isTry) {
        String token = RequestUtil.get(request, tokenName);
        if (StringUtil.isEmpty(token)) {
            ExceptionUtil.info(BasePairConsts.VERIFY_ERROR, "缺少AT");
        }
        String resourceCode = resetAnn.permission();
        //add cache
        boolean res = verifyOauthToken(token, resourceCode);
        if (!res) {
            ExceptionUtil.info(BasePairConsts.VERIFY_ERROR);
        }
        return token;
    }

    //再访问URL 验证Token
    private RestResp<String> getVerifyOauthTokenResult(String accessToken, String resourceCode) {

        try {
            if (StringUtil.isEmpty(verifyUrl)) {
                logger.error("oauth.password.verifyUrl not found, in AppContext!");
                return null;
            }
            List<Header> headerList = new ArrayList<Header>();
            headerList.add(new BasicHeader(tokenName, accessToken));
            String res = httpClient.get(headerList, verifyUrl);
            logger.debug("res:{}", res);
            if (StringUtil.isEmpty(res)) {
                logger.error("get verify token info empty error.");
                return null;
            } else {
                RestResp<String> resp = JSON.parseObject(res, new TypeReference<RestResp<String>>() {
                });
                return resp;
            }
        } catch (Exception e) {
            logger.error("getVerifyOauthTokenResult error:" + verifyUrl, e);
            return null;
        }
    }

    //先取缓存 如果没有 再访问URL
    private boolean verifyOauthToken(String token, String resourceCode) {
        try {
            String cacheKey = token + "_" + resourceCode;
            RestResp<String> verifyTokenResp = cacheService.get(cacheKey, new TypeReference<RestResp<String>>() {
            });
            if (verifyTokenResp == null) {
                verifyTokenResp = getVerifyOauthTokenResult(token, resourceCode);
                Integer expireTime = AppContext.getAsInteger("oauth.password.expireIn");
                if (expireTime == null) {
                    expireTime = 5;//cache 5s
                    logger.warn("oauth.password.expireIn not config, use default:{}", expireTime);
                }
                if (!verifyTokenResp.isOK()) {
                    logger.error("verify token failed." + StringUtil.toJson(verifyTokenResp));
                }
                cacheService.set(cacheKey, expireTime, verifyTokenResp);
            }
            OauthContext.setAccessToken(token);
            OauthContext.setVerifyResult(verifyTokenResp.result);
            return verifyTokenResp.isOK();
        } catch (Exception e) {
            logger.error("verifyOauthToken error!", e);
            return false;
        }
    }


}
