package com.jsmframe.auth;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.jsmframe.annotation.RestAnn;
import com.jsmframe.context.AppContext;
import com.jsmframe.exception.AuthFailException;
import com.jsmframe.http.HttpClient;
import com.jsmframe.oauth.client.OauthContext;
import com.jsmframe.rest.resp.RestResp;
import com.jsmframe.service.CacheService;
import com.jsmframe.utils.LogUtil;
import com.jsmframe.utils.RequestUtil;
import com.jsmframe.utils.StringUtil;
import org.apache.http.Header;
import org.apache.http.message.BasicHeader;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;

@Component
public class ClientAuth implements Auth {

    private Logger logger = LogUtil.log(ClientAuth.class);

    @Resource
    private CacheService cacheService;

    private String tokenName = AppContext.get("oauth.client.tokenName");

    private String verifyUrl = AppContext.get("oauth.client.verifyUrl");

    private HttpClient httpClient = new HttpClient();

    @Override
    public String checkAuth(HttpServletRequest request, RestAnn resetAnn, boolean isTry) {
        String token = RequestUtil.get(request, tokenName);
        String resourceCode = resetAnn.permission();
        //add cache
        boolean res = verifyOauthToken(token, resourceCode);
        if (!res) {
            throw new AuthFailException();
        }
        return token;
    }

    //再访问URL 验证Token
    private RestResp<String> getVerifyOauthTokenResult(String accessToken, String resourceCode) {
        try {
            if (StringUtil.isEmpty(verifyUrl)) {
                logger.error("oauth.client.verifyUrl not found, in AppContext!");
                return null;
            }
            List<Header> headerList = new ArrayList<Header>();
            headerList.add(new BasicHeader(tokenName, accessToken));
            String res = httpClient.get(headerList, verifyUrl);
            logger.debug("res:{}", res);
            if (StringUtil.isEmpty(res)) {
                logger.error("get verify token info empty error.");
                return null;
            } else {
                RestResp<String> resp = JSON.parseObject(res, new TypeReference<RestResp<String>>() {
                });
                return resp;
            }
        } catch (Exception e) {
            logger.error("getVerifyOauthTokenResult error:" + verifyUrl, e);
            return null;
        }
    }

    //先取缓存 如果没有 再访问URL
    private boolean verifyOauthToken(String token, String resourceCode) {
        try {
            String cacheKey = token + "_" + resourceCode;
            RestResp<String> verifyTokenResp = cacheService.get(cacheKey, new TypeReference<RestResp<String>>() {
            });
            if (verifyTokenResp == null) {
                verifyTokenResp = getVerifyOauthTokenResult(token, resourceCode);
                Integer expireTime = AppContext.getAsInteger("oauth.client.expireIn");
                if (expireTime == null) {
                    expireTime = 5;//cache 5s
                    logger.warn("oauth.client.expireIn not config, use default:{}", expireTime);
                }
                if (!verifyTokenResp.isOK()) {
                    logger.error("verify token failed." + StringUtil.toJson(verifyTokenResp));
                }
                cacheService.set(cacheKey, expireTime, verifyTokenResp);
            }
            OauthContext.setAccessToken(token);
            logger.debug("setVerifyResult:{}",verifyTokenResp.result);
            OauthContext.setVerifyResult(verifyTokenResp.result);
            return verifyTokenResp.isOK();
        } catch (Exception e) {
            logger.error("verifyOauthToken error!", e);
            return false;
        }
    }

}
