package com.jsmframe.auth;

import com.jsmframe.annotation.RestAnn;
import com.jsmframe.consts.BasePairConsts;
import com.jsmframe.context.AppContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.exception.AuthFailException;
import com.jsmframe.exception.NoLoginException;
import com.jsmframe.exception.NoPermissionException;
import com.jsmframe.session.Session;
import com.jsmframe.utils.*;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;

@Component
public class AppAuth implements Auth {

    private Logger logger = LogUtil.log(AppAuth.class);

    @Override
    public String checkAuth(HttpServletRequest request, RestAnn resetAnn, boolean isTry) {
        String token = "";
        if (AppContext.isJedisSession()) {
            String at = RequestUtil.get(request, WebContext.JSM_AT);
            String ht = RequestUtil.get(request, WebContext.JSM_HT);
            logger.debug("checkSession, auth type {},at {}, ht {}", resetAnn.authType(), at, ht);
            token = decodeAppToken(at, ht, request);
            if (StringUtil.isEmpty(token)) {
                if (isTry) {
                    return token;
                } else {
                    throw new AuthFailException();
                }
            }
        }
        Session session = new Session(token);
        if (session.isExpire()) {
            if (isTry) {
                return token;
            } else {
                throw new NoLoginException();
            }
        } else {
            session.live();
            request.setAttribute(BasePairConsts.SESSION_KEY, session);
        }
        //APP WEB 判断是否有系统权限
        if (!isTry && !StringUtil.isEmpty(resetAnn.permission())) {
            if (!WebUtil.hasPermissions(resetAnn.permission())) {
                throw new NoPermissionException();
            }
        }
        return token;
    }

    private String decodeAppToken(String at, String ht, HttpServletRequest request) {
        String token = null;
        try {
            if (!StringUtil.isEmpty(at) && !StringUtil.isEmpty(ht)) {
                String decAuth = BlowFishUtil.dec(at);
                if (decAuth == null || !decAuth.contains(",")) {
                    logger.error("illegal at: {}", at);
                    return null;
                }
                String[] arr = decAuth.split(",");
                token = arr[0];
                String timestamp = arr[1];
                Long cTime = Long.valueOf(timestamp);
                long sTime = DateUtil.currentTime();
                Long res = Math.abs(sTime - cTime);
                Long expireTime = AppContext.getAsLong("at.expire.time");
                if (expireTime == null) {
                    expireTime = 60000L;//60s
                }
                if (res > expireTime) {
                    logger.error("sTime {} - cTime {} = {}", new Object[]{sTime, cTime, res});
                    return null;
                }

                String sAuth = BlowFishUtil.enc(token + "," + timestamp);
                if (!sAuth.equals(at)) {
                    logger.error("sAt {} != at {}", new Object[]{sAuth, at});
                    return null;
                }

                String sHash = EncryptUtil.md5(token + "," + timestamp);
                if (!sHash.equals(at)) {
                    logger.error("sHt {} != ht {}", new Object[]{sHash, at});
                    return null;
                }
                request.setAttribute("token", token);
            }
        } catch (Exception e) {
            logger.error("decodeAppToken error!", e);
            return null;
        }
        return token;
    }

}
