package com.jsmframe.aop;

import com.jsmframe.annotation.ClusterJobLockAnn;
import com.jsmframe.annotation.ClusterLockAnn;
import com.jsmframe.context.AppContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.jedis.JedisService;
import com.jsmframe.stat.model.Alarm;
import com.jsmframe.utils.*;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionException;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 分布式锁 切面
 */
@Aspect
@Component
public class ClusterJobLockAop {
    private static Logger logger = LogUtil.log(ClusterJobLockAop.class);

    private String clusterId = AppContext.get("cluster.id");
    private AtomicInteger reqIdAi = new AtomicInteger(0);

    @Resource
    private JedisService jedisService;

    @Around("@annotation(clusterJobLockAnn)")
    public Object intercept(ProceedingJoinPoint jpt, ClusterJobLockAnn clusterJobLockAnn) throws Throwable {
        Object result = null;
        String key = clusterJobLockAnn.value();
        if (StringUtil.isEmpty(key)) {
            key = getKey(jpt);
        }
        try {
            logger.debug("clusterJobLockAnn lock:{}", key);
            boolean locked = jedisService.lock(key, 0, clusterJobLockAnn.nextTime());
            if (!locked) {
                logger.warn("clusterJobLockAnn lock:{} failed, return.", key);
                return result;
            }
            //set reqId
            int intCount = reqIdAi.getAndIncrement();
            if (intCount + 100 > Integer.MAX_VALUE) {
                reqIdAi.set(0);
            }
            WebContext.setRequestId(String.format("job-%s-%s-%s", clusterId, key, intCount));
            result = jpt.proceed();
        } catch (Exception e) {
            logger.error("clusterJobLockAnn error,clusterJobLockAnn key:" + key, e);
            SystemUtil.addAlarm(new Alarm("ClusterJobLockAop",  key + ":Exception," + e.getMessage()));
        } finally {
            WebContext.clearRequestId();
        }
        return result;
    }

    private String getKey(ProceedingJoinPoint jpt) {
        StringBuilder sb = new StringBuilder();
        sb.append(jpt.toString());
        for (Object o : jpt.getArgs()) {
            sb.append("|").append(o.hashCode());
        }
        return sb.toString();
    }

}
