package com.jsmframe.aop;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.concurrent.locks.ReentrantLock;

import com.jsmframe.context.AppContext;
import com.jsmframe.service.CacheService;
import com.jsmframe.utils.TemplateUtil;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;

import com.jsmframe.annotation.CacheAnn;
import com.jsmframe.annotation.CacheType;
import com.jsmframe.context.SpringContext;
import com.jsmframe.jedis.JedisService;
import com.jsmframe.utils.LogUtil;
import com.jsmframe.utils.StringUtil;

import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;

@Aspect
@Component
public class CacheAop {
    private static Logger logger = LogUtil.log(CacheAop.class);

    /**
     * 一般缓存
     */
    @Resource
    private JedisService jedisService;
    @Resource
    private CacheService cacheService;

    private Cache queryCache;

    @Around("@annotation(cacheAnn)")
    public Object intercept(ProceedingJoinPoint jpt, CacheAnn cacheAnn) throws Throwable {
        Object result = null;
        Method method = ((MethodSignature) (jpt.getSignature())).getMethod();
        if (method.getReturnType().getName().equals("void")) {
            return jpt.proceed();
        }
        if (cacheAnn.cacheType().equals(CacheType.EHCACHE)) {
            result = ehcache(jpt, cacheAnn);
        } else if (cacheAnn.cacheType().equals(CacheType.REDIS)) {
            result = jedisCache(jpt, cacheAnn);
        } else if (cacheAnn.cacheType().equals(CacheType.EHCACHE_REDIS)) {//二级缓存
            result = ehRedisCache(jpt, cacheAnn);
        }
        return result;
    }

    private Object ehRedisCache(ProceedingJoinPoint jpt, CacheAnn cacheAnn) throws Throwable {
        Object result;
        String key = cacheAnn.cacheKey();
        if (StringUtil.isEmpty(key)) {
            key = getKey(jpt);
        } else {
            key = TemplateUtil.formatWithContextVar(key);
        }
        Method method = ((MethodSignature) (jpt.getSignature())).getMethod();
        result = cacheService.get(key, method.getGenericReturnType());
        if (result == null) {
            synchronized (jpt) {
                result = cacheService.get(key, method.getGenericReturnType());
                if (result == null) {
                    result = jpt.proceed();
                    cacheService.set(key, cacheAnn.expireSeconds(), (Serializable) result);
                    logger.debug("update ehRedisCache key: " + key);
                }
            }
        }
        return result;
    }

    private Object jedisCache(ProceedingJoinPoint jpt, CacheAnn cacheAnn) throws Throwable {
        Object result = null;
        String key = cacheAnn.cacheKey();
        if (StringUtil.isEmpty(key)) {
            key = getKey(jpt);
        } else {
            key = TemplateUtil.formatWithContextVar(key);
        }
        int expireTime = cacheAnn.expireSeconds();
        String jresult = jedisService.get(key);
        if (jresult == null) {
            synchronized (jpt) {
                jresult = jedisService.get(key);
                if (jresult == null) {
                    result = jpt.proceed();
                    jedisService.setex(key, expireTime, StringUtil.toJson(result));
                    logger.debug("update jedisCache key: " + key);
                } else {
                    result = parseReturnObject(jpt, jresult);
                }
            }
        } else {
            result = parseReturnObject(jpt, jresult);
        }
        return result;
    }

    private Object parseReturnObject(ProceedingJoinPoint jpt, String jresult) {
        Method method = ((MethodSignature) (jpt.getSignature())).getMethod();
        Object result = StringUtil.parseObject(jresult, method.getGenericReturnType());
        return result;
    }

    private Object ehcache(ProceedingJoinPoint jpt, CacheAnn cacheAnn) throws Throwable {
        Object result = null;
        String key = cacheAnn.cacheKey();
        if (StringUtil.isEmpty(key)) {
            key = getKey(jpt);
        } else {
            key = TemplateUtil.formatWithContextVar(key);
        }
        Element element = getQueryCache().get(key);
        if (element == null) {
            synchronized (jpt) {
                element = getQueryCache().get(key);
                if (element == null) {
                    result = jpt.proceed();
                    getQueryCache().put(new Element(key, (Serializable) result));
                    logger.debug("update ehcache key: " + key);
                } else {
                    result = element.getObjectValue();
                }
            }
        } else {
            result = element.getObjectValue();
        }
        return result;
    }

    private Cache getQueryCache() {
        if (queryCache == null) {
            String ehcacheName = AppContext.get("cache.service.cache.name");
            if (StringUtil.isEmpty(ehcacheName)) {
                ehcacheName = "queryCache";
            }
            queryCache = SpringContext.getBean(ehcacheName);
        }
        return queryCache;
    }

    private String getKey(ProceedingJoinPoint jpt) {
        StringBuilder sb = new StringBuilder();
        sb.append(jpt.toString());
        for (Object o : jpt.getArgs()) {
            sb.append("|").append(o.hashCode());
        }
        return sb.toString();
    }

}
