package com.jsmframe.utils;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;

import com.jsmframe.annotation.EscapeType;
import com.jsmframe.annotation.FieldAnn;
import com.jsmframe.annotation.FieldType;
import com.jsmframe.base.ValidateCallback;
import com.jsmframe.base.ValidateModel;
import com.jsmframe.context.ProjectContext;
import com.jsmframe.context.SpringContext;
import com.jsmframe.context.WebContext;

public class ValidateUtil {
	private static Logger logger = LogUtil.log(ValidateUtil.class);
	private static Map<String, ValidateCallback> cbCache = new HashMap<String, ValidateCallback>();
	
	@SuppressWarnings("unchecked")
	public static String validate(Object instance){
		String message = null;
		Field[] fields = instance.getClass().getDeclaredFields();
		for(Field field:fields){
			String name = field.getName();
			FieldAnn fieldAnn = field.getAnnotation(FieldAnn.class);
			if(fieldAnn == null){
				continue;
			}
			Object value = getFieldValue(field,instance);
			if(value == null && !fieldAnn.required()){
				continue;
			}
			
			if(ValidateModel.class.isAssignableFrom(field.getType())){
				if(fieldAnn.required() && value == null){
					message = SpringContext.getMessage("valid.required",name);
					return message;
				}
				if(value != null){
					String msg =  ValidateUtil.validate(value);
					if(!StringUtil.isEmpty(msg)){
						msg = name+"."+msg;
						return msg;
					}
				}
			}else if(List.class.isAssignableFrom(field.getType())){
				List<Object> reqList = (List<Object>)getFieldValue(field,instance);
				if(fieldAnn.required() && (reqList == null||reqList.isEmpty())){
					message = SpringContext.getMessage("valid.required",name);
					return message;
				}
				if(reqList != null){
					for(Object req:reqList){
						if(req != null && ValidateModel.class.isAssignableFrom(req.getClass())){
							String msg =  ValidateUtil.validate(req);
							if(!StringUtil.isEmpty(msg)){
								msg = name+"."+msg;
								return msg;
							}
						}
					}
				}
			}else{
				if(fieldAnn.required() && StringUtil.isEmpty(value)){
					message = SpringContext.getMessage("valid.required",name);
					break;
				}
				FieldType type = fieldAnn.type();
				if(!StringUtil.isEmpty(value) && !type.equals(FieldType.ALL)){//判断类型
					if(FieldType.NUMBER.equals(type) && !StringUtil.isNumber(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.EN_NUMBER.equals(type) && !StringUtil.isEnNum(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.CHINESE.equals(type) && !StringUtil.isChinese(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.DATE.equals(type) && !StringUtil.isDate(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.DATE_TIME.equals(type) && !StringUtil.isDateTime(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.EMAIL.equals(type) && !StringUtil.isEmail(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.ID_CARD.equals(type) && !StringUtil.isIdcard(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.IPV4.equals(type) && !StringUtil.isIpv4(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.MOB_NO.equals(type) && !StringUtil.isMobNo(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.TEL_NO.equals(type) && !StringUtil.isTelNo(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.MONEY.equals(type) && !StringUtil.isMoney(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.URL.equals(type) && !StringUtil.isUrl(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
					if(FieldType.BANK_CARD.equals(type) && !StringUtil.isBankCard(value.toString())){
						message = SpringContext.getMessage("valid.mustbe",name,type.t());
						break;
					}
				}
				
				String regex = fieldAnn.regex();
				if(!StringUtil.isEmpty(regex) && (value instanceof String) && !((String)value).matches(regex)){
					String msg = SpringContext.getMessage(fieldAnn.message());
					message = SpringContext.getMessage("valid.regexLimit",name,msg);
					break;
				}
				
				int minLen = fieldAnn.minLen();
				if(minLen != -1 && (value instanceof String) && ((String)value).length() < minLen){
					message =  SpringContext.getMessage("valid.minLenLimit",name,minLen);
					break;
				}
				
				int maxLen = fieldAnn.maxLen();
				if(maxLen != -1 && (value instanceof String) && ((String)value).length() > maxLen){
					message =  SpringContext.getMessage("valid.maxLenLimit",name,maxLen);
					break;
				}
				
				String el = fieldAnn.el();
				if(!StringUtil.isEmpty(el)){
					boolean res = exeEl(el,instance);
					if(!res){
						String msg = SpringContext.getMessage(fieldAnn.message());
						message = SpringContext.getMessage("valid.regexLimit",name,msg);
						break;
					}
				}
				
				String cb = fieldAnn.cb();
				if(!StringUtil.isEmpty(cb)){
					String res = exeCb(cb,BeanUtil.getFieldValue(field, instance),(ValidateModel)instance);
					if(!StringUtil.isEmpty(res)){
						message = SpringContext.getMessage("valid.regexLimit",name,res);
						break;
					}
				}
				
				//escape
				if(value instanceof String){
					String valueStr = (String)value;
					EscapeType escapeType = fieldAnn.escape();
					if(EscapeType.ALL.equals(escapeType)){
						setFieldValue(field,instance,HtmlUtil.filter(valueStr));
					}else if(EscapeType.CSS.equals(escapeType)){
						setFieldValue(field,instance,HtmlUtil.cssFilter(valueStr));
					}else if(EscapeType.JS.equals(escapeType)){
						setFieldValue(field,instance,HtmlUtil.jsFilter(valueStr));
					}else if(EscapeType.HTML.equals(escapeType)){
						setFieldValue(field,instance,HtmlUtil.htmlFilter(valueStr));
					}
				}
			}
		}
		
		return message;
	}
	
	private static String exeCb(String cb,Object fieldVal, ValidateModel vm) {
		try {
			String basePackage = ProjectContext.get("validator.package");
			if(!cb.contains(".")){
				if(StringUtil.isEmpty(basePackage)){
					logger.warn("validator.package not config in project.properties");
				}else{
					cb = basePackage + "." + cb;
				}
			}
			ValidateCallback vc = cbCache.get(cb);
			if(vc == null){
				Class<?> clazz = Class.forName(cb);
				if(!ValidateCallback.class.isAssignableFrom(clazz)){
					logger.error("cb class is not implements ValidateCallback, ignore.");
					return null;
				}
				vc = (ValidateCallback) clazz.newInstance();
				cbCache.put(cb, vc);
			}
			return vc.exe(fieldVal,vm);
		} catch (Exception e) {
			logger.error("validata cb class can't load."+cb, e);
		}
		return null;
	}

	public static boolean exeEl(String el,Object vm){
		ExpressionParser parser = new SpelExpressionParser();
		EvaluationContext context = new StandardEvaluationContext();
		context.setVariable("vm", vm);
		context.setVariable("session", WebContext.currentSession());
		context.setVariable("request", WebContext.getRequest());
		String elsStr = createExpression(el,vm);
		return parser.parseExpression(elsStr).getValue(context,Boolean.class);
	}
	
	private static String createExpression(String el,Object vm) {
		Pattern p = Pattern.compile("#\\{(.*?)\\}");
		Matcher m = p.matcher(el);
		Set<String> vars = new HashSet<String>();
		while(m.find()){
			vars.add(m.group(1));
		}
		for(String var:vars){
			String val = (String) BeanUtil.getFieldValue(var, vm);
			if(!StringUtil.isEmpty(val)){
				el = el.replaceAll("#\\{"+var+"\\}", val);
			}
		}
		return el;
	}

	private static Object getFieldValue(Field field,Object instance){
		try {
			//ReflectionUtil.getFieldValue(this, name)
			return field.get(instance);
		} catch (Exception e) {
			logger.error("getFieldValue error!", e);
			return null;
		}
	}
	private static void setFieldValue(Field field,Object instance, Object value){
		try {
			//ReflectionUtil.getFieldValue(this, name)
			field.set(instance,value);
		} catch (Exception e) {
			logger.error("getFieldValue error!", e);
		}
	}
}
