package com.jsmframe.oauth;

import java.util.Date;
import java.util.List;

import org.slf4j.Logger;

import com.alibaba.fastjson.JSON;
import com.jsmframe.context.ProjectContext;
import com.jsmframe.oauth.model.App;
import com.jsmframe.oauth.model.AppResourceCode;
import com.jsmframe.oauth.model.OauthInfo;
import com.jsmframe.utils.DateUtil;
import com.jsmframe.utils.HttpClientUtil;
import com.jsmframe.utils.LogUtil;
import com.jsmframe.utils.StringUtil;

public class OauthContext {
	
	private static Logger logger = LogUtil.log(OauthContext.class);
	private static OauthInfo oauthInfo = null;
	private static Date atExpireTime = new Date();
	private static Date rtExpireTime = new Date();
	private static ThreadLocal<AppResourceCode> currentAppResourceCodeTl = new ThreadLocal<AppResourceCode>();
	
	public static String getAccessToken(){
		//第一次或者rt过期了 就要去重新取token
		if(oauthInfo == null || DateUtil.reduce(rtExpireTime, DateUtil.currentDate(), DateUtil.MILLIS_PER_SECOND) < 10){
			oauthInfo = getAuthInfo();
			atExpireTime = DateUtil.addDay(DateUtil.currentDate(), oauthInfo.atExpireTime);
			rtExpireTime = DateUtil.addDay(DateUtil.currentDate(), oauthInfo.rtExpireTime);
		}else if(DateUtil.reduce(atExpireTime, DateUtil.currentDate(), DateUtil.MILLIS_PER_SECOND) < 10){//at过期了 使用rt刷新
			oauthInfo = getRefreshAuthInfo();
			atExpireTime = DateUtil.addDay(DateUtil.currentDate(), oauthInfo.atExpireTime);
			rtExpireTime = DateUtil.addDay(DateUtil.currentDate(), oauthInfo.rtExpireTime);
		}
		return oauthInfo != null? oauthInfo.accessToken:null;
	}
	
	private static OauthInfo getRefreshAuthInfo() {
		String refreshTokenUrl = ProjectContext.get("oauth.refreshToken.url");
		if(StringUtil.isEmpty(refreshTokenUrl)){
			logger.error("oauth.refreshToken.url is not config in ProjectContext.");
			return null;
		}
		String refreshUrl = refreshTokenUrl.replace("{refresh_token}", oauthInfo.refreshToken);
		String res = HttpClientUtil.get(refreshUrl);
		logger.info("oauth info:{}",res);
		if(!StringUtil.isEmpty(res)){
			return JSON.parseObject(res, OauthInfo.class);
		}else{
			logger.error("refresh token error!,oauth.refreshToken.url {}",refreshTokenUrl);
		}
		return null;
	}


	private static OauthInfo getAuthInfo(){
		App app = new App(); 
		app.appKey = ProjectContext.get("oauth.appKey");
		app.appSecret = ProjectContext.get("oauth.appSecret");
		if(StringUtil.isEmpty(app.appKey)){
			logger.error("oauth.appKey is not config in ProjectContext.");
			return null;
		}
		if(StringUtil.isEmpty(app.appSecret)){
			logger.error("oauth.appSecret is not config in ProjectContext.");
			return null;
		}
		String getTokenUrl = ProjectContext.get("oauth.getToken.url");
		if(StringUtil.isEmpty(getTokenUrl)){
			logger.error("oauth.getToken.url is not config in ProjectContext.");
			return null;
		}
		String res = HttpClientUtil.post(getTokenUrl,JSON.toJSONString(app));
		logger.info("oauth info:{}",res);
		if(!StringUtil.isEmpty(res)){
			return JSON.parseObject(res, OauthInfo.class);
		}else{
			logger.error("get token error!,oauth.getToken.url {}",getTokenUrl);
		}
		return null;
	}
	
	public static List<AppResourceCode> getAuthorizedResources(){
		if(oauthInfo == null){
			logger.error("oauthInfo is null, maybe you need call open method first.");
			return null;
		}
		return oauthInfo.authorizedResources;
	}
	
	public static void setCurrentAppResourceCode(AppResourceCode appResourceCode){
		currentAppResourceCodeTl.set(appResourceCode);
	}
	
	public static AppResourceCode getCurrentAppResourceCode(){
		return currentAppResourceCodeTl.get();
	}
	
}
