package com.jsmframe.interceptor;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.jsmframe.annotation.AuthType;
import com.jsmframe.annotation.RestAnn;
import com.jsmframe.consts.BasePairConsts;
import com.jsmframe.context.ProjectContext;
import com.jsmframe.context.SpringContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.jedis.JedisService;
import com.jsmframe.oauth.OauthContext;
import com.jsmframe.oauth.model.AppResourceCode;
import com.jsmframe.rest.resp.RestResp;
import com.jsmframe.session.Session;
import com.jsmframe.utils.*;
import org.slf4j.Logger;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Method;

public class JsmRestInterceptor extends HandlerInterceptorAdapter {
	
	private static Logger logger = LogUtil.log(JsmRestInterceptor.class);

	@Override
	public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
		//设置跨域
        response.setHeader("Access-Control-Allow-Origin", "*");
        response.setHeader("Access-Control-Allow-Methods", "POST, GET, OPTIONS, DELETE");
        response.setHeader("Access-Control-Max-Age", "3600");
        response.setHeader("Access-Control-Allow-Headers", "x-requested-with");
		
        HandlerMethod hm = (HandlerMethod) handler;
//		Object target = hm.getBean();
//		Class<?> clazz = hm.getBeanType();
		Method method = hm.getMethod();
		RestAnn resetAnn = method.getAnnotation(RestAnn.class);
		if (resetAnn == null) {
			return true;
		}
		
		RestResp<String> baseResp = new RestResp<String>();
		// 判断是否要登录 判断权限是否够
		if (!resetAnn.authType().equals(AuthType.NONE)) {
			baseResp = checkAuth(request, resetAnn);
		}

		if (!baseResp.isOK()) {
			handleError(request, response, baseResp);
			return false;
		}

		//事务放AOP里
		
		WebContext.setRequest(request);
		WebContext.setResponse(response);
		
		return true;
	}

	private RestResp<String> checkAuth(HttpServletRequest request, RestAnn resetAnn) {
		
		RestResp<String> baseResp = checkSession(request,resetAnn);
		if(!baseResp.isOK()){
			return baseResp;
		}
		
		return baseResp;
	}

	private RestResp<String> checkSession(HttpServletRequest request, RestAnn resetAnn) {
		String auth = request.getHeader(WebContext.JSM_AT);
		String hash = request.getHeader(WebContext.JSM_HT);
		if(StringUtil.isEmpty(auth) && StringUtil.isEmpty(hash)){
			auth = request.getParameter(WebContext.JSM_AT);
			hash = request.getParameter(WebContext.JSM_HT);
		}
		if(StringUtil.isEmpty(auth) && StringUtil.isEmpty(hash)){
			Cookie[] cookies = request.getCookies();
			if(cookies != null){
				for(Cookie cookie:cookies){
					if(cookie.getName().equals(WebContext.JSM_AT)){
						auth = cookie.getValue();
					}
					if(cookie.getName().equals(WebContext.JSM_HT)){
						hash = cookie.getValue();
					}
				}
			}
		}
		String authorization = request.getHeader(WebContext.JSM_AUTHORIZATION);
		logger.debug("checkSession, auth type {},at {}, ht {},authorization {}",resetAnn.authType(),auth,hash,authorization);
		RestResp<String> baseResp = new RestResp<String>();
		String token = null;
		
		//OAUTH认证
		if(AuthType.OAUTH.equals(resetAnn.authType())){//OAUTH
			if(!StringUtil.isEmpty(authorization)){
				token = authorization.replace("Bearer", "").trim();
				String resourceCode = resetAnn.permission();
				//add cache
				boolean res = verifyOauthToken(token,resourceCode);
				//
				if(!res){
					baseResp.setPair(BasePairConsts.AUTH_FAIL);
				}
				return baseResp;
			}
		}
		
		//APP 或者 WEB认证
		if(!StringUtil.isEmpty(auth) && !StringUtil.isEmpty(hash)){
			if(AuthType.APP.equals(resetAnn.authType())){//APP
				token = verifyAppApi(auth,hash,request);
			}else{//WEB
				token = verifyWebApi(auth,hash,request);
			}
		}
		
		if(ProjectContext.isDevModel() && StringUtil.isEmpty(token)){
			token = request.getHeader(WebContext.JSM_TOKEN);
			if(StringUtil.isEmpty(token)){
				token = request.getParameter(WebContext.JSM_TOKEN);
			}
		}
		
		if(StringUtil.isEmpty(token)){
			baseResp.setPair(BasePairConsts.AUTH_FAIL);
		}else{
			//fetch session
			//check token
			Session session = new Session(token);
			if(session.isExpire()){
				baseResp.setPair(BasePairConsts.NO_LOGIN);
			}else{
				session.live();
				request.setAttribute(BasePairConsts.SESSION_KEY, session);
			}
		}
		
		//APP WEB 判断是否有系统权限
		if(!StringUtil.isEmpty(resetAnn.permission())){
			if (!WebUtil.hasPermissions(resetAnn.permission())) {
				baseResp.setPair(BasePairConsts.NO_PERMISSION);
			}
		}
		
		return baseResp;
	}

	//再访问URL 验证Token
	private String getVerifyOauthTokenResult(String token,String resourceCode) {
		String authUrl = ProjectContext.get("oauth.verifyToken.url");
		if(StringUtil.isEmpty(authUrl)){
			logger.error("oauth.verifyToken.url not found, in ProjectContext!");
			return null;
		}
		try {
			authUrl = authUrl.replace("{access_token}", token);
			authUrl = authUrl.replace("{resource_code}", resourceCode);
			String res = HttpClientUtil.get(authUrl);
			logger.debug("res:{}", res);
			return res;
		} catch (Exception e) {
			logger.error("getVerifyOauthTokenResult error:"+authUrl,e);
			return null;
		}
	}
	
	//先取缓存 如果没有 再访问URL
	private boolean verifyOauthToken(String token,String resourceCode) {
		try {
			JedisService jedisService = SpringContext.getBean(JedisService.class);
			String cacheKey = token+"_"+resourceCode;
			String res = jedisService.get(cacheKey);
			if(StringUtil.isEmpty(res)){
				res = getVerifyOauthTokenResult(token,resourceCode);
				Integer expireTime = ProjectContext.getAsInteger("oauth.verifyToken.expireTime");
				if(expireTime == null){
					expireTime = 5;//cache 5s
					logger.warn("oauth.verifyToken.expireTime not config, use default:{}",expireTime);
				}
				jedisService.setex(cacheKey, expireTime, res+"");
			}
			
			JSONObject resp = JSON.parseObject(res);
			if(BasePairConsts.OK.getCode().equals(resp.getString("code"))){//success
				String result = resp.getString("result");
				AppResourceCode appResourceCode = JSON.parseObject(result, AppResourceCode.class);
				OauthContext.setCurrentAppResourceCode(appResourceCode);
				return true;
			}else{
				logger.error("verifyOauthToken token:{} , resourceCode:{}",token,resourceCode);
				return false;
			}
		} catch (Exception e) {
			logger.error("verifyOauthToken error!",e);
			return false;
		}
	}

	private void handleError(HttpServletRequest request, HttpServletResponse response, RestResp<String> baseResp) {
		PrintWriter writer = null;
		try {
	        response.setCharacterEncoding("UTF-8");
	        response.setContentType("application/json; charset=UTF-8");
	        writer = response.getWriter();
	        logger.error("error:{}",baseResp);
            writer.print(baseResp.toString());
		} catch (IOException e) {
			logger.error("response ajax data error!", e);
		} finally {
            if (writer != null)
                writer.close();
        }
	}

	/**
	 * @param auth
	 * @param request 
	 * @return
	 */
	private String verifyWebApi(String auth, String hash, HttpServletRequest request) {
		logger.debug("auth: {}",auth);// base64(token_timestamp)
		logger.debug("hash: {}",hash);// md5(token_timestamp)
		String token = null;
		try {
			String decAuth = EncryptUtil.base64Decode(auth);
			if(decAuth == null || !decAuth.contains("_")){
				logger.error("illegal auth: {}", auth);
				return null;
			}
			String[] arr = decAuth.split("_");
			token = arr[0];
			String timestamp = arr[1];
			Long ctime = Long.valueOf(timestamp);
			long sTime = DateUtil.currentTime();
			Long res = Math.abs(sTime - ctime);
			Long expireTime = ProjectContext.getAsLong("at.expire.time");
			if(expireTime == null){
				expireTime = 60000L;//60s
			}
			if(res > expireTime ){
				logger.error("sTime {} - ctime {} = {}",new Object[]{sTime,ctime,res});
				return null;
			}
			
			String sAuth = EncryptUtil.base64Encode(token+"_"+timestamp);
			if(!sAuth.equals(auth)){
				logger.error("sAuth {} != auth {}",new Object[]{sAuth,auth});
				return null;
			}

			String sHash = EncryptUtil.md5(token+"_"+timestamp);
			if(!sHash.equals(hash)){
				logger.error("sHash {} != hash {}",new Object[]{sHash,hash});
				return null;
			}
			request.setAttribute("token", token);
		} catch (Exception e) {
			logger.error("verifyWebApi error!", e);
			return null;
		}
		
		return token;
	}
	
	/**
	 * @param auth
	 * @param request 
	 * @return
	 */
	private String verifyAppApi(String auth, String hash, HttpServletRequest request) {
		logger.debug("auth: {}",auth);// blowfish(token,timestamp)
		logger.debug("hash: {}",hash);//md5(token,timestamp)
		String token = null;
		try {
			String decAuth = BlowFishUtil.dec(auth);
			if(decAuth == null || !decAuth.contains(",")){
				logger.error("illegal auth: {}", auth);
				return null;
			}
			String[] arr = decAuth.split(",");
			token = arr[0];
			String timestamp = arr[1];
			Long ctime = Long.valueOf(timestamp);
			long sTime = DateUtil.currentTime();
			Long res = Math.abs(sTime - ctime);
			Long expireTime = ProjectContext.getAsLong("at.expire.time");
			if(expireTime == null){
				expireTime = 60000L;//60s
			}
			if(res > expireTime ){
				logger.error("sTime {} - ctime {} = {}",new Object[]{sTime,ctime,res});
				return null;
			}
			
			String sAuth = BlowFishUtil.enc(token+","+timestamp);
			if(!sAuth.equals(auth)){
				logger.error("sAuth {} != auth {}",new Object[]{sAuth,auth});
				return null;
			}
			
			String sHash = EncryptUtil.md5(token+","+timestamp);
			if(!sHash.equals(hash)){
				logger.error("sHash {} != hash {}",new Object[]{sHash,hash});
				return null;
			}
			request.setAttribute("token", token);
		} catch (Exception e) {
			logger.error("verifyAppApi error!", e);
			return null;
		}
		
		return token;
	}
	
	@Override
	public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
			ModelAndView modelAndView) throws Exception {
//		logger.info("postHandle");
//		Session session = (Session) request.getAttribute(CommonConsts.SESSION_KEY);
//		session.flush();
	}

	@Override
	public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex)
			throws Exception {
//		logger.info("afterCompletion");
//		WebContext.removeRequest();
//		WebContext.removeResponse();
	}

}
