package com.jsmframe.context;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.jsmframe.utils.StringUtil;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.slf4j.Logger;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.context.request.ServletWebRequest;

import com.alibaba.fastjson.JSON;
import com.jsmframe.consts.BasePairConsts;
import com.jsmframe.rest.resp.RestResp;
import com.jsmframe.session.Session;
import com.jsmframe.session.SessionUser;
import com.jsmframe.utils.DateUtil;
import com.jsmframe.utils.LogUtil;

/**
 * web工具类
 */
public class WebContext {
    private static Logger logger = LogUtil.log(WebContext.class);
    public static String JSM_REQ_ID = "jsm_req_id";
    public static String JSM_AT = "at";
    public static String JSM_HT = "ht";
    public static String JSM_TOKEN = "_token";
    public static String JSM_AUTHORIZATION = "Authorization";
    public static String NO_AUTH_HEADER = "no-auth-header";
    private static ThreadLocal<HttpServletRequest> reqTL = new ThreadLocal<HttpServletRequest>();
    private static ThreadLocal<HttpServletResponse> resTL = new ThreadLocal<HttpServletResponse>();
    private static ThreadLocal<String> jspPath = new ThreadLocal<String>();
    private static AtomicInteger reqIdAi = new AtomicInteger(0);
    private static String clsId = "";
    private static String clusterId = ProjectContext.get("cluster.id");

    public static void setRequest(HttpServletRequest request) {
        reqTL.set(request);
    }

    public static HttpServletRequest getRequest() {
        if (reqTL.get() != null) {
            return reqTL.get();
        } else {
            return getJspRequest();
        }
    }

    public static String getRequestId() {
        HttpServletRequest request = getRequest();
        String reqId = null;
        if (request != null) {
            reqId = request.getHeader(WebContext.JSM_REQ_ID);
            if (reqId == null) {
                reqId = request.getParameter(WebContext.JSM_REQ_ID);
            }
            if (reqId == null) {
                reqId = (String) request.getAttribute(WebContext.JSM_REQ_ID);
            }
            if (reqId == null) {//如果还没有则生成个
                int intCount = reqIdAi.getAndIncrement();
                if (intCount + 1000 > Integer.MAX_VALUE) {
                    reqIdAi.set(0);
                }
                if (StringUtil.isEmpty(clsId) && !StringUtil.isEmpty(clusterId)) {
                    clsId = "jri-" + clusterId + "-";
                }
                reqId = clsId + String.valueOf(intCount);
                request.setAttribute(WebContext.JSM_REQ_ID, reqId);
            }
        }
        return reqId;
    }

    public static String getRequestAt() {
        HttpServletRequest request = getRequest();
        String at = null;
        if (request != null && StringUtil.isEmpty(request.getAttribute(NO_AUTH_HEADER))) {
            at = request.getHeader(WebContext.JSM_AT);
            if (at == null) {
                at = request.getParameter(WebContext.JSM_AT);
            }
            if (at == null) {
                if (at == null) {
                    Cookie[] cookies = request.getCookies();
                    if (cookies != null) {
                        for (Cookie cookie : cookies) {
                            if (cookie.getName().equals(WebContext.JSM_AT)) {
                                at = cookie.getValue();
                            }
                        }
                    }
                }
            }
        }
        return at;
    }

    public static String getRequestHt() {
        HttpServletRequest request = getRequest();
        String ht = null;
        if (request != null && StringUtil.isEmpty(request.getAttribute(NO_AUTH_HEADER))) {
            ht = request.getHeader(WebContext.JSM_HT);
            if (ht == null) {
                ht = request.getParameter(WebContext.JSM_HT);
            }
            if (ht == null) {
                if (ht == null) {
                    Cookie[] cookies = request.getCookies();
                    if (cookies != null) {
                        for (Cookie cookie : cookies) {
                            if (cookie.getName().equals(WebContext.JSM_HT)) {
                                ht = cookie.getValue();
                            }
                        }
                    }
                }
            }
        }
        return ht;
    }

    public static String getRequestToken() {
        HttpServletRequest request = getRequest();
        String token = null;
        if (request != null && StringUtil.isEmpty(request.getAttribute(NO_AUTH_HEADER))) {
            token = request.getHeader(WebContext.JSM_TOKEN);
            if (StringUtil.isEmpty(token)) {
                token = request.getParameter(WebContext.JSM_TOKEN);
            }
        }
        return token;
    }

    public static String getRequestAuthorization() {
        HttpServletRequest request = getRequest();
        String authorization = null;
        if (request != null && StringUtil.isEmpty(request.getAttribute(NO_AUTH_HEADER))) {
            authorization = request.getHeader(WebContext.JSM_AUTHORIZATION);
        }
        return authorization;
    }

    private static HttpServletRequest getJspRequest() {
        ServletRequestAttributes servletRequestAttributes = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes());
        return servletRequestAttributes == null ? null : servletRequestAttributes.getRequest();
    }

    public static String currentSessionId() {
        Session session = currentSession();
        return session == null ? null : session.getToken();
    }

    public static Session currentSession() {
        HttpServletRequest request = getRequest();
        return (Session) (request == null ? null : request.getAttribute(BasePairConsts.SESSION_KEY));
    }

    public static void removeRequest() {
        reqTL.remove();
    }

    public static void setResponse(HttpServletResponse response) {
        resTL.set(response);
    }

    public static HttpServletResponse getResponse() {
        if (resTL.get() != null) {
            return resTL.get();
        } else {
            return getJspResponse();
        }
    }

    public static HttpServletResponse getJspResponse() {
        ServletWebRequest servletWebRequest = ((ServletWebRequest) RequestContextHolder.getRequestAttributes());
        return servletWebRequest == null ? null : servletWebRequest.getResponse();
    }

    public static void removeResponse() {
        resTL.remove();
    }

    public static void redirect(String action) throws IOException {
        getResponse().sendRedirect(getContextPath() + action);
    }

    public static void toErrorPage(String msg) {
//		currentSession().set("error", msg);
        try {
            if (WebContext.isAjax() || currentSession() == null || WebContext.isRest()) {
                sendAjaxError(msg);
            } else {
                redirect(ProjectContext.get("error.url"));
            }
        } catch (Exception e) {
            logger.error("redirect error page fail!", e);
        }
    }

    private static boolean isRest() {
        String prefix = getRequest().getContextPath() + "/rest";
        return getRequest().getRequestURI().startsWith(prefix);
    }

    private static void sendAjaxError(String msg) {
        RestResp<String> baseResp = new RestResp<String>();
        baseResp.setPair(BasePairConsts.ERROR);
        baseResp.result = msg;
        try {
            getResponse().addHeader("Content-Type", "text/json;charset=UTF-8");
            getResponse().addHeader("Content-Type", "application/json;charset=UTF-8");
            getResponse().getWriter().write(JSON.toJSONString(baseResp));
        } catch (Exception e) {
            logger.error("sendAjaxError error!", e);
        }
    }

    private static String contextPath;

    private static String realPath;

    public static String getRealPath() {
        return realPath;
    }

    public static void setRealPath(String realPath) {
        WebContext.realPath = realPath;
    }

    public static String getContextPath() {
        return contextPath;
    }

    public static void setContextPath(String contextPath) {
        WebContext.contextPath = contextPath;
    }


    /**
     * @param ext 返回一个随机产生指定扩展名的文件名
     */
    public static String getRandomFileName(String type, String ext) {
        StringBuilder sb = new StringBuilder();
        sb.append(type).append("_").append(DateUtil.format(new Date(), "yyyyMMddHHmmss"));
        sb.append("_").append(new Random().nextInt(1000)).append(".").append(ext);
        return sb.toString();
    }

    /**
     * 保存excel对象并返回保存后的文件名
     */
    public static String saveExcel(HSSFWorkbook workbook) throws IOException {
        String fileName = getRandomFileName("temp", "xls");
        OutputStream os = new FileOutputStream(fileName);
        workbook.write(os);
        os.close();
        return fileName.substring(fileName.lastIndexOf("\\") + 1);
    }

    /**
     * 取得当前用户
     */
    public static SessionUser currentUser() {
        Session session = currentSession();
        if (session == null) {
            return null;
        } else {
            return session.getSessionUser();
        }
    }

    /**
     * 取得当前用户权限
     */
    public static List<String> currentPermissions() {
        SessionUser sessionUser = currentUser();
        if (sessionUser == null) {
            return null;
        }
        return sessionUser.getPermissions();
    }

    /**
     * 取得当前用户角色
     */
    public static List<Long> currentRoles() {
        SessionUser sessionUser = currentUser();
        if (sessionUser == null) {
            return null;
        }
        return sessionUser.getRoles();
    }

    /**
     * 有些问题
     */
    public static boolean isAjax() {
        String type = getRequest().getHeader("X-Requested-With");
        return "XMLHttpRequest".equals(type);
    }

    public static String getRemoteIp() {
        HttpServletRequest request = getRequest();
        if (request == null) {
            return null;
        }
        return getRemoteIp(request);
    }

    public static String getRemoteIp(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Real-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        if (ip != null && ip.contains(",")) {
            ip = ip.split(",")[0];
        }
        return ip;
    }

    public static void setJspPath(String path) {
        jspPath.set(path);
    }

    public static String getJspPath() {
        String path = jspPath.get();
        jspPath.remove();
        return path;
    }

    public static String getBackUrl() {
        if ("GET".equals(WebContext.getRequest().getMethod())) {
            return getActionUri();
        } else {
            String url = WebContext.getRequest().getHeader("REFERER");
            return url.substring(url.indexOf("=") + 1);
        }
    }

    public static byte[] getRequestPostBytes(HttpServletRequest request) {
        try {
            int contentLength = request.getContentLength();
            if (contentLength < 0) {
                return null;
            }
            byte buffer[] = new byte[contentLength];
            for (int i = 0; i < contentLength; ) {

                int readlen = request.getInputStream().read(buffer, i, contentLength - i);
                if (readlen == -1) {
                    break;
                }
                i += readlen;
            }
            return buffer;
        } catch (IOException e) {
            logger.error("getRequestPostBytes error!", e);
            return null;
        }
    }

    public static String getRequestPostStr(HttpServletRequest request) {
        try {
            byte buffer[] = getRequestPostBytes(request);
            String charEncoding = request.getCharacterEncoding();
            if (charEncoding == null) {
                charEncoding = "UTF-8";
            }
            if (buffer != null) {
                return new String(buffer, charEncoding);
            }
        } catch (Exception e) {
            logger.error("getRequestPostStr error!", e);
        }
        return null;
    }

    /**
     * 取得相对于当前系统actionURI
     */
    public static String getActionUri() {
        return WebContext.getRequest().getRequestURI().substring(WebContext.getContextPath().length());
    }
}
