package com.jsmframe.aop;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletOutputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.jsmframe.jedis.JedisService;
import com.jsmframe.rest.resp.*;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.web.multipart.MultipartFile;

import com.alibaba.fastjson.JSON;
import com.jsmframe.annotation.RestAnn;
import com.jsmframe.annotation.Transaction;
import com.jsmframe.base.ValidateModel;
import com.jsmframe.consts.BasePairConsts;
import com.jsmframe.context.AppContext;
import com.jsmframe.context.SpringContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.dao.model.OptLog;
import com.jsmframe.service.LogService;
import com.jsmframe.session.Session;
import com.jsmframe.session.SessionUser;
import com.jsmframe.utils.EncryptUtil;
import com.jsmframe.utils.HttpClientUtil;
import com.jsmframe.utils.LogUtil;
import com.jsmframe.utils.StringUtil;
import com.jsmframe.utils.TransactionUtil;
import com.jsmframe.utils.ValidateUtil;

/**
 *@author longzl / @createOn 2010-9-26
 *@desc
 */

public class RestAop {
	private static Logger logger = LogUtil.log(RestAop.class);
	private PlatformTransactionManager transactionManager;
	
	private static String REQUEST_PREFIX = "request.";
	private static String SESSION_PREFIX = "session.";
	
	public Object intercept(ProceedingJoinPoint jpt) throws Throwable {
		Object result = null;
		Method method = ((MethodSignature) (jpt.getSignature())).getMethod();
		RestAnn resetAnn = method.getAnnotation(RestAnn.class);
		if(resetAnn==null){
			return jpt.proceed();
		}
		//参数验证
		if(resetAnn.validate()){
			String message = validate(jpt);
			if(!StringUtil.isEmpty(message)){
				RestResp<String> baseResp = new RestResp<String>();
				baseResp.setPair(BasePairConsts.PARAMS_ERROR);
				baseResp.result = message;
				logger.info("resp:"+JSON.toJSONString(baseResp));
				return baseResp;
			}
		}

		if(resetAnn.sync()){
			synchronized (method) {
				result = process(jpt, result, method, resetAnn);
			}
		}else{
			result = process(jpt, result, method, resetAnn);
		}


		
		//记录日志
		if(!StringUtil.isEmpty(resetAnn.log())){
			OptLog optLog = new OptLog();
			optLog.ip = WebContext.getRemoteIp();
			
			Session session = WebContext.currentSession();
			optLog.mac = session != null?(String)session.get("mac"):"";
			optLog.funcName = resetAnn.logType();
			
			SessionUser sessionUser = WebContext.currentUser();
			optLog.uid = sessionUser != null?sessionUser.getUid().toString():"";
			optLog.optDesc = formatLog(resetAnn.log());
			optLog.time = new Date();
			LogService logService = SpringContext.getBean(LogService.class);
			String loggerUrl = AppContext.get("logger.url");
			if(logService != null){
				logService.addLog(optLog);
			}else if(StringUtil.isEmpty(loggerUrl)){
				HttpClientUtil.post(loggerUrl, optLog);
			}else{
				logger.error("LogService or logger.url not found! you can implement LogService Or add a logger.url in app.properties");
			}
		}
		
		return result;
	}

	private String formatLog(String log) {
		Pattern p = Pattern.compile("#\\{(.*?)\\}");
		Matcher m = p.matcher(log);
		Set<String> vars = new HashSet<String>();
		while(m.find()){
			vars.add(m.group(1));
		}
		for(String var:vars){
			String val = getLogVarValue(var);
			if(!StringUtil.isEmpty(val)){
				log = log.replaceAll("#\\{"+var+"\\}", val);
			}
		}
		return log;
	}

	private String getLogVarValue(String var) {
		if(var.startsWith(REQUEST_PREFIX)){
			return StringUtil.toString(WebContext.getRequest().getAttribute(var.replace(REQUEST_PREFIX, "")));
		}else if(var.startsWith(SESSION_PREFIX)){
			return StringUtil.toString(WebContext.currentSession().get(var.replace(SESSION_PREFIX, "")));
		}else{
			String value = StringUtil.toString(WebContext.getRequest().getAttribute(var));
			if(StringUtil.isEmpty(value)){
				value = StringUtil.toString(WebContext.currentSession().get(var));
			}
			return value;
		}
		
	}

	private Object process(ProceedingJoinPoint jpt, Object result, Method method, RestAnn resetAnn)
			throws Throwable, Exception {

		//事务处理
		TransactionStatus status = null;
		JedisService jedisService = null;
		try {

			if (!StringUtil.isEmpty(resetAnn.clusterSyncLock())){
				jedisService = SpringContext.getBean(JedisService.class);
				jedisService.lock(resetAnn.clusterSyncLock());
			}

			if (!resetAnn.transaction().equals(Transaction.NONE)) {
				String transManagerName = "transactionManager";
				if(!StringUtil.isEmpty(resetAnn.transManagerName())){
					transManagerName = resetAnn.transManagerName();
				}
				this.transactionManager = ((PlatformTransactionManager) SpringContext.getBean(transManagerName));
				status = this.transactionManager.getTransaction(createTransactionDefinition(resetAnn));
				TransactionUtil.setTransactionStatus(status);
			}
			result = execMethod(jpt);
			if(status != null && !status.isCompleted()){
				if(status.isRollbackOnly()){
					logger.warn("transaction rollback!");
					this.transactionManager.rollback(status);
				}else{
					this.transactionManager.commit(status);
				}
			}
		} catch (Exception e) {
			if(status != null){
				logger.error("error: transaction rollback!", e);
				this.transactionManager.rollback(status);
			}
			result = dealException(e,result,method);
		}finally {
			TransactionUtil.removeTransactionStatus();
			if (jedisService != null){
				jedisService.unlock(resetAnn.clusterSyncLock());
			}
		}
		logger.info("resp:"+JSON.toJSONString(result));
		return result;
	}

	private Object execMethod(ProceedingJoinPoint jpt) throws Throwable {
		Object result =  jpt.proceed();
		if(result == null ){
			return result;
		}
		if(result instanceof FileResp){
			FileResp fileResp = (FileResp) result;
			outputFile(fileResp);
			logger.info("FileResp:{}",fileResp.fileName);
		}else if(result instanceof TextResp){
			TextResp textResp = (TextResp) result;
			outputText(textResp);
			logger.info("textResp:{}",textResp.text);
		}else if(result instanceof JsonResp){
			JsonResp jsonResp = (JsonResp) result;
			outputJson(jsonResp);
			logger.info("jsonResp:{}",jsonResp.jsonStr);
		}else if(result instanceof XmlResp){
			XmlResp xmlResp = (XmlResp) result;
			outputXml(xmlResp);
			logger.info("xmlResp:{}",xmlResp.xml);
		}else if(result instanceof HtmlResp){
			HtmlResp htmlResp = (HtmlResp) result;
			outputHtml(htmlResp);
			logger.info("htmlResp:{}",htmlResp.html);
		}else if(result instanceof ExcelResp){
			ExcelResp excelResp = (ExcelResp) result;
			outputExcel(excelResp);
			logger.info("excelResp:{}",excelResp.fileName);
		}else if(result instanceof FreeResp){
			FreeResp freeResp = (FreeResp) result;
			outputFreeResp(freeResp);
			logger.info("freeResp: type:{} content:{}",freeResp.contentType,freeResp.content);
		}else if(result instanceof RestResp){
			return result;
		}else{
			if(result instanceof String){
				String res = (String) result;
				if(res.startsWith("redirect:")){
					WebContext.getResponse().sendRedirect(res.substring(9));
				}
				if(res.startsWith("forward:")){
					ServletRequest request = WebContext.getRequest();
					ServletResponse response = WebContext.getResponse();
					request.getRequestDispatcher(res.substring(8)).forward(request,response);
				}
			}
            logger.warn("unknown response type!");
            return result;
		}
		return null;
		
	}
	
	private void outputExcel(ExcelResp excelResp) {
		HttpServletResponse response = WebContext.getResponse();
		response.setHeader("Content-Disposition", "attachment; filename="+EncryptUtil.urlEncode(excelResp.fileName));
		response.setContentType("application/octet-stream;charset=UTF-8");
//		response.setContentLength(excelResp.workbook);
		if(excelResp.workbook != null){
			try {
				excelResp.workbook.write(response.getOutputStream());
			} catch (IOException e) {
				logger.error("write workbook error! {}",excelResp.fileName);
			}finally{
				try {
					excelResp.workbook.close();
				} catch (IOException e) {
					logger.error("close workbook error! {}",excelResp.fileName);
				}
			}
		}
	}

	private void outputXml(XmlResp xmlResp){
		PrintWriter pw = null;
		try {
			WebContext.getResponse().setContentType("text/xml;charset=UTF-8");
			pw = WebContext.getResponse().getWriter();
			pw.write(xmlResp.xml);
			pw.flush();
			pw.close();
		} catch (IOException e) {
			logger.error("outputXml error!", e);
		}finally {
			if(pw != null){
				pw.close();
			}
		}
	}

	private void outputHtml(HtmlResp htmlResp){
		PrintWriter pw = null;
		try {
			WebContext.getResponse().setContentType("text/html;charset=UTF-8");
			pw = WebContext.getResponse().getWriter();
			pw.write(htmlResp.html);
			pw.flush();
			pw.close();
		} catch (IOException e) {
			logger.error("htmlResp error!", e);
		}finally {
			if(pw != null){
				pw.close();
			}
		}
	}

	private void outputFreeResp(FreeResp freeResp){
		PrintWriter pw = null;
		try {
			WebContext.getResponse().setContentType(freeResp.contentType);
			pw = WebContext.getResponse().getWriter();
			pw.write(freeResp.content);
			pw.flush();
			pw.close();
		} catch (IOException e) {
			logger.error("freeResp error!", e);
		}finally {
			if(pw != null){
				pw.close();
			}
		}
	}

	private void outputText(TextResp textResp){
		PrintWriter pw = null;
		try {
			WebContext.getResponse().setContentType("text/plain;charset=UTF-8");
			pw = WebContext.getResponse().getWriter();
			pw.write(textResp.text);
			pw.flush();
			pw.close();
		} catch (IOException e) {
			logger.error("outputText error!", e);
		}finally {
			if(pw != null){
				pw.close();
			}
		}
	}
	
	private void outputJson(JsonResp jsonResp){
		PrintWriter pw = null;
		try {
			WebContext.getResponse().setContentType("application/json;charset=UTF-8");
			pw = WebContext.getResponse().getWriter();
			pw.write(jsonResp.jsonStr);
			pw.flush();
			pw.close();
		} catch (IOException e) {
			logger.error("outputJson error!", e);
		}finally {
			if(pw != null){
				pw.close();
			}
		}
	}
	
	private void outputFile(FileResp fileResp) {
		HttpServletResponse response = WebContext.getResponse();
		response.setHeader("Content-Disposition", "attachment; filename="+EncryptUtil.urlEncode(fileResp.fileName));
		response.setCharacterEncoding("UTF-8");
		ServletOutputStream sos = null;
		InputStream is = null;
		PrintWriter pw = null;
		try {
			
			if(fileResp.outputObject instanceof String){
				pw = response.getWriter();
				pw.print((String) fileResp.outputObject);
			}else if(fileResp.outputObject instanceof InputStream){
				sos = response.getOutputStream();
				int offset = 0;
				byte[] bt = new byte[1024];
				is = (InputStream) fileResp.outputObject;
				while ((offset = is.read(bt)) != -1) {
					sos.write(bt, 0, offset);
				}
			}else{//Other Object
				pw = response.getWriter();
				pw.print(JSON.toJSONString(fileResp.outputObject));
			}
		} catch (Exception e) {
			logger.error("outputFile error!"+fileResp.fileName, e);
		}finally {
			try {
				if(sos != null){
					sos.flush();
					sos.close();
				}
				if(is != null){
					is.close();
				}
				if(pw != null){
					pw.close();
				}
			} catch (IOException e) {
				logger.error("outputFile error!"+fileResp.fileName, e);
			}
		}
		
//		WebContext.removeRequest();
//		WebContext.removeResponse();
	}

	private String validate(ProceedingJoinPoint jpt) {
		String message = null;
		Object[] args = jpt.getArgs();
		if(args != null && args.length > 0){
			for(Object arg:args){
				if(arg instanceof ValidateModel){
					logger.info("request arg:{}",JSON.toJSONString(arg));
					message = ValidateUtil.validate(arg);
					if(!StringUtil.isEmpty(message)){
						break;
					}
				}else{
					logger.info("request arg type:{}",arg.getClass().getName());
				}
			}
		}
		return message;
	}

	private RestResp<String> dealException(Exception e,Object result,Method method) throws Exception{
		logger.error(BasePairConsts.ERROR.toString(), e);
		RestResp<String> baseResp = new RestResp<String>();
		baseResp.setPair(BasePairConsts.ERROR);
		baseResp.result = e.getMessage();
		return baseResp;
	}

	private DefaultTransactionDefinition createTransactionDefinition(RestAnn restAnn) {
		DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
		transactionDefinition.setPropagationBehavior(restAnn.transPropagationBehavior());
		transactionDefinition.setIsolationLevel(restAnn.transIsolationLevel());
		transactionDefinition.setReadOnly(restAnn.transaction().equals(Transaction.READ)?true:false);
		transactionDefinition.setTimeout(600);//seconds
		return transactionDefinition;
	}

}
