package com.jsmframe.aop;

import java.io.Serializable;
import java.lang.reflect.Method;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;

import com.alibaba.fastjson.JSON;
import com.jsmframe.annotation.CacheAnn;
import com.jsmframe.annotation.CacheType;
import com.jsmframe.context.SpringContext;
import com.jsmframe.jedis.JedisService;
import com.jsmframe.utils.LogUtil;
import com.jsmframe.utils.StringUtil;

import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;

/**
 *@author longzl / @createOn 2011-8-31
 *@desc
 */
public class CacheAop {
	private static Logger logger = LogUtil.log(CacheAop.class);

	/** 一般缓存*/
	private Cache ehQueryCache;
	/** 永久缓存*/
	private Cache ehEnternalCache;
	
	private JedisService jedisService;

	public Object intercept(ProceedingJoinPoint jpt) throws Throwable {
		Object result = null;
		Method method = ((MethodSignature) (jpt.getSignature())).getMethod();
		CacheAnn cacheAnn = method.getAnnotation(CacheAnn.class);
		if (cacheAnn == null) {
			return jpt.proceed();
		} else {
			synchronized (method) {
				if(cacheAnn.cacheType().equals(CacheType.EHCACHE)){
					result = ehcache(jpt, cacheAnn);
				}else{
					result = jedisCache(jpt, cacheAnn);
				}
			}
			return result;
		}

	}

	private Object jedisCache(ProceedingJoinPoint jpt, CacheAnn cacheAnn)  throws Throwable {
		Object result;
		String key = cacheAnn.cacheKey();
		if(StringUtil.isEmpty(key)){
			key = getKey(jpt);
		}
		int expireTime = cacheAnn.expireSeconds();
		String jresult = getJedisService().get(key);
		if(jresult == null){
			result = jpt.proceed();
			getJedisService().setex(key, expireTime,JSON.toJSONString(result));
			logger.debug("update jedis key: " + key);
		}else{
			Method method = ((MethodSignature) (jpt.getSignature())).getMethod();
			Class<?> returnType = method.getReturnType();
			result = JSON.parseObject(jresult, returnType);
		}		
		return result;
	}

	private Object ehcache(ProceedingJoinPoint jpt, CacheAnn cacheAnn) throws Throwable {
		Object result;
		String key = cacheAnn.cacheKey();
		if(StringUtil.isEmpty(key)){
			key = getKey(jpt);
		}
		if (cacheAnn.eternal()) {
			Element element = getEhEnternalCache().get(key);
			if (element == null) {
				result = jpt.proceed();
				getEhEnternalCache().put(new Element(key, (Serializable) result));
				logger.debug("update eternalCache key: " + key);
			} else {
				result = element.getObjectValue();
			}
		} else {
			Element element = getEhQueryCache().get(key);
			if (element == null) {
				result = jpt.proceed();
				getEhQueryCache().put(new Element(key, (Serializable) result));
				logger.debug("update cache key: " + key);
			} else {
				result = element.getObjectValue();
			}
		}
		return result;
	}

	private String getKey(ProceedingJoinPoint jpt) {
		StringBuilder sb = new StringBuilder();
		sb.append(jpt.toString());
		for (Object o : jpt.getArgs()) {
			sb.append("|").append(o.hashCode());
		}
		return sb.toString();
	}

	public Cache getEhQueryCache() {
		if(this.ehQueryCache == null){
			this.ehQueryCache = SpringContext.getBean("ehQueryCache");
		}
		return ehQueryCache;
	}

	public Cache getEhEnternalCache() {
		if(this.ehEnternalCache == null){
			this.ehEnternalCache = SpringContext.getBean("ehEnternalCache");
		}
		return ehEnternalCache;
	}

	public JedisService getJedisService() {
		if(this.jedisService == null){
			this.jedisService = SpringContext.getBean("jedisService");
		}
		return jedisService;
	}
	
}
