package com.jsmframe.session;

import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSON;
import com.jsmframe.context.ProjectContext;
import com.jsmframe.context.SpringContext;
import com.jsmframe.context.WebContext;
import com.jsmframe.jedis.JedisService;
import com.jsmframe.utils.StringUtil;

public class Session {
	private static Logger logger = LoggerFactory.getLogger(Session.class);
	private String token;
	private JedisService jedisService;
	private SessionUser sessionUser;
	private HttpSession session;
	private boolean invalid = false;
	
	private static String SESSION_USER = "SESSION_USER";
	private static String SESSION_USER_KEY_PREFIX = "SESSION_USER_KEY_PFRFIX_";
	
	public static Session create(String token) {
		Session session = new Session(token);
		session.set("sid", token);
		session.live();
		return session;
	}
	
	public Session(String token) {
		super();
		this.token = token;
		if(ProjectContext.isJedisSession()){
			this.jedisService = SpringContext.getBean(JedisService.class);
		}else{
			session = WebContext.getRequest().getSession();
		}
	}
	
	public void set(String key,String value){
		if(ProjectContext.isJedisSession()){
			jedisService.hset(this.token, key,value);
		}else{
			session.setAttribute(key, value);
		}
	}
	
	/**
	 * @param key
	 * @return if jedis will string
	 */
	public Object get(String key){
		if(ProjectContext.isJedisSession()){
			return jedisService.hget(this.token, key);
		}else{
			return session.getAttribute(key);
		}
	}
	
	public void del(String key){
		if(ProjectContext.isJedisSession()){
			jedisService.hdel(this.token, key);
		}else{
			session.removeAttribute(key);
		}
	}
	
	/**
	 *  jedis only
	 */
	public void live(){
		logger.debug("live check!");
		if(ProjectContext.isJedisSession() && !this.invalid){
			logger.debug("live");
			Integer liveTime = ProjectContext.getAsInteger("session.timeout");
			jedisService.expire(this.token, liveTime);
			if(this.sessionUser != null){
				String ukey = SESSION_USER_KEY_PREFIX + this.sessionUser.getUid();
				jedisService.expire(ukey, liveTime);
			}
		}
	}

	public String getToken() {
		return token;
	}

	public SessionUser getSessionUser() {
		if(this.sessionUser == null){
			if(ProjectContext.isJedisSession()){
				String str = (String) get(SESSION_USER);
				if(!StringUtil.isEmpty(str)){
					this.sessionUser = JSON.parseObject(str, SessionUser.class);
				}
			}else{
				this.sessionUser = (SessionUser) session.getAttribute("sessionUser");
			}
		}
		return sessionUser;
	}

	public void setSessionUser(SessionUser sessionUser) {
		this.sessionUser = sessionUser;
		if(ProjectContext.isJedisSession()){
			set(SESSION_USER, JSON.toJSONString(sessionUser));
			String ukey = SESSION_USER_KEY_PREFIX + this.sessionUser.getUid();
			jedisService.set(ukey, this.token);
			logger.debug("setSessionUser ukey:{} , token:{}",ukey,token);
			this.live();
		}else{
			session.setAttribute(SESSION_USER, sessionUser);
		}
	}

	public boolean isExpire() {
		if(this.invalid){
			return true;
		}
		if(ProjectContext.isJedisSession()){
			return !jedisService.exists(token);
		}
		return false;
	}

	public void flush() {
		if(!this.invalid){
			setSessionUser(sessionUser);
		}
	}

	public void invalid() {
		this.invalid = true;
		if(ProjectContext.isJedisSession()){
			jedisService.del(this.token);
			String ukey = "";
			if(this.getSessionUser() != null){
				ukey = SESSION_USER_KEY_PREFIX + this.getSessionUser().getUid();
				jedisService.del(ukey);
			}
			logger.debug("invalid ukey:{} , token:{}",ukey,token);
		}
	}

	public static boolean expire(Long uid) {
		if(ProjectContext.isJedisSession()){
			JedisService jedisService = SpringContext.getBean(JedisService.class);
			String ukey = SESSION_USER_KEY_PREFIX + uid;
			String token = jedisService.get(ukey);
			jedisService.del(ukey);
			if(!StringUtil.isEmpty(token)){
				jedisService.del(token);
			}
			logger.debug("expire ukey:{} , token:{}",ukey,token);
		}else{
//			
		}
		return true;
	}

	public static boolean isLogon(Long uid) {
		if(ProjectContext.isJedisSession()){
			JedisService jedisService = SpringContext.getBean(JedisService.class);
			String ukey = SESSION_USER_KEY_PREFIX + uid;
			boolean res = jedisService.exists(ukey);
			logger.debug("uid:{}, isLogon:{}",uid,res);
			return res;
		}else{
//			
		}
		return false;
	}
	
}
