package com.jsmframe.context;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.List;
import java.util.Random;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.slf4j.Logger;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.context.request.ServletWebRequest;

import com.alibaba.fastjson.JSON;
import com.jsmframe.consts.BasePairConsts;
import com.jsmframe.rest.resp.RestResp;
import com.jsmframe.session.Session;
import com.jsmframe.session.SessionUser;
import com.jsmframe.utils.DateUtil;
import com.jsmframe.utils.LogUtil;

/**
 * web工具类
 * 
 */
public class WebContext {
	private static Logger logger = LogUtil.log(WebContext.class);

	private static ThreadLocal<HttpServletRequest> reqTL = new ThreadLocal<HttpServletRequest>();
	private static ThreadLocal<HttpServletResponse> resTL = new ThreadLocal<HttpServletResponse>();
	private static ThreadLocal<String> jspPath = new ThreadLocal<String>();

	public static void setRequest(HttpServletRequest request) {
		reqTL.set(request);
	}

	public static HttpServletRequest getRequest() {
		if(reqTL.get() != null){
			return reqTL.get();
		}else{
			return getJspRequest();
		}
	}
	
	private static HttpServletRequest getJspRequest() {
		ServletRequestAttributes servletRequestAttributes = ((ServletRequestAttributes)RequestContextHolder.getRequestAttributes());
		return servletRequestAttributes==null?null:servletRequestAttributes.getRequest();
	}
	
	public static Session currentSession(){
		HttpServletRequest request = getRequest();
		return (Session) (request==null?null:request.getAttribute(BasePairConsts.SESSION_KEY));
	}

	public static void removeRequest() {
		reqTL.remove();
	}

	public static void setResponse(HttpServletResponse response) {
		resTL.set(response);
	}
	
	public static HttpServletResponse getResponse() {
		if(resTL.get() != null){
			return resTL.get();
		}else{
			return getJspResponse();
		}
	}
	
	public static HttpServletResponse getJspResponse() {
		ServletWebRequest servletWebRequest = ((ServletWebRequest)RequestContextHolder.getRequestAttributes());
		return servletWebRequest==null?null:servletWebRequest.getResponse();
	}

	public static void removeResponse() {
		resTL.remove();
	}

	public static void redirect(String action) throws IOException{
		getResponse().sendRedirect(getContextPath()+action);
	}
	
	public static void toErrorPage(String msg){
//		currentSession().set("error", msg);
		try {
			if(WebContext.isAjax() || currentSession() == null || WebContext.isRest()){
				sendAjaxError(msg);
			}else{
				redirect(ProjectContext.get("error.url"));
			}
		} catch (Exception e) {
			logger.error("redirect error page fail!", e);
		}
	}

	private static boolean isRest() {
		String prefix = getRequest().getContextPath()+"/rest";
		return getRequest().getRequestURI().startsWith(prefix);
	}

	private static void sendAjaxError(String msg) {
		RestResp<String> baseResp = new RestResp<String>();
		baseResp.setPair(BasePairConsts.ERROR);
		baseResp.result = msg;
		try {
			getResponse().addHeader("Content-Type", "text/json;charset=UTF-8");
			getResponse().addHeader("Content-Type", "application/json;charset=UTF-8");
			getResponse().getWriter().write(JSON.toJSONString(baseResp));
		} catch (IOException e) {
			logger.error("sendAjaxError error!", e);
		}
	}

	private static String contextPath;

	private static String realPath;

	public static String getRealPath() {
		return realPath;
	}

	public static void setRealPath(String realPath) {
		WebContext.realPath = realPath;
	}

	public static String getContextPath() {
		return contextPath;
	}

	public static void setContextPath(String contextPath) {
		WebContext.contextPath = contextPath;
	}

	/**
	 * 通过真实路径获得上下文路径
	 */
	public static String realPath2ContextPath(String realPath) {
//		String fileName = realPath.substring(realPath.lastIndexOf("\\") + 1);
		return getContextPath() + "/download?fileName=" + realPath;
	}
	
	public static String realPath2ContextPath2(String realPath) {
		logger.info("realPath:"+realPath);
		return getContextPath() + "/download?fileName=" + realPath;
	}

	/**
	 * @param ext
	 *            返回一个随机产生指定扩展名的文件名
	 */
	public static String getRandomFileName(String type, String ext) {
		StringBuilder sb = new StringBuilder();
		sb.append(type).append("_").append(DateUtil.format(new Date(), "yyyyMMddHHmmss"));
		sb.append("_").append(new Random().nextInt(1000)).append(".").append(ext);
		return sb.toString();
	}

	/**
	 * @param type
	 * @param fileName
	 *            返回一个文件真实路径
	 */
//	public static String getFilePath(String fileName) {
//		return getUploadDir()+File.separator + fileName;
//	}
	
//	public static String getUploadDir(){
//		WebPropService webPropService = SpringContext.getBean(WebPropService.class);
//		String dir = webPropService.getVal("upload.dir");
//		if(StringUtils.isEmpty(dir)){
//			dir = getRealPath()+ "WEB-INF"+File.separator+"upload";
//			logger.warn("upload.dir is unset,use default dir: {}",dir);
//		}
//		File file = new File(dir);
//		if(!file.exists()){
//			try {
//				file.mkdir();
//			} catch (Exception e) {
//				logger.error("make uploadDir {} error",dir);
//			}
//		}
//		return dir;
//	}

	/**
	 * 保存excel对象并返回保存后的文件名
	 */
	public static String saveExcel(HSSFWorkbook workbook) throws IOException {
		String fileName = getRandomFileName("temp", "xls");
		OutputStream os = new FileOutputStream(fileName);
		workbook.write(os);
		os.close();
		return fileName.substring(fileName.lastIndexOf("\\") + 1);
	}

	/**
	 * 取得当前用户
	 */
	public static SessionUser currentUser() {
		Session session = currentSession();
		if (session == null) {
			return null;
		} else {
			return session.getSessionUser();
		}
	}

	/**
	 * 取得当前用户权限
	 */
	public static List<String> currentPermissions() {
		SessionUser sessionUser = currentUser();
		if (sessionUser == null) {
			return null;
		}
		return sessionUser.getPermissions();
	}
	/**
	 * 取得当前用户角色
	 */
	public static List<Long> currentRoles() {
		SessionUser sessionUser = currentUser();
		if (sessionUser == null) {
			return null;
		}
		return sessionUser.getRoles();
	}

	/**
	 * 有些问题
	 */
	public static boolean isAjax() {
		String type = getRequest().getHeader("X-Requested-With");
		return "XMLHttpRequest".equals(type);
	}
	
	public static String getRmoteIp() {
		HttpServletRequest request = getRequest();
		String ip = request.getHeader("x-forwarded-for");
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("Proxy-Client-IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("WL-Proxy-Client-IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getRemoteAddr();
		}
		return ip;
	}

	public static void setJspPath(String path) {
		jspPath.set(path);
	}
	public static String getJspPath(){
		String path = jspPath.get();
		jspPath.remove();
		return path;
	}

	public static String getBackUrl() {
		if ("GET".equals(WebContext.getRequest().getMethod())) {
			return getActionUri();
		} else {
			String url = WebContext.getRequest().getHeader("REFERER");
			return url.substring(url.indexOf("=") + 1);
		}
	}
	
    public static byte[] getRequestPostBytes(HttpServletRequest request) {  
        try {
			int contentLength = request.getContentLength();  
			if(contentLength<0){  
			    return null;  
			}  
			byte buffer[] = new byte[contentLength];  
			for (int i = 0; i < contentLength;) {  
  
			    int readlen = request.getInputStream().read(buffer, i,contentLength - i);  
			    if (readlen == -1) {  
			        break;  
			    }  
			    i += readlen;  
			}  
			return buffer;
		} catch (IOException e) {
			logger.error("getRequestPostBytes error!", e);
			return null;
		}  
    }  
  
    public static String getRequestPostStr(HttpServletRequest request) {  
        try {
			byte buffer[] = getRequestPostBytes(request);  
			String charEncoding = request.getCharacterEncoding();  
			if (charEncoding == null) {  
			    charEncoding = "UTF-8";  
			}
			if(buffer != null){
				return new String(buffer, charEncoding);
			}
		} catch (Exception e) {
			logger.error("getRequestPostStr error!", e);
		}  
        return null;
    }  
	
	/**
	 * 取得相对于当前系统actionURI
	 */
	public static String getActionUri() {
		return WebContext.getRequest().getRequestURI().substring(WebContext.getContextPath().length());
	}
}
