package cn.ibaijia.isocket.session;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.ibaijia.isocket.Context;
import cn.ibaijia.isocket.protocol.Protocol;
import cn.ibaijia.isocket.util.BufferUtil;

public class Session<T> {
    private static final Logger logger = LoggerFactory.getLogger(Session.class);
    private static final byte SESSION_STATUS_OPENED = 1;
    private static final byte SESSION_STATUS_CLOSING = 2;
    private static final byte SESSION_STATUS_CLOSED = 3;

    private String sessionId;
    private Context<T> context;

    private SocketChannel channel;
    private ByteBuffer readBuffer;
    private byte status = SESSION_STATUS_OPENED;
    private Object attachment;
    private SocketAddress localAddress;
    private SocketAddress remoteAddress;
    /**
     * 响应消息缓存队列。 queue ehcache redis?
     */
    private Queue<ByteBuffer> writeCacheQueue;

    private volatile boolean writeLocked = false;
    private volatile boolean readLocked = false;

    private ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();

    public Session(SocketChannel channel, Context context) {
        this.channel = channel;
        this.context = context;
        try {
            this.sessionId = SessionManager.genId(channel);
            if (context.isUseCompactQueue()) {
                if (context.getCompactBuffSize() > 100) {
                    writeCacheQueue = new CompactBufferQueue(context.getCompactBuffSize());
                } else {
                    writeCacheQueue = new CompactBufferQueue();
                }
            } else {
                writeCacheQueue = new ConcurrentLinkedQueue<ByteBuffer>();
            }
            readBuffer = BufferUtil.allocate(context.getReadBuffSize(), context.isUseDirectBuffer());
            context.getSessionListener().onCreate(this);
        } catch (Exception e) {
            logger.error("create session error!", e);
        }
    }

    public void writeNext() {
        logger.debug("writeNext");
        if (this.status == SESSION_STATUS_CLOSED) {// closing也会 继续写
            logger.info("writeNext give up, session status:{}", this.status);
            return;
        }
        if (!writeCacheQueue.isEmpty()) {// 还有就写下一条
            addReadWriteKey();
        } else {
            addReadKey();
        }
        check();
    }

    public void readNext() {
        logger.debug("readNext");
        if (this.status != SESSION_STATUS_OPENED) {// closing 也不会读了
            logger.info("readNext give up, session status:{}", this.status);
            return;
        }
        if (!writeCacheQueue.isEmpty()) {// 还有就写下一条
            addReadWriteKey();
        } else {
            addReadKey();
        }
    }

    private void check() {
        // 触发流控 先写数据
        int cacheSize = writeCacheQueue.size();
        if (cacheSize > context.getWriteWarnLimit()) {
            context.getSessionListener().writeWarn(this, cacheSize);
        }
//        logger.info("readBuffer,pos:{} limit:{}, writeCacheQueue:{}", readBuffer.position(), readBuffer.limit(), writeCacheQueue.size());
//        SessionManager.status();
    }

    private void addWriteKey() {
        try {
            if (context.getSelector() == null) {
                SessionManager.close(this);
            }
            if (!this.isWriteLocked()) {
                logger.debug("addWriteKey");
                context.setSelectionKey(channel, SelectionKey.OP_WRITE);
            }
        } catch (Exception e) {
            logger.error("addWriteKey error!", e);
        }
    }

    private void addReadKey() {
        try {
            if (context.getSelector() == null) {
                SessionManager.close(this);
            }
            if (!this.isReadLocked()) {
                logger.debug("addReadKey");
                context.setSelectionKey(channel, SelectionKey.OP_READ);
            }
        } catch (Exception e) {
            logger.error("addReadKey error!", e);
        }
    }

    private void addReadWriteKey() {
        try {
            if (context.getSelector() == null) {
                SessionManager.close(this);
            }
            if (!this.isReadLocked()) {
                logger.debug("addReadWriteKey");
                context.setSelectionKey(channel, SelectionKey.OP_READ | SelectionKey.OP_WRITE);
            }
        } catch (Exception e) {
            logger.error("addReadWriteKey error!", e);
        }
    }

    /**
     * 加到队列最后 等待写出 写到当前 writeBuffer
     */
    private void write(ByteBuffer buffer) {
        buffer.flip();
        writeCacheQueue.add(buffer);
        addWriteKey();
    }

    /**
     * 强制关闭当前
     */
    public void close() {
        close(true);
    }

    /**
     * 是否立即关闭会话
     *
     * @param immediate true:立即关闭,false:响应消息发送完后关闭
     */
    public synchronized void close(boolean immediate) {
        if (channel == null) {
            return;
        }
        this.status = immediate ? SESSION_STATUS_CLOSED : SESSION_STATUS_CLOSING;
        if (immediate) {
            try {
                channel.shutdownInput();
            } catch (IOException e) {
                logger.debug(e.getMessage(), e);
            }
            try {
                channel.shutdownOutput();
            } catch (IOException e) {
                logger.debug(e.getMessage(), e);
            }
            try {
                channel.close();
                channel = null;
            } catch (IOException e) {
                logger.error("close session exception", e);
            }
            context.getSessionListener().closed(this);
        } else if (writeCacheQueue.size() == 0) {
            close(true);
            context.getSessionListener().closed(this);
        } else {
            context.getSessionListener().closing(this);
        }
    }

    /**
     * 触发通道的读操作，当发现存在严重消息积压时,会触发流控
     */
    @SuppressWarnings("unchecked")
	public void processReadBuffer() {
        readWriteLock.readLock().lock();
        try {
            int readLength = channel.read(readBuffer);
            logger.debug("read length:{}", readLength);
            context.getSessionListener().readComplete(this, readLength);
            if (readLength == -1) {
                logger.info("session:{} read complete.length:{}", this.getSessionID(), readLength);
                SessionManager.close(this);
            } else {
                if (readLength > 0) {
                    readBuffer.flip();
                    T dataEntry;
                    while (true) {
                        dataEntry = (T)decode(readBuffer);
                        if (dataEntry == null) {
                            logger.debug("break decode.");
                            break;
                        }
                        // 处理消息
                        try {
                            context.getSessionListener().beforeProcess(this, dataEntry);
                            boolean success = context.getProcessor().process(this, dataEntry);
                            context.getSessionListener().afterProcess(this, dataEntry);
                            if (success) {
                                context.getSessionListener().processSuccess(this, dataEntry);
                            } else {
                                context.getSessionListener().processFailed(this, dataEntry, null);
                            }
                        } catch (Exception e) {
                            context.getSessionListener().processFailed(this, dataEntry, e);
                        }
                    }

                    if (status == SESSION_STATUS_CLOSING) {
                        close(false);
                        return;
                    }
                    if (status == SESSION_STATUS_CLOSED) {
                        return;
                    }

                    // 数据读取完毕 直接清空 readBuffer.position() readBuffer.limit()
                    if (readBuffer.remaining() == 0) {
                        readBuffer.clear();// pos to 0 limit 变为 capacity
                    } else if (readBuffer.position() > 0) {// 还剩下部分数据
                        readBuffer.compact();// pos 不变，limit 变为 capacity
                    } else {// readBuffer.position() == 0 还原flip前的 状态 decode中不处理
                        // 读取buffer的情况下
                        readBuffer.position(readBuffer.limit());
                        readBuffer.limit(readBuffer.capacity());
                    }
//                    check();
                }
            }
        } catch (Exception e) {
            logger.error("nio read fail:", e);
            context.getSessionListener().readFailed(this, readBuffer, e);
            SessionManager.close(this);
        }
        readWriteLock.readLock().unlock();
    }

    private Object decode(ByteBuffer readBuffer) {
        List<Protocol> protocolList = context.getProtocolList();
        Object inputData = readBuffer;
        for (int i = protocolList.size() - 1; i >= 0; i--) {
            inputData = protocolList.get(i).decode(inputData, this);
        }
        return inputData;
    }

    private ByteBuffer encode(Object object) {
        List<Protocol> protocolList = context.getProtocolList();
        Object outputData = object;
        for (int i = 0; i < protocolList.size(); i++) {
            outputData = protocolList.get(i).encode(outputData, this);
        }
        if (outputData instanceof ByteBuffer) {
            return (ByteBuffer) outputData;
        } else {
            logger.error("protocol not encode to ByteBuffer,please check protocol chain.");
            SessionManager.close(this);
            return null;
        }
    }

    public void write(Object t) {
        write(encode(t));
    }

    public String getSessionID() {
        return this.sessionId;
    }

    public Context<T> getContext() {
        return context;
    }

    public <T> T getAttachment() {
        return (T) attachment;
    }

    public <T> void setAttachment(T attachment) {
        this.attachment = attachment;
        logger.info("setAttachment:{}", attachment.getClass());
    }

    public SocketAddress getLocalAddress() {
        if (this.localAddress == null && this.channel != null) {
            try {
                this.localAddress = channel.getLocalAddress();
            } catch (IOException e) {
                logger.error("getLocalAddress error!", e);
            }
        }
        return this.localAddress;
    }

    public SocketAddress getRemoteAddress() {
        if (this.remoteAddress == null && this.channel != null) {
            try {
                this.remoteAddress = channel.getRemoteAddress();
            } catch (IOException e) {
                logger.error("getRemoteAddress error!", e);
            }
        }
        return remoteAddress;
    }

    public void writeBuffer() {
        readWriteLock.writeLock().lock();
        if (!writeCacheQueue.isEmpty()) {
            ByteBuffer byteBuffer = writeCacheQueue.peek();
            try {
                int result = channel.write(byteBuffer);
                logger.debug("pos:" + byteBuffer.position() + " limit:" + byteBuffer.limit() + " result:" + result
                        + " remain:" + byteBuffer.remaining());
                if (!byteBuffer.hasRemaining()) {
                    writeCacheQueue.poll();
                    logger.debug("write completed:{}", result);
                    context.getSessionListener().writeComplete(this, result);
                } else {
                    logger.info("send:" + result);
                }
            } catch (Exception e) {
                logger.error("writeFailed error!", e);
                context.getSessionListener().writeFailed(this, byteBuffer, e);
                SessionManager.close(this);
            }
        }
        readWriteLock.writeLock().unlock();
    }

    public SocketChannel getChannel() {
        return channel;
    }

    public synchronized boolean isWriteLocked() {
        return writeLocked;
    }

    public synchronized void setWriteLocked(boolean writeLocked) {
        this.writeLocked = writeLocked;
    }

    public synchronized boolean isReadLocked() {
        return readLocked;
    }

    public synchronized void setReadLocked(boolean readLocked) {
        this.readLocked = readLocked;
    }
}
