package cn.ibaijia.isocket;

import cn.ibaijia.isocket.handler.ReadCompletionHandler;
import cn.ibaijia.isocket.handler.WriteCompletionHandler;
import cn.ibaijia.isocket.processor.Processor;
import cn.ibaijia.isocket.handler.AcceptCompletionHandler;
import cn.ibaijia.isocket.protocol.Protocol;
import cn.ibaijia.isocket.session.Session;
import cn.ibaijia.isocket.session.SessionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketOption;
import java.nio.channels.*;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.*;

public class Server<T> extends Context implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(Server.class);

    private Map<SocketOption<Object>, Object> options = new HashMap<SocketOption<Object>, Object>();
    private String host;
    private int port;
    private int backlog = 1000;

    private ServerSocketChannel serverSocketChannel = null;
    private ExecutorService executorService = null;

    public Server(String host, int port, Protocol protocol, Processor<T> processor) {
        this.host = host;
        this.port = port;
        this.addProtocol(protocol);
        this.processor = processor;
    }

    public Server(String host, int port) {
        this.host = host;
        this.port = port;
    }

    /**
     * 启动
     */
    public void start() {
        if (this.protocolList.isEmpty() || this.processor == null) {
            logger.error("protocol or processor can't be empty");
            return;
        }
        startInfo();
        logger.info("start server host:{},port:{},threads:{}", this.host, this.port, threadNumber);
        try {
            executorService = Executors.newFixedThreadPool(threadNumber, new ThreadFactory() {
                byte index = 0;

                @Override
                public Thread newThread(Runnable runnable) {
                    return new Thread(runnable, "is-nio-s" + (++index));
                }
            });
            executorService.execute(this);
        } catch (Exception e) {
            logger.error("server start error!", e);
            shutdown();
        }
    }


    @Override
    public void run() {
        try {
            this.serverSocketChannel = ServerSocketChannel.open();
            //set isocket options
            if (!options.isEmpty()) {
                for (Map.Entry<SocketOption<Object>, Object> entry : options.entrySet()) {
                    this.serverSocketChannel.setOption(entry.getKey(), entry.getValue());
                }
            }
            serverSocketChannel.configureBlocking(false);
            //bind host
            if (host != null) {
                serverSocketChannel.bind(new InetSocketAddress(host, port), backlog);
            } else {
                serverSocketChannel.bind(new InetSocketAddress(port), backlog);
            }
            selector = Selector.open();
            acceptAndProcess();
        } catch (Exception e) {
            logger.error("server run error!", e);
            shutdown();
        }
    }

    private void acceptAndProcess() {
        try {
            serverSocketChannel.register(selector, SelectionKey.OP_ACCEPT);
            while (true) {
                logger.debug("select!");
                selector.select();
                Set<SelectionKey> readyKeys = selector.selectedKeys();
                logger.debug("key size:" + readyKeys.size());
                Iterator<SelectionKey> it = readyKeys.iterator();
                while (it.hasNext()) {
                    SelectionKey key = it.next();
                    it.remove();
                    if (key.isAcceptable()) {
                        processAcceptKey(key);
                        serverSocketChannel.register(selector, SelectionKey.OP_ACCEPT);
                    } else if (key.isReadable()) {
                        processReadKey(key);
                    } else if (key.isWritable()) {
                        processWriteKey(key);
                    } else if (key.isValid()) {
                        logger.debug("key isValid");
                        SocketChannel socketChannel = (SocketChannel) key.channel();
                        SessionManager.close(socketChannel);
                    }
                }
                regEvent();
            }
        } catch (Exception e) {
            logger.error("unknown error!", e);
            shutdown();
        }
    }

    private void processWriteKey(SelectionKey key) throws IOException {
        logger.debug("key isWritable");
        SocketChannel socketChannel = (SocketChannel) key.channel();
        Session session = null;
        try {
            socketChannel.configureBlocking(false);
            session = SessionManager.get(socketChannel);
            if (session != null) {
                if (!session.isWriteLocked()) {
                    session.setWriteLocked(true);
                    executorService.execute(new WriteCompletionHandler<T>(session, socketChannel));
                } else {
                    logger.debug("isWriteLocked");
                }
            }
        } catch (Exception e) {
            logger.error("processWriteKey error", e);
            if(session != null){
                SessionManager.close(session);
            }
        }
    }

    private void processReadKey(SelectionKey key) {
        logger.debug("key isReadable");
        SocketChannel socketChannel = (SocketChannel) key.channel();
        if(socketChannel == null){
            return;
        }
        Session session = null;
        try {
            socketChannel.configureBlocking(false);
            session = SessionManager.get(socketChannel);
            if (session != null) {
                if (!session.isReadLocked()) {
                    session.setReadLocked(true);
                    executorService.execute(new ReadCompletionHandler<T>(session, socketChannel));
                } else {
                    logger.debug("isReadLocked");
                }
            }
        } catch (Exception e) {
            logger.error("processReadKey error", e);
            if(session != null){
                SessionManager.close(session);
            }
        }
    }

    private void processAcceptKey(SelectionKey key) {
        try {
            serverSocketChannel = (ServerSocketChannel) key.channel();
            SocketChannel socketChannel = serverSocketChannel.accept().socket().getChannel();
            socketChannel.configureBlocking(false);
            logger.debug("key isAcceptable");
            //check blacklist
            if (isBlackList(socketChannel)) {
                return;
            }
            if (socketChannel.isConnectionPending()) {
                socketChannel.finishConnect();
            }
            executorService.execute(new AcceptCompletionHandler<T>(this, socketChannel));
        } catch (Exception e) {
            logger.error("processAcceptKey error", e);
        }
    }


    private boolean isBlackList(SocketChannel channel) {
        Set<String> blackList = getBlackList();
        if (blackList == null || blackList.isEmpty()) {
            return false;
        }
        try {
            String remoteIp = ((InetSocketAddress) channel.getRemoteAddress()).getHostName();
            boolean res = blackList.contains(remoteIp);
            logger.debug("remoteIp:{} is in blackList:{}", remoteIp, res);
            return res;
        } catch (Exception e) {
            logger.error("check isBlackList error.", e);
            return false;
        }

    }


    private void startInfo() {
        logger.info("Server StartInfo host:{} port:{} ", host, port);
        logger.info("protocol:{} ", protocolList.size());
        logger.info("processor:{} ", processor.getClass());
        logger.info("sessionListener:{} ", sessionListener.getClass());
        logger.info("readBuffSize:{} ", readBuffSize);
        logger.info("useCompactQueue:{},compactBuffSize:{}", useCompactQueue, compactBuffSize);
    }

    /**
     * 关闭
     */
    public void shutdown() {
        try {
            if (serverSocketChannel != null) {
                serverSocketChannel.close();
                serverSocketChannel = null;
            }
            if (this.executorService != null) {
                this.executorService.shutdown();
            }
        } catch (IOException e) {
            logger.error("serverSocketChannel close error!", e);
        }
    }


    /**
     * @param name
     * @param value
     */
    public Server<T> setOption(SocketOption name, Object value) {
        options.put(name, value);
        return this;
    }

    /**
     * @param threadNumber
     */
    public Server<T> setThreadNumber(int threadNumber) {
        if (threadNumber > 1) {
            this.threadNumber = threadNumber;
        }
        return this;
    }

    /**
     * @param backlog accept queue length
     */
    public Server<T> setBacklog(int backlog) {
        if (backlog > 0) {
            this.backlog = backlog;
        }
        return this;
    }

    public Server<T> setReadBuffSize(int readBuffSize) {
        if (readBuffSize > 4) {
            this.readBuffSize = readBuffSize;
        }
        return this;
    }

    public Server<T> setUseCompactQueue(boolean useCompactQueue) {
        this.useCompactQueue = useCompactQueue;
        return this;
    }

    public Server<T> setCompactBuffSize(int compactBuffSize) {
        this.compactBuffSize = compactBuffSize;
        return this;
    }

    public Server<T> setUseDirectBuffer(boolean useDirectBuffer) {
        this.useDirectBuffer = useDirectBuffer;
        return this;
    }

    public Server<T> setWriteWarnLimit(int writeWarnLimit) {
        this.writeWarnLimit = writeWarnLimit;
        return this;
    }


}




