package cn.ibaijia.isocket;

import cn.ibaijia.isocket.handler.ReadCompletionHandler;
import cn.ibaijia.isocket.handler.WriteCompletionHandler;
import cn.ibaijia.isocket.processor.Processor;
import cn.ibaijia.isocket.protocol.Protocol;
import cn.ibaijia.isocket.session.Session;
import cn.ibaijia.isocket.session.SessionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.net.SocketOption;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

public class Client<T> extends Context implements Runnable {
    private Map<SocketOption<Object>, Object> options = new HashMap<SocketOption<Object>, Object>();
    private String host;
    private int port;
    private Session<T> session;

    private ExecutorService executorService = null;

    private SocketChannel socketChannel = null;
    private Object connLock = new Object();


    public Client(String host, int port) {
        this.host = host;
        this.port = port;
    }

    public Client(String host, int port, Protocol protocol, Processor<T> processor) {
        this.host = host;
        this.port = port;
        this.addProtocol(protocol);
        this.processor = processor;
    }

    public void start() {
        if (this.protocolList.isEmpty() || this.processor == null) {
            logger.error("protocol or processor can't be empty");
            return;
        }
        startInfo();
        try {
            executorService = Executors.newFixedThreadPool(threadNumber, new ThreadFactory() {
                byte index = 0;

                @Override
                public Thread newThread(Runnable runnable) {
                    return new Thread(runnable, "is-nio-c" + (++index));
                }
            });
            executorService.execute(this);
        } catch (Exception e) {
            logger.error("", e);
            shutdown();
        }
    }

    @Override
    public void run() {
        try {
            socketChannel = SocketChannel.open();
            //set isocket options
            if (!options.isEmpty()) {
                for (Map.Entry<SocketOption<Object>, Object> entry : options.entrySet()) {
                    socketChannel.setOption(entry.getKey(), entry.getValue());
                }
            }
            selector = Selector.open();
            //bind host
            socketChannel.connect(new InetSocketAddress(this.host, this.port));

            socketChannel.socket().setSoLinger(true, 2);
            socketChannel.configureBlocking(false);
            if (socketChannel.isConnectionPending()) {
                socketChannel.finishConnect();
            }
            //连接成功则构造AIOSession对象
            session = new Session<T>(socketChannel, this);
            SessionManager.put(session);
            synchronized (connLock) {
                connLock.notifyAll();
            }
            process();
        } catch (Exception e) {
            logger.error("", e);
            shutdown();
        }
    }

    private void process() {
        try {
            socketChannel.register(selector, SelectionKey.OP_READ);
            while (true) {
                regEvent();
                selector.select();//waiting
                Set<SelectionKey> readyKeys = selector.selectedKeys();
                logger.debug("key size:" + readyKeys.size());
                if (session == null) {
                    logger.error("session is null.");
                    break;
                }
                Iterator<SelectionKey> it = readyKeys.iterator();
                while (it.hasNext()) {
                    SelectionKey key = it.next();
                    logger.debug("key.interestOps():{}", key.interestOps());
                    it.remove();
                    SocketChannel socketChannel = (SocketChannel) key.channel();
                    socketChannel.configureBlocking(false);
                    if (key.isReadable()) {
                        if (!session.isReadLocked()) {
                            session.setReadLocked(true);
                            executorService.execute(new ReadCompletionHandler<T>(session, socketChannel));
                        } else {
                            logger.debug("isReadLocked");
                        }
                    } else if (key.isWritable()) {
                        if (!session.isWriteLocked()) {
                            session.setWriteLocked(true);
                            executorService.execute(new WriteCompletionHandler<T>(session, socketChannel));
                        } else {
                            logger.debug("isWriteLocked");
                        }
                    } else if (key.isValid()) {
                        logger.info("key isValid");
                        shutdown();
                    }
                }
            }
        } catch (Exception e) {
            logger.error("unknown error!", e);
            shutdown();
        }
    }


    private void startInfo() {
        logger.info("Client StartInfo host:{} port:{} ", host, port);
        logger.info("protocol:{} ", protocolList.size());
        logger.info("processor:{} ", processor.getClass());
        logger.info("sessionListener:{} ", sessionListener.getClass());
        logger.info("readBuffSize:{} ", readBuffSize);
        logger.info("useCompactQueue:{},compactBuffSize:{}", useCompactQueue, compactBuffSize);
    }

    public void shutdown() {
        if (session != null) {
            SessionManager.close(session);
            session = null;
        }
        //仅Client内部创建的ChannelGroup需要shutdown
        if (executorService != null) {
            executorService.shutdown();
        }
    }

    public Client<T> setOption(SocketOption<Object> socketOption, Object value) {
        this.options.put(socketOption, value);
        return this;
    }

    public Client<T> setThreadNumber(Integer threadNumber) {
        this.threadNumber = threadNumber;
        return this;
    }

    public Client<T> setReadBuffSize(int readBuffSize) {
        if (readBuffSize > 4) {
            this.readBuffSize = readBuffSize;
        }
        return this;
    }

    public Client<T> setUseCompactQueue(boolean useCompactQueue) {
        this.useCompactQueue = useCompactQueue;
        return this;
    }

    public Client<T> setCompactBuffSize(int compactBuffSize) {
        this.compactBuffSize = compactBuffSize;
        return this;
    }

    public Client<T> setUseDirectBuffer(boolean useDirectBuffer) {
        this.useDirectBuffer = useDirectBuffer;
        return this;
    }

    public Client<T> setWriteWarnLimit(int writeWarnLimit) {
        this.writeWarnLimit = writeWarnLimit;
        return this;
    }

    public Session<T> getSession() {
        synchronized (connLock) {
            if (session == null) {
                try {
                    connLock.wait(10000l);
                } catch (Exception e) {
                    logger.error("getSession timeout!", e);
                }
            }
            return session;
        }
    }

}
