/**
 * Copyright © 2019 organization humingfeng
 * <pre>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * <pre/>
 */
package com.humingfeng.dynamic.datasource.processor;

import org.aopalliance.intercept.MethodInvocation;

/**
 * @author humingfeng
 * @since 2.5.0
 */
public abstract class DsProcessor {

  private DsProcessor nextProcessor;

  public void setNextProcessor(DsProcessor dsProcessor) {
    this.nextProcessor = dsProcessor;
  }

  /**
   * 抽象匹配条件 匹配才会走当前执行器否则走下一级执行器
   *
   * @param key DS注解里的内容
   * @return 是否匹配
   */
  public abstract boolean matches(String key);

  /**
   * 决定数据源
   * <pre>
   *     调用底层doDetermineDatasource，
   *     如果返回的是null则继续执行下一个，否则直接返回
   * </pre>
   *
   * @param invocation 方法执行信息
   * @param key DS注解里的内容
   * @return 数据源名称
   */
  public String determineDatasource(MethodInvocation invocation, String key) {
    if (matches(key)) {
      String datasource = doDetermineDatasource(invocation, key);
      if (datasource == null && nextProcessor != null) {
        return nextProcessor.determineDatasource(invocation, key);
      }
      return datasource;
    }
    if (nextProcessor != null) {
      return nextProcessor.determineDatasource(invocation, key);
    }
    return null;
  }

  /**
   * 抽象最终决定数据源
   *
   * @param invocation 方法执行信息
   * @param key DS注解里的内容
   * @return 数据源名称
   */
  public abstract String doDetermineDatasource(MethodInvocation invocation, String key);
}
