package cn.hperfect.nbquerier.spring.transaction;

import cn.hperfect.nbquerier.core.components.datasouce.INbDataSource;
import cn.hperfect.nbquerier.core.transaction.INbTransaction;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.datasource.DataSourceUtils;

import java.sql.Connection;
import java.sql.SQLException;

/**
 * spring 事务
 */
@Slf4j
public class SpringNbTransaction implements INbTransaction {

    private final boolean closeConnection;
    private Connection connection;
    private INbDataSource dataSource;
    private boolean autoCommit;
    private boolean isConnectionTransactional;


    public SpringNbTransaction(Connection conn) {
        this.connection = conn;
        this.closeConnection = false;
    }

    public SpringNbTransaction(INbDataSource nbDataSource) {
        this.closeConnection = true;
        this.dataSource = nbDataSource;
    }

    @Override
    public Connection getConnection(String dsName) throws SQLException {
        if (this.connection == null) {
            openConnection(dsName);
        }
        return this.connection;
    }

    private void openConnection(String dsName) throws SQLException {
        Assert.isTrue(CollUtil.isNotEmpty(this.dataSource.getAll()), "未配置数据源,请检查配置");
        //spring 接管connection
        this.connection = DataSourceUtils.getConnection(this.dataSource);
        this.autoCommit = this.connection.getAutoCommit();
        this.isConnectionTransactional = DataSourceUtils.isConnectionTransactional(this.connection, this.dataSource);

        if (log.isDebugEnabled()) {
            log.debug(
                    "JDBC Connection ["
                            + this.connection
                            + "] will"
                            + (this.isConnectionTransactional ? " " : " not ")
                            + "be managed by Spring");
        }
    }

    /**
     * 不是事务连接 才提交
     *
     * @throws SQLException
     */
    @Override
    public void commit() throws SQLException {
        if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
            if (log.isDebugEnabled()) {
                log.debug("Committing JDBC Connection [" + this.connection + "]");
            }
            this.connection.commit();
        }
    }

    @Override
    public void rollback() throws SQLException {
        if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
            if (log.isDebugEnabled()) {
                log.debug("Rolling back JDBC Connection [" + this.connection + "]");
            }
            this.connection.rollback();
        }
    }

    @Override
    public void close() {
        if (closeConnection) {
            DataSourceUtils.releaseConnection(this.connection, this.dataSource);
        }
    }
}
