package cn.hperfect.nbquerier.spring.mapper;

import cn.hperfect.nbquerier.session.ISqlMapperBuilder;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;

import static org.springframework.util.Assert.notNull;

/**
 * 包 扫描器配置
 */
@Getter
@Setter
public class MapperScannerConfigurer implements BeanDefinitionRegistryPostProcessor, InitializingBean, ApplicationContextAware, BeanNameAware {

    private String beanName;
    private String basePackage;
    private Class<? extends Annotation> annotationClass;
    /**
     * 通过工程获取 动态代理的 bean
     */
    private Class<? extends MapperFactoryBean> mapperFactoryBeanClass;
    private ApplicationContext applicationContext;

    private ISqlMapperBuilder sqlMapperBuilder;


    @Override
    public void setBeanName(String name) {
        this.beanName = name;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        notNull(this.basePackage, "Property 'basePackage' is required");
        notNull(this.sqlMapperBuilder, "Property 'sqlMapperBuilder' is required");
    }

    /**
     * 扫描 mapper 注册 BeanDefinition,spring 会通过 FactoryBean 获取对象实例
     *
     * @throws BeansException
     */
    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        ClassPathMapperScanner scanner = new ClassPathMapperScanner(registry);
        scanner.setMapperFactoryBeanClass(this.mapperFactoryBeanClass);
        scanner.setAnnotationClass(annotationClass);
        scanner.setSqlMapperBuilder(sqlMapperBuilder);
        scanner.registerFilters();
        scanner.scan(
                StringUtils.tokenizeToStringArray(this.basePackage, ConfigurableApplicationContext.CONFIG_LOCATION_DELIMITERS));
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        // left intentionally blank
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
