package cn.hperfect.nbquerier.core.components.datasouce;

import cn.hperfect.nbquerier.enums.DbType;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.baomidou.dynamic.datasource.ds.AbstractRoutingDataSource;
import com.baomidou.dynamic.datasource.exception.CannotFindDataSourceException;
import com.baomidou.dynamic.datasource.provider.DynamicDataSourceProvider;
import com.baomidou.dynamic.datasource.support.DdConstants;
import com.baomidou.dynamic.datasource.toolkit.DynamicDataSourceContextHolder;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import javax.sql.DataSource;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author huanxi
 * @version 1.0
 * @date 2021/11/23 4:57 下午
 * @see com.baomidou.dynamic.datasource.DynamicRoutingDataSource
 */
@Slf4j
public class NbDataSource extends AbstractRoutingDataSource implements INbDataSource, InitializingBean, DisposableBean {
    private final Map<String, DataSource> dataSourceMap = new LinkedHashMap<>();
    @Setter
    private DynamicDataSourceProvider provider;

    private DataSource master;

    public NbDataSource(DynamicDataSourceProvider provider) {
        this.provider = provider;
    }

    @Override
    public DataSource switchDataSource(String name) {
        if (StrUtil.isBlank(name)) {
            return determineDataSource();
        }
        DataSource dataSource = dataSourceMap.get(name);
        Assert.notNull(dataSource, "数据源:{}不存在", name);
        return dataSource;
    }

    @Override
    public DbType getDbType(String name) {
//        switchDataSource(name).unwrap();
        return null;
    }

    @Override
    public Collection<DataSource> getAll() {
        return dataSourceMap.values();
    }

    @Override
    protected DataSource determineDataSource() {
        String ds = DynamicDataSourceContextHolder.peek();
        //切换数据源->
        //事务连接处理
        if (StrUtil.isBlank(ds)) {
            return master;
        } else if (dataSourceMap.containsKey(ds)) {
            log.debug("dynamic-datasource switch to the datasource named [{}]", ds);
            return dataSourceMap.get(ds);
        } else {
            throw new CannotFindDataSourceException("dynamic-datasource could not find a datasource named:" + ds);
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Map<String, DataSource> dataSources = provider.loadDataSources();
        // 添加并分组数据源
        for (Map.Entry<String, DataSource> dsItem : dataSources.entrySet()) {
            addDataSource(dsItem.getKey(), dsItem.getValue());
        }
        Assert.notNull(master, "主数据源不能为空");

    }

    @Override
    public void destroy() throws Exception {
        log.info("dynamic-datasource start closing ....");
        for (Map.Entry<String, DataSource> item : dataSourceMap.entrySet()) {
            closeDataSource(item.getKey(), item.getValue());
        }
        log.info("dynamic-datasource all closed success,bye");
    }

    private void closeDataSource(String name, DataSource dataSource) throws InvocationTargetException, IllegalAccessException {
        Class<? extends DataSource> clazz = dataSource.getClass();
        try {
            Method closeMethod = clazz.getDeclaredMethod("close");
            closeMethod.invoke(dataSource);
        } catch (NoSuchMethodException e) {
            log.warn("dynamic-datasource close the datasource named [{}] failed,", name);
        }
    }

    public synchronized void addDataSource(String ds, DataSource dataSource) {
        if (!dataSourceMap.containsKey(ds)) {
            //todo 此处省略wrapDataSource,分布式事务...
//            dataSource = wrapDataSource(ds, dataSource);
            dataSourceMap.put(ds, dataSource);
            if (ds.equals(DdConstants.MASTER)) {
                this.master = dataSource;
            }
            log.info("dynamic-datasource - load a datasource named [{}] success", ds);
        } else {
            log.warn("dynamic-datasource - load a datasource named [{}] failed, because it already exist", ds);
        }
    }


}
