package cn.hperfect.nbquerier.toolkit;

import cn.hperfect.nbquerier.core.metedata.QueryField;
import cn.hperfect.nbquerier.core.metedata.QueryFields;
import cn.hperfect.nbquerier.core.metedata.QueryItem;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.map.BiMap;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author huanxi
 * @version 1.0
 * @date 2021/11/30 5:48 下午
 */
public class BuildUtils {
    private BuildUtils() {
    }

    /**
     * 通过查询元素,字段获取表名
     * t.field table.field field,null
     *
     * @param item
     * @param tableAliasMap
     * @return
     */
    public static String getTableByField(QueryItem item, BiMap<String, String> tableAliasMap) {
        String fieldTableAlias = item.getTableAlias();
        String table = null;
        if (fieldTableAlias == null) {
            return null;
        }
        if (tableAliasMap.containsValue(item.getTableAlias())) {
            //如果能获取到表
            table = tableAliasMap.getKey(item.getTableAlias());
        } else if (tableAliasMap.containsKey(fieldTableAlias)) {
            table = fieldTableAlias;
        }
        Assert.notBlank(table, () -> new NbSQLMessageException("字段:{},表别名为{}的表不存在", item.getName(), fieldTableAlias));
        return table;
    }


    /**
     * 从字段名称中找表
     *
     * @param fields    所有字段
     * @param fieldName 字段名
     * @param table     表名  {sub_xx}
     * @return
     */
    public static Collection<QueryField> find(QueryFields fields, String fieldName, String table) {
        //先找一个在找多个
        Collection<QueryField> queryFields = fields.findByName(fieldName);
        if (CollUtil.size(queryFields) > 1 && StrUtil.isNotBlank(table)) {
            //过滤table
            queryFields = CollUtil.filterNew(queryFields, k -> table.equals(k.getTableName()));
        }
        return queryFields;
    }

    /**
     * 查询字段,可以很好的防止字段处的sql注入
     *
     * @param fields        表单的所有字段
     * @param item          查询item
     * @param tableAliasMap
     * @return
     */
    public static QueryField findOne(QueryFields fields, QueryItem item, String masterTable, BiMap<String, String> tableAliasMap) {
        //字段所在表名
        String fieldTable = BuildUtils.getTableByField(item, tableAliasMap);
        //table 可能是子查询 {sub_t}
        Collection<QueryField> fieldList = BuildUtils.find(fields, item.getName(), fieldTable);
        Assert.notEmpty(fieldList, () -> new NbSQLMessageException("查询表单:{},字段:{}不存在", ObjectUtil.defaultIfBlank(fieldTable, masterTable), item.getName()));
        if (fieldList.size() > 1) {
            if (fieldTable == null && StrUtil.isNotBlank(masterTable)) {
                //取主表
                return CollUtil.findOne(fieldList, i -> StrUtil.equals(masterTable, i.getTableName()));
            }
            List<String> tables = fieldList.stream()
                    .map(QueryField::getTableName)
                    .collect(Collectors.toList());
            throw new NbSQLMessageException("字段:{},在表:{},中都存在,请指定表名", item.getName(), tables);
        }

        return fieldList.stream().iterator().next();
    }
}
