package cn.hperfect.nbquerier.enums;

import cn.hperfect.nbquerier.core.conditions.ISqlSegment;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hutool.core.util.EnumUtil;
import lombok.Getter;
import lombok.Setter;

import java.util.LinkedHashMap;
import java.util.Objects;

/**
 * Query 规则 常量
 *
 * @Author Scott
 * @Date 2019年02月14日
 */
public enum QueryRuleEnum implements ISqlSegment {

    /**
     * 大于
     */
    GT(">", "gt", SqlKeyword.GT),
    /**
     * 大于等于
     */
    GE(">=", "ge", SqlKeyword.GE),
    /**
     * 小于
     */
    LT("<", "lt", SqlKeyword.LT),
    /**
     * <=
     */
    LE("<=", "le", SqlKeyword.LE),
    /**
     * =
     */
    EQ("=", "eq", SqlKeyword.EQ),
    /**
     * !=
     */
    NE("!=", "ne", SqlKeyword.NE),
    /**
     * 字段 再 (数组)中
     */
    IN("IN", "in", SqlKeyword.IN),
    /**
     * 字段不在数组中
     */
    NOT_IN("NOT IN", "!in", SqlKeyword.NOT_IN),
    /**
     * 模糊查询
     */
    LIKE("LIKE", "like", SqlKeyword.LIKE),
    /**
     * 非null
     */
    NOT_NULL("NOT_NULL", "!null", SqlKeyword.IS_NOT_NULL),
    NULL("NULL", "null", SqlKeyword.IS_NULL),
    BETWEEN("BETWEEN", "bt", SqlKeyword.BETWEEN),
    /**
     * 值=any(数组字段)
     */
    ANY("SQL", "any", null),
    /**
     * 废弃
     */
    NOT_ANY("SQL", "any", null),
    SQL("SQL", "sql", null),
    /**
     * 空,not 则为非空
     */
    EMPTY("SQL", "emp", null),
    NOT_EMPTY("SQL", "emp", null),
    BLANK("SQL", "emp", null),
    NOT_BLANK("SQL", "emp", null),
    /**
     * 交集
     */
    INTERS("&&", null, () -> "&&"),
    EXISTS("SQL", null,SqlKeyword.EXISTS),;

    private final String sql;
    /**
     * 操作符解析
     */
    @Getter
    @Setter
    private String value;
    /**
     * 字符串条件
     */
    @Getter
    private final String condition;

    QueryRuleEnum(String value, String condition, ISqlSegment sqlSegment) {
        this.value = value;
        this.condition = condition;
        this.sql = sqlSegment != null ? sqlSegment.getSqlSegment() : null;
    }

    public static QueryRuleEnum parse(String condition) {
        LinkedHashMap<String, QueryRuleEnum> enumMap = EnumUtil.getEnumMap(QueryRuleEnum.class);
        for (QueryRuleEnum ruleEnum : enumMap.values()) {
            if (Objects.equals(condition, ruleEnum.getCondition()) || Objects.equals(condition, ruleEnum.getValue())) {
                return ruleEnum;
            }
        }
        throw new NbSQLMessageException("不支持条件:[{}]查询", condition);
    }


    @Override
    public String getSqlSegment() {
        return sql;
    }
}
