package cn.hperfect.nbquerier.core.type;

import cn.hperfect.nbquerier.annotation.NbExcel;
import cn.hperfect.nbquerier.annotation.NbJsonProp;
import cn.hperfect.nbquerier.core.components.type.INbQueryType;
import cn.hperfect.nbquerier.core.components.type.NbQueryType;
import cn.hperfect.nbquerier.core.conditions.segments.MergeSegments;
import cn.hperfect.nbquerier.core.metedata.JsonPropMeteData;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.enums.DbDataType;
import cn.hperfect.nbquerier.enums.QueryRuleEnum;
import cn.hperfect.nbquerier.exceptions.TypeConvertException;
import cn.hutool.core.bean.BeanDesc;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.PropDesc;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.StrUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.lang.Nullable;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

@Data
@NoArgsConstructor
public class JsonNbType implements INbQueryType {
    /**
     * 是否是数组
     *
     * @Nullable 可以不是数组也可以是数组
     */
    @Nullable
    protected Boolean jsonArray;
    /**
     * 属性
     */
    protected List<JsonPropMeteData> props;
    public static final List<QueryRuleEnum> rules = ListUtil.toList(QueryRuleEnum.EMPTY,
            QueryRuleEnum.LIKE, QueryRuleEnum.NOT_EMPTY);
    protected Class<? extends JsonPropVO> voClass;


    public JsonNbType(Boolean jsonArray) {
        this.jsonArray = jsonArray;
        props = new ArrayList<>();
    }

    public JsonNbType(boolean isArray, Class<? extends JsonPropVO> clazz) {
        this.jsonArray = isArray;
        this.voClass = clazz;
        BeanDesc beanDesc = BeanUtil.getBeanDesc(clazz);
        this.props = new ArrayList<>();
        for (PropDesc prop : beanDesc.getProps()) {
            JsonPropMeteData data = new JsonPropMeteData();
            data.setName(prop.getFieldName());
            Field field = prop.getField();
            INbQueryType queryType = NbQueryType.convertFromClass(field.getType(), field.getGenericType());
            data.setQueryType(queryType);
            NbJsonProp nbJsonProp = prop.getField().getAnnotation(NbJsonProp.class);
            if (nbJsonProp != null) {
                data.setIndex(nbJsonProp.index());
                data.setTitle(nbJsonProp.title());
            }
            NbExcel nbExcel = prop.getField().getAnnotation(NbExcel.class);
            if (nbExcel != null) {
                data.setExcelIgnore(nbExcel.ignore());
            }


            this.props.add(data);
        }
        CollUtil.sort(props, Comparator.comparingInt(JsonPropMeteData::getIndex));
    }

    @Override
    public DbDataType getDbDataType() {
        return DbDataType.JSON;
    }

    /**
     * 返回json 字符串
     *
     * @param value
     * @return
     * @throws TypeConvertException
     */
    @Override
    public Object convert(Object value) throws TypeConvertException {
        return value;
    }

    @Override
    public String getDbTypeSql() {
        return "jsonb";
    }

    @Override
    public String getTitle() {
        return "json";
    }


    @Override
    public void whereLike(NbQuerier<?> querier, MergeSegments expression, String name, Object value) {
        toStrWhereLike(querier, expression, name, value);
    }


    @Override
    public List<QueryRuleEnum> getQueryRules() {
        return rules;
    }

    @Override
    public boolean orderAble() {
        return false;
    }


    /**
     * 转换成字符串like
     *
     * @param querier
     * @param expression
     * @param name
     * @param value
     */
    protected final void toStrWhereLike(NbQuerier<?> querier, MergeSegments expression, String name, Object value) {
        assert value instanceof String : "模糊查询必须传入字符串";
        String valueStr = (String) value;
        valueStr = StrUtil.removePrefix(valueStr, "%");
        valueStr = StrUtil.removeSuffix(valueStr, "%");
        value = StrUtil.concat(true, "%", valueStr, "%");
        String paramKey = querier.formatVariable(NbQueryType.TEXT, value, false);
        expression.add(() -> name + "::text", QueryRuleEnum.LIKE, () -> paramKey);
    }


    public boolean isJsonArray() {
        return BooleanUtil.isTrue(jsonArray);
    }
}
