/*
 *    Copyright 2009-2012 the original author or authors.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package cn.hperfect.nbquerier.core.transaction.jdbc;

import cn.hperfect.nbquerier.core.transaction.INbTransaction;
import cn.hperfect.nbquerier.core.transaction.TransactionException;
import cn.hperfect.nbquerier.core.transaction.TransactionIsolationLevel;
import cn.hutool.core.lang.Assert;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import javax.sql.DataSource;
import java.sql.*;

/**
 * {@link Transaction} that makes use of the JDBC commit and rollback facilities directly.
 * It relies on the connection retrieved from the dataSource to manage the scope of the transaction.
 * Delays connection retrieval until getConnection() is called.
 * Ignores commit or rollback requests when autocommit is on.
 *
 * @author Clinton Begin
 * @see cn.hperfect.nbquerier.core.transaction.jdbc.JdbcTransactionFactory
 */

/**
 * 手动事务
 * Jdbc事务。直接利用JDBC的commit,rollback。
 * 它依赖于从数据源得 到的连接来管理事务范围。
 */
@Slf4j
public class JdbcTransaction implements INbTransaction {


    protected Connection connection;
    protected DataSource dataSource;

    protected TransactionIsolationLevel level;
    protected boolean autoCommmit;

    public JdbcTransaction(DataSource ds, TransactionIsolationLevel desiredLevel, boolean desiredAutoCommit) {
        dataSource = ds;
        level = desiredLevel;
        autoCommmit = desiredAutoCommit;
    }

    public JdbcTransaction(Connection connection) {
        this.connection = connection;
    }

    @Override
    public Connection getConnection(String dsName) throws SQLException {
        if (connection == null) {
            openConnection();
        }
        return connection;
    }

    @Override
    public void commit() throws SQLException {
        if (connection != null && !connection.getAutoCommit()) {
            if (log.isDebugEnabled()) {
                log.debug("Committing JDBC Connection [" + connection + "]");
            }
            connection.commit();
        }
    }

    @Override
    public void rollback() throws SQLException {
        if (connection != null && !connection.getAutoCommit()) {
            if (log.isDebugEnabled()) {
                log.debug("Rolling back JDBC Connection [" + connection + "]");
            }
            connection.rollback();
        }
    }

    @Override
    public void setAutoCommit(boolean autoCommit) throws SQLException {

    }

    @SneakyThrows
    @Override
    public void close() {
        //自己关闭
    }

    public void close(boolean reallyClose) throws SQLException {
        if (reallyClose) {
            if (connection != null) {
                resetAutoCommit();
                if (log.isDebugEnabled()) {
                    log.debug("Closing JDBC Connection [" + connection + "]");
                }
                connection.close();
            }
        }
    }

    protected void setDesiredAutoCommit(boolean desiredAutoCommit) {
        try {
            //和原来的比一下，再设置autocommit，是考虑多次重复设置的性能问题？
            if (connection.getAutoCommit() != desiredAutoCommit) {
                if (log.isDebugEnabled()) {
                    log.debug("Setting autocommit to " + desiredAutoCommit + " on JDBC Connection [" + connection + "]");
                }
                connection.setAutoCommit(desiredAutoCommit);
            }
        } catch (SQLException e) {
            // Only a very poorly implemented driver would fail here,
            // and there's not much we can do about that.
            throw new TransactionException("Error configuring AutoCommit.  "
                    + "Your driver may not support getAutoCommit() or setAutoCommit(). "
                    + "Requested setting: " + desiredAutoCommit + ".  Cause: " + e, e);
        }
    }

    //见下面注释，貌似是说是对有些DB的一个workaround
    protected void resetAutoCommit() {
        try {
            if (!connection.getAutoCommit()) {
                if (log.isDebugEnabled()) {
                    log.debug("Resetting autocommit to true on JDBC Connection [" + connection + "]");
                }
                connection.setAutoCommit(true);
            }
        } catch (SQLException e) {
            log.debug("Error resetting autocommit to true "
                    + "before closing the connection.  Cause: " + e);
        }
    }

    protected void openConnection() throws SQLException {
        if (log.isDebugEnabled()) {
            log.debug("Opening JDBC Connection");
        }
        connection = dataSource.getConnection();
        /*PreparedStatement preparedStatement = connection.prepareStatement("SELECT CAST(txid_current() AS text)");
        ResultSet resultSet = preparedStatement.executeQuery();
        if (resultSet.next()) {
            int txId = resultSet.getInt(0);
            log.info("当前事务id:{}", txId);
        }*/
        Assert.notNull(connection, "数据源打开连接失败");
        if (level != null) {
            connection.setTransactionIsolation(level.getLevel());
        }
        setDesiredAutoCommit(autoCommmit);
    }

}
