package cn.hperfect.nbquerier.core.querier.qrs;

import cn.hperfect.nbquerier.core.metedata.InputNbTable;
import cn.hperfect.nbquerier.core.metedata.JsonPropMeteData;
import cn.hperfect.nbquerier.core.metedata.QueryField;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.core.type.JsonNbType;
import cn.hperfect.nbquerier.enums.QueryRuleEnum;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONNull;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * 查询表达式解析器
 *
 * @author huanxi
 * @version 1.0
 * @date 2021/3/17 4:58 下午
 */
@Slf4j
public enum QueryExprParser {
    /**
     * 实例
     */
    INSTANCE;

    /**
     * 顶级操作符
     */
    enum RootOperator {
        OPERATOR_OR("$or"),
        OPERATOR_AND("$and"),
        OPERATOR_FIELD("$field"),
        ;

        @Getter
        private String key;

        RootOperator(String key) {
            this.key = key;
        }

        public static RootOperator convertFromKey(String key) {
            for (RootOperator value : values()) {
                if (value.key.equals(key)) {
                    return value;
                }
            }
            throw new NbSQLMessageException("未知操作符类型:{}", key);
        }

    }

    /**
     * 操作符
     */
    private static final String OPERATOR = "$";
    private static final String OPERATOR_OR = "$or";
    private static final String OPERATOR_AND = "$and";
    private static final String OPERATOR_FIELD = "$field";
    private static final String OPERATOR_TYPE = "$type";
    private static final String OPERATOR_VALUE = "$value";
    private static final String OPERATOR_JSON = "$json";

    /**
     * 解析入口
     *
     * @param query
     * @param queryExpr
     */
    public void parseJsonObjectToCondition(NbQuerier<?> query, String queryExpr) {
        if (StrUtil.isNotEmpty(queryExpr)) {
            //查询表达式解析
            JSONObject jsonObject = JSONUtil.parseObj(queryExpr);
            if (jsonObject.size() > 0) {
                this.parseJsonObjectToCondition(query, jsonObject);
            }
        }
    }


    /**
     * 将查询表达式解析为
     * 使用方法
     * 查询等于 field=123
     * {"field":"123"}
     * 查询大于 field>123
     * {"filed":{"$>":123}}
     * {"field_name":"$NULL"} aaa='$NULL' 这种语法不行
     * {"field_name":{"$TYPE":"$NULL"}}
     * $or
     * <p>
     * "/phoneList[]/memo$like":"test"
     *
     * @param query
     * @param jsonObject
     */
    public static void parseJsonObjectToCondition(NbQuerier<?> query, JSONObject jsonObject) {
        jsonObject.forEach((key, value) -> {
            if (key.startsWith(OPERATOR)) {
                //特殊处理,(and 和 or)
                RootOperator rootOperator = RootOperator.convertFromKey(key);
                switch (rootOperator) {
                    case OPERATOR_OR: {
                        if (value instanceof JSONArray) {
                            JSONArray vArray = (JSONArray) value;
                            query.where(i -> {
                                //如果数组下面内容全空 添加1=1
                                for (Object jsonObj : vArray) {
                                    JSONObject valObj = (JSONObject) jsonObj;
                                    parseJsonObjectToCondition(i.or(), valObj);
                                }
                            });
                        } else {
                            //and
                            query.where(i -> {
                                JSONObject objectOr = (JSONObject) value;
                                objectOr.forEach((k, v) -> {
                                    JSONObject val = new JSONObject();
                                    val.set(k, v);
                                    parseJsonObjectToCondition(i.or(), val);
                                });
                            });
                        }
                    }
                    break;
                    case OPERATOR_AND: {
                        if (value instanceof JSONArray) {
                            JSONArray vArray = (JSONArray) value;
                            query.where(i -> {
                                for (Object jsonObj : vArray) {
                                    parseJsonObjectToCondition(i, (JSONObject) jsonObj);
                                }
                            });
                        } else {
                            query.where(i -> parseJsonObjectToCondition(i, (JSONObject) value));
                        }
                    }
                    break;
                    case OPERATOR_FIELD: {
                        //设置字段
                        query.field(value.toString());
                    }
                    break;
                    default:
                        //直接处理
                        throw new NbSQLMessageException("不支持指令操作符:{}", rootOperator);
                }
            } else {
                //处理非操作符, 字段 或者 字段$操作符
                String dbKey = StrUtil.toUnderlineCase(key);
                if (dbKey.contains(OPERATOR)) {
//                    操作符转发 xxx$lt  ->  {xxx:{$lt:50}}
                    //操作符转发 xxx$lt  ->  {xxx:{$type:$lt,$value:50}}
                    List<String> tow = StrUtil.split(dbKey, OPERATOR);
                    JSONObject obg = new JSONObject();
                    JSONObject obgValue = new JSONObject();
                    obgValue.set(OPERATOR_TYPE, OPERATOR + tow.get(1));
                    obgValue.set(OPERATOR_VALUE, value);
                    obg.set(tow.get(0), obgValue);
                    parseJsonObjectToCondition(query, obg);
                    return;
                }
                if (value instanceof JSONNull) {
                    return;
                }
                if (value instanceof JSONObject) {
                    JSONObject conditionValue = ((JSONObject) value);
                    if (((JSONObject) value).containsKey(OPERATOR_TYPE)) {
                        String opType = conditionValue.getStr(OPERATOR_TYPE);
                        if (StrUtil.isNotBlank(opType)) {
                            //type->value
                            parseOperation(query, dbKey, conditionValue.getStr(OPERATOR_TYPE), conditionValue.get(OPERATOR_VALUE));
                        }
                    } else {
                        //必有$ 如{$lt:50}
                        ((JSONObject) value).forEach((k2, v2) -> {
                            if (k2.startsWith(OPERATOR) && v2 != null) {
                                //特殊运算符处理
                                parseOperation(query, dbKey, k2, v2);
                            }
                        });
                    }
                } else if (value instanceof JsonQueryValue) {
                    JsonQueryValue jsonQueryValue = (JsonQueryValue) value;
                    QueryRuleEnum rule = QueryRuleEnum.parse(jsonQueryValue.getType());
                    query.where(jsonQueryValue, rule, jsonQueryValue.getQueryValue(), false);
                } else {
                    //数据库查询 dbKey=value
                    query.where(dbKey, QueryRuleEnum.EQ, value);
                }
            }
        });
    }

    /**
     * "/phoneList[]/memo$like":"test"
     * 解析json表达式
     *
     * @param query
     * @param field
     * @param value
     */
    private static void parseJsonExpr(NbQuerier<?> query, String field, Object value) {
        Assert.isTrue(value instanceof JSONArray || value instanceof JSONObject, "value 必须为数组或对象");
        QueryField queryField = query.findField(field);
        Assert.notNull(queryField, "字段:{}不存在", queryField);
        Assert.isTrue(queryField.getType() instanceof JsonNbType, "json查询操作只支持json字段");
        JsonNbType jsonType = (JsonNbType) queryField.getType();
        List<JsonPropMeteData> props = ((JsonNbType) queryField.getType()).getProps();
        if (jsonType.isJsonArray()) {
            parseJsonArray(query, field, value, jsonType, props);
        } else {
            parseJsonObject(query, field, value, jsonType, props);
        }
    }

    /**
     * 解析 json object 类型
     *
     * @param query
     * @param field
     * @param value
     * @param jsonNbType
     * @param props
     */
    private static void parseJsonObject(NbQuerier<?> query, String field, Object value,
                                        JsonNbType jsonNbType,
                                        List<JsonPropMeteData> props) {
        Assert.isTrue(value instanceof JSONObject, "暂时只支持object");
        JSONObject jsonValue = (JSONObject) value;
        JSONObject newQrs = new JSONObject();
        jsonValue.forEach((k, v) -> {
            String propName;
            String type;
            if (StrUtil.contains(k, OPERATOR)) {
                List<String> split = StrUtil.split(k, OPERATOR);
                propName = split.get(0);
                type = split.get(1);
            } else {
                type = "=";
                propName = k;
            }
            JsonPropMeteData one = CollUtil.findOne(props, i -> i.getName().equals(propName));
            Assert.notNull(one, "未找到json属性:{}", propName);
            JsonQueryValue jsonQueryValue = new JsonQueryValue();
            jsonQueryValue.setType(type);
            jsonQueryValue.setValue(v);
            jsonQueryValue.setFieldName(field);
            jsonQueryValue.setPropName(propName);
            jsonQueryValue.setQueryType(one.getQueryType());
            newQrs.set(field + "." + propName, jsonQueryValue);
        });
        parseJsonObjectToCondition(query, newQrs);
        //(table.phone->'memo')::text='test'
        //"nickname$=":"test" -> xxx->'nickname'
        //{{queryItem}:{$lt:50}}


    }

    /**
     * 解析json array类型
     * 把数组类型变成一个子查询，递归到子查询下
     *
     * @param query
     * @param value
     * @param jsonType
     * @param props
     */
    private static void parseJsonArray(NbQuerier<?> query, String field, Object value, JsonNbType jsonType, List<JsonPropMeteData> props) {
        //数组
        //如何确定字段类型
        Assert.notEmpty(props, "json属性未定义查询失败");
        InputNbTable inputNbTable = new InputNbTable(StrUtil.format("jsonb_array_elements({})", field), props);
        JSONObject qrs;
        if (value instanceof JSONArray) {
            qrs = new JSONObject().set(OPERATOR_AND, value);
        } else {
            qrs = (JSONObject) value;
        }
        // (json_item(.)->'memo')::text='test'
        //构建字段
        String alias = "json_item";
        NbQuerier<?> sub = NbQuerier.from(
                        NbQuerier.table(inputNbTable, alias)
                                .field("")
                                .additionFields(buildJsonFields(alias, props))
                        , "json_arr")
                .whereQrs(qrs);
        query.buildConnect(sub);
        query.where(StrUtil.format("({})>0", sub.buildFuncSql("count(*)")));
    }

    public static List<String> buildJsonFields(String columnName, List<JsonPropMeteData> props) {
        List<String> fields = new ArrayList<>();
        for (JsonPropMeteData prop : props) {
            String propName = StrUtil.toUnderlineCase(prop.getName());
            fields.add(StrUtil.format("({}->>'{}')::{} {}", columnName, propName, prop.getQueryType().getDbTypeSql(), propName));

        }
        return fields;
    }


    /**
     * 处理操作符
     *
     * @param query     查询对象
     * @param fieldName 字段 xxx,(xxx->xxx)::text
     * @param k2        操作符 $json,$=
     * @param v2        值
     */
    private static void parseOperation(NbQuerier<?> query, String fieldName, String k2, Object v2) {
        if (OPERATOR_JSON.equals(k2)) {
            parseJsonExpr(query, fieldName, v2);
            return;
        }
        QueryRuleEnum rule = QueryRuleEnum.parse(StrUtil.removePrefix(k2, OPERATOR));
        if (rule == QueryRuleEnum.IN || rule == QueryRuleEnum.NOT_IN) {
            Assert.isInstanceOf(Collection.class, v2, "操作符:{},值必须为数组", rule);
            if (CollUtil.isEmpty(Convert.toList(v2))) {
                return;
            }
        }
        query.where(fieldName, rule, v2);
    }

}
