package cn.hperfect.nbquerier.core.querier.condition;

import cn.hperfect.nbquerier.core.metedata.QueryValParam;
import cn.hperfect.nbquerier.core.querier.NbQuerier;
import cn.hperfect.nbquerier.enums.NbOrderType;
import cn.hperfect.nbquerier.enums.QueryRuleEnum;
import cn.hperfect.nbquerier.exceptions.NbSQLMessageException;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;

import java.util.Collection;
import java.util.Map;
import java.util.function.Consumer;

/**
 * where 查询条件(默认实现)
 * <T> 查询器类型
 * <F> 字段类型
 *
 * @author huanxi
 * @version 1.0
 * @date 2021/11/26 9:33 上午
 */
public interface DefaultDoCondition<T> extends DoConditionTopside<T> {

    /**
     * 主键查询
     *
     * @param ruleEnum
     * @param value
     * @return
     */
    default NbQuerier<T> wherePk(QueryRuleEnum ruleEnum, Object value) {
        Assert.notNull(value, "pk值不能为空");
        return where(getPkName(), ruleEnum, value);
    }

    /**
     * 通过主键查询
     *
     * @param value
     * @return
     */
    default NbQuerier<T> wherePk(Object value) {
        return wherePk(QueryRuleEnum.EQ, value);
    }

    /**
     * value = any(filed)
     *
     * @param field
     * @param value
     * @return
     */
    default NbQuerier<T> whereAny(String field, Object value) {
        return whereAny(field, value, false);
    }

    default NbQuerier<T> whereNotAny(String field, Object value) {
        return where(field, QueryRuleEnum.NOT_ANY, value, false);
    }


    default NbQuerier<T> whereAny(String field, Object value, boolean required) {
        return where(field, QueryRuleEnum.ANY, value, required);
    }


    default NbQuerier<T> whereLike(String field, java.lang.String search) {
        return this.where(field, QueryRuleEnum.LIKE, search);
    }

    /**
     * 右模糊
     *
     * @param field
     * @param search
     * @return
     */
    default NbQuerier<T> whereLikeRight(String field, java.lang.String search) {
        return this.where(field, QueryRuleEnum.LIKE, search + "%");
    }

    /**
     * 左模糊
     *
     * @param field
     * @param search
     * @return
     */
    default NbQuerier<T> whereLikeLeft(String field, java.lang.String search) {
        return this.where(field, QueryRuleEnum.LIKE, "%" + search);
    }

    default NbQuerier<T> whereQrs(String qrs) {
        JSONObject jsonObject = null;
        if (StrUtil.isNotBlank(qrs)) {
            try {
                jsonObject = JSONUtil.parseObj(qrs);
            } catch (Throwable throwable) {
                throw new NbSQLMessageException("qrs格式错误");
            }
        }
        return whereQrs(jsonObject);
    }

    default NbQuerier<T> whereQrs(Map<String, Object> qrs) {
        return whereQrs(qrs == null ? null : new JSONObject(qrs));
    }

    default NbQuerier<T> where(String field, Object value, boolean notNull) {
        return where(field, QueryRuleEnum.EQ, value, notNull);
    }

    default NbQuerier<T> where(String field, QueryRuleEnum ruleEnum, Object value) {
        return where(field, ruleEnum, value, false);
    }

    default NbQuerier<T> where(String field, java.lang.String condition, Object value) {
        //rule 转查询枚举
        return where(field, QueryRuleEnum.parse(condition), value);
    }

    default NbQuerier<T> whereInters(String field, Collection<?> list) {
        Assert.isFalse(CollUtil.isEmpty(list), "字段:" + field + "数组不能为空");
        return where(field, QueryRuleEnum.INTERS, list);
    }

    default NbQuerier<T> whereIn(String field, Collection<?> list) {
        return whereIn(field, list, true);
    }

    default NbQuerier<T> whereIn(String field, Collection<?> list, boolean notNull) {
        Assert.isFalse(CollUtil.isEmpty(list) && notNull, "字段:" + field + "数组不能为空");
        return where(field, QueryRuleEnum.IN, list, notNull);
    }

    default NbQuerier<T> whereNotIn(String field, Collection<?> list) {
        return where(field, QueryRuleEnum.NOT_IN, list, false);
    }

     NbQuerier<T> whereOr(Consumer<NbQuerier<T>> consumer) ;

    default NbQuerier<T> whereNotNull(String field) {
        return where(field, QueryRuleEnum.NOT_NULL, null, false);
    }

    default NbQuerier<T> whereNull(String field) {
        return where(field, QueryRuleEnum.NULL, null, false);
    }

    /**
     * 字段值为value 或字段值问null
     *
     * @param field
     * @param value
     * @return
     */
    default NbQuerier<T> whereOrNull(String field, Object value) {
        return where(i -> i.where(field, value, true).or().whereNull(field));
    }

    /**
     * 查询值相等
     *
     * @param field
     * @param value
     * @return
     */
    default NbQuerier<T> where(String field, Object value) {
        return where(field, value, false);
    }

    /**
     * sql 表达式条件
     *
     * @param sql   表达式
     * @param value
     * @return
     */
    default NbQuerier<T> whereExpr(String sql, QueryValParam... value) {
        return where(parseSqlParam(sql, value));
    }

    NbQuerier<T> page(int pageNo, int pageSize);

    default NbQuerier<T> limit(int length) {
        Assert.isTrue(length >= 0, "limit length必须大于或等于0");
        return this.last("LIMIT " + length);
    }

    /**
     * 字段排序
     *
     * @param sort
     * @param fields
     */
    default NbQuerier<T> order(String sort, String... fields) {
        return order(NbOrderType.parse(sort), fields);
    }

    /**
     * 升序
     *
     * @param fields
     * @return
     */
    default NbQuerier<T> asc(String... fields) {
        return order(NbOrderType.ASC, fields);
    }

    /**
     * 降序
     *
     * @param fields
     * @return
     */
    default NbQuerier<T> desc(String... fields) {
        return order(NbOrderType.DESC, fields);
    }


}
